<?php

namespace MEC_Square\Core\Gateway;

require 'vendor/autoload.php';

use Square\SquareClient;
use Square\Environment;
use Square\Exceptions\ApiException;
// Don't load directly
if (!defined('ABSPATH')) {
    header('Status: 403 Forbidden');
    header('HTTP/1.1 403 Forbidden');
    exit;
}

/**
 *  Init.
 *
 *  @author      Webnus <info@webnus.biz>
 *  @package     Modern Events Calendar
 *  @since       1.0.0
 **/
class Init extends \MEC_gateway
{

    /**
     * Gateway ID
     *
     * @var integer
     */
    public $id = 2022;

    /**
     * Options
     *
     * @var array
     */
    public $options;

    /**
     *  Instance of this class.
     *
     *  @since   1.0.0
     *  @access  public
     *  @var     MEC_Square
     */
    public static $instance;

    /**
     *  The directory of this file
     *
     *  @access  public
     *  @var     string
     */
    public static $dir;

    /**
     *  Provides access to a single instance of a module using the Singleton pattern.
     *
     *  @since   1.0.0
     *  @return  object
     */
    public static function instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }

        return self::$instance;
    }


    public function __construct()
    {
        if (self::$instance === null) {
            self::$instance = $this;
        }

        parent::__construct();

        // Gateway options
        $this->options = $this->options();

        $this->factory->action('wp_ajax_mec_do_transaction_square_payment', array($this, 'do_transaction'));
        $this->factory->action('wp_ajax_nopriv_mec_do_transaction_square_payment', array($this, 'do_transaction'));

        $this->factory->action('wp_ajax_mec_do_cart_square_payment', array($this, 'cart_do_transaction'));
        $this->factory->action('wp_ajax_nopriv_mec_do_cart_square_payment', array($this, 'cart_do_transaction'));
    }

    public function label()
    {
        return esc_html__('Square', 'mec-square');
    }

    public function enabled()
    {
        return ((isset($this->options['status']) and $this->options['status'] and isset($this->options['application_id']) and trim($this->options['application_id']) and isset($this->options['access_token']) and trim($this->options['access_token']) and isset($this->options['location_id']) and trim($this->options['location_id'])) ? true : false);
    }

    public function options_form()
    {
        $account_status = isset($this->options['access_token']) && isset($this->options['mode']) ? $this->get_square_account_status($this->options['access_token'], $this->options['mode']) : '';
?>
        <div class="mec-form-row mec-click-pay">
            <label>
                <input type="hidden" name="mec[gateways][<?php echo esc_attr($this->id()); ?>][status]" value="0" />
                <input onchange="jQuery('#mec_gateways<?php echo esc_attr($this->id()); ?>_container_toggle').toggle();" value="1" type="checkbox" name="mec[gateways][<?php echo esc_attr($this->id()); ?>][status]"
                    <?php
                    if (isset($this->options['status']) and $this->options['status']) {
                        echo 'checked="checked"';
                    }
                    ?> /> <?php esc_html_e('Square', 'mec-square'); ?>
            </label>
        </div>
        <div id="mec_gateways<?php echo esc_attr($this->id()); ?>_container_toggle" class="mec-gateway-options-form
										<?php
                                        if ((isset($this->options['status']) and !$this->options['status']) or !isset($this->options['status'])) {
                                            echo 'mec-util-hidden';
                                        }
                                        ?>">
            <div class="mec-form-row">
                <label class="mec-col-3"
                    for="mec_gateways<?php echo esc_attr($this->id()); ?>_title"><?php esc_html_e('Title', 'mec-square'); ?></label>
                <div class="mec-col-9">
                    <input type="text" id="mec_gateways<?php echo esc_attr($this->id()); ?>_title"
                        name="mec[gateways][<?php echo esc_attr($this->id()); ?>][title]"
                        value="<?php echo (isset($this->options['title']) and trim($this->options['title'])) ? esc_attr($this->options['title']) : ''; ?>"
                        placeholder="<?php echo esc_attr($this->label()); ?>" />
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3"
                    for="mec_gateways<?php echo esc_attr($this->id()); ?>_comment"><?php esc_html_e('Comment', 'mec-square'); ?></label>
                <div class="mec-col-9">
                    <textarea id="mec_gateways<?php echo esc_attr($this->id()); ?>_comment"
                        name="mec[gateways][<?php echo esc_attr($this->id()); ?>][comment]"><?php echo (isset($this->options['comment']) and trim($this->options['comment'])) ? esc_textarea(stripslashes($this->options['comment'])) : esc_html__('Square Gateway Description', 'mec-square'); ?></textarea>
                    <span class="mec-tooltip">
                        <div class="box left">
                            <h5 class="title"><?php esc_html_e('Comment', 'mec-square'); ?></h5>
                            <div class="content">
                                <p><?php esc_attr_e('Add a customized description for this payment gateway option on the booking module. HTML allowed.', 'mec-square'); ?><a
                                        href="https://webnus.net/dox/modern-events-calendar/payment-gateways/"
                                        target="_blank"><?php esc_html_e('Read More', 'mec-square'); ?></a></p>
                            </div>
                        </div>
                        <i title="" class="dashicons-before dashicons-editor-help"></i>
                    </span>
                </div>
            </div>
            <div class="mec-form-row">
                <div class="mec-col-12"><?php echo $account_status; ?></div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3"
                    for="mec_gateways<?php echo esc_attr($this->id()); ?>_application_id"><?php esc_html_e('Application ID', 'mec-square'); ?></label>
                <div class="mec-col-9">
                    <input class="mec-required" type="text" id="mec_gateways<?php echo esc_attr($this->id()); ?>_application_id"
                        name="mec[gateways][<?php echo esc_attr($this->id()); ?>][application_id]"
                        value="<?php echo isset($this->options['application_id']) ? esc_attr($this->options['application_id']) : ''; ?>" />
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3"
                    for="mec_gateways<?php echo esc_attr($this->id()); ?>_access_token"><?php esc_html_e('Access Token', 'mec-square'); ?></label>
                <div class="mec-col-9">
                    <input class="mec-required" type="text" id="mec_gateways<?php echo esc_attr($this->id()); ?>_access_token"
                        name="mec[gateways][<?php echo esc_attr($this->id()); ?>][access_token]"
                        value="<?php echo isset($this->options['access_token']) ? esc_attr($this->options['access_token']) : ''; ?>" />
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3"
                    for="mec_gateways<?php echo esc_attr($this->id()); ?>_location_id"><?php esc_html_e('Location ID', 'mec-square'); ?></label>
                <div class="mec-col-9">
                    <input class="mec-required" type="text" id="mec_gateways<?php echo esc_attr($this->id()); ?>_location_id"
                        name="mec[gateways][<?php echo esc_attr($this->id()); ?>][location_id]"
                        value="<?php echo isset($this->options['location_id']) ? esc_attr($this->options['location_id']) : ''; ?>" />
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3"
                    for="mec_gateways<?php echo esc_attr($this->id()); ?>_mode"><?php esc_html_e('Mode', 'mec-square'); ?></label>
                <div class="mec-col-9">
                    <select id="mec_gateways<?php echo esc_attr($this->id()); ?>_mode"
                        name="mec[gateways][<?php echo esc_attr($this->id()); ?>][mode]">
                        <option value="production" <?php echo (isset($this->options['mode']) and $this->options['mode'] == 'production') ? 'selected="selected"' : ''; ?>><?php esc_html_e('Production', 'mec-square'); ?></option>
                        <option value="sandbox" <?php echo (isset($this->options['mode']) and $this->options['mode'] == 'sandbox') ? 'selected="selected"' : ''; ?>><?php esc_html_e('Sandbox', 'mec-square'); ?></option>
                    </select>
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3" for="mec_gateways<?php echo esc_attr($this->id()); ?>_index"><?php esc_html_e('Position', 'mec-square'); ?></label>
                <div class="mec-col-9">
                    <input type="number" min="0" step="1" id="mec_gateways<?php echo esc_attr($this->id()); ?>_index"
                        name="mec[gateways][<?php echo esc_attr($this->id()); ?>][index]"
                        value="<?php echo (isset($this->options['index']) and trim($this->options['index'])) ? esc_attr($this->options['index']) : 10; ?>"
                        placeholder="<?php echo esc_attr__('Position', 'mec-square'); ?>" />
                </div>
            </div>
        </div>
    <?php
    }

    public function checkout_form($transaction_id, $params = array())
    {
        // Get Options Compatible with Organizer Payment
        $options = $this->options($transaction_id);

        $transaction = $this->book->get_transaction($transaction_id);
        $event_id = isset($transaction['event_id']) ? $transaction['event_id'] : 0;
        $currency =  $this->main->get_currency_code($event_id);
        $price = isset($transaction['price']) ? $transaction['price'] : 0;
        $email = isset($transaction['tickets'][0]['email']) ? $transaction['tickets'][0]['email'] : 'example@example.com';

        $application_id = (isset($options['application_id']) && !empty($options['application_id'])) ? $options['application_id'] : '';
        $location_id = (isset($options['location_id']) && !empty($options['location_id'])) ? $options['location_id'] : '';
    ?>
        <script type="text/javascript" src="<?php echo $this->get_square_endpoint_js($options['mode']); ?>"></script>
        <script>
            if (document.readyState !== 'loading') {
                console.log('document is already ready, just execute code here');
                var mec_cart_square_interval = setInterval(function() {
                    if (!window.Square) {
                        throw new Error('Square.js failed to load properly');
                    } else {
                        async function createPayment(token, squareVerificationToken) {
                            let sToken = token.token;
                            let vToken = squareVerificationToken
                            jQuery.ajax({
                                type: "GET",
                                url: "<?php echo admin_url('admin-ajax.php', NULL); ?>",
                                data: 'action=mec_do_transaction_square_payment&_wpnonce=<?php echo wp_create_nonce('mec_transaction_form_' . $transaction_id); ?>&transaction_id=<?php echo esc_attr($transaction_id); ?>&verification_token=' + vToken + '&square_token=' + sToken,
                                dataType: "JSON",
                                success: function(data) {
                                    jQuery("#mec-square-loader").hide();
                                    if (data.success === 1) {
                                        jQuery(".mec-gateway-comment").hide();
                                        jQuery(".mec-book-form-gateway-label").remove();
                                        jQuery(".mec-book-form-coupon").hide();

                                        jQuery("#mec_do_transaction_square_payment_message<?php echo esc_attr($transaction_id); ?>").addClass("mec-success").html(data.message).show();
                                        jQuery("#mec-book-form-back-btn-step-3").remove();
                                        jQuery("#mec-square-payment-form").remove();

                                        // Mark progress bar as completed
                                        jQuery('.mec-booking-progress-bar-complete').addClass('mec-active');
                                        jQuery('.mec-booking-progress-bar-complete.mec-active').parents().eq(2).addClass("row-done");

                                        // Show Invoice Link
                                        if (typeof data.data.invoice_link !== "undefined" && data.data.invoice_link != "") {
                                            jQuery("#mec_do_transaction_square_payment_message<?php echo esc_attr($transaction_id); ?>").append(' <a class="mec-invoice-download" target="_blank" href="' + data.data.invoice_link + '"><?php echo esc_js(__('Download Invoice', 'mec-square')); ?></a>');
                                        }

                                        // Show Downloadable Link
                                        if (typeof data.data.dl_file_link !== "undefined" && data.data.dl_file_link != "") {
                                            jQuery("#mec_do_transaction_square_payment_message<?php echo esc_attr($transaction_id); ?>").append('  — <a class="mec-dl-file-download" href="' + data.data.dl_file_link + '"><?php echo esc_js(__('Download File', 'mec-square')); ?></a>');
                                        }

                                        // Show Extra info
                                        if (typeof data.data.extra_info !== "undefined" && data.data.extra_info != "" && data.data.extra_info != null) {
                                            jQuery("#mec_do_transaction_square_payment_message<?php echo esc_attr($transaction_id); ?>").append('<div>' + data.data.extra_info + '</div>');
                                        }

                                        // Redirect to thank you page
                                        if (typeof data.data.redirect_to !== "undefined" && data.data.redirect_to !== "") {
                                            setTimeout(function() {
                                                window.location.href = data.data.redirect_to;
                                            }, <?php echo absint($this->main->get_thankyou_page_time($transaction_id)); ?>);
                                        }
                                    } else {
                                        jQuery("#mec_do_transaction_square_payment_message<?php echo esc_attr($transaction_id); ?>").addClass("mec-error").html(data.message).show();
                                    }
                                },
                                error: function(jqXHR, textStatus, errorThrown) {
                                    jQuery("#mec-square-loader").hide();
                                    jQuery("#mec_do_transaction_square_payment_message<?php echo esc_attr($transaction_id); ?>").addClass("mec-error").html("<?php echo esc_js(esc_html__('Something went wrong!', 'mec-square')); ?>").show();
                                }
                            });
                        }

                        async function verifyBuyer(payments, token) {
                            const verificationDetails = {
                                amount: '<?php echo $price; ?>',
                                /* collected from the buyer */
                                billingContact: {
                                    email: '<?php echo $email; ?>',
                                },
                                currencyCode: '<?php echo $currency; ?>',
                                intent: 'CHARGE',
                            };

                            const verificationResults = await payments.verifyBuyer(
                                token,
                                verificationDetails
                            );
                            return verificationResults.token;
                        }

                        const loader = document.querySelector('#mec-square-loader')
                        async function main() {
                            const appId = '<?php echo $application_id; ?>';
                            const locationId = '<?php echo $location_id; ?>';
                            const payments = window.Square.payments(appId, locationId);
                            loader.style.display = 'block'
                            const card = await payments.card();
                            loader.style.display = 'none'
                            await card.attach('#mec-square-card-container');
                            async function eventHandler(event) {
                                event.preventDefault();
                                loader.style.display = 'block'
                                try {
                                    const result = await card.tokenize();
                                    const paymentVerificationToken = await verifyBuyer(payments, result.token);
                                    const paymentResults = await createPayment(result, paymentVerificationToken);
                                } catch (e) {
                                    console.error(e);
                                }
                            };
                            const cardButton = document.getElementById('mec-square-card-button');
                            cardButton.addEventListener('click', eventHandler);
                        }
                        main();
                    }
                    clearInterval(mec_cart_square_interval);
                }, 100);
            } else {
                document.addEventListener('DOMContentLoaded', function() {
                    console.log('document was not ready, place code here');
                });
            }
        </script>
        <form id="mec-square-payment-form" style="margin-top: 30px; position: relative">
            <div id="mec-square-loader" style="position: absolute;width: 100%;height: 100%;background: #ffffffa8;z-index: 9;">
                <div class="mec-loader" style="top: calc(50% - 35px);"></div>
            </div>
            <div id="mec-square-card-container"></div>
            <button id="mec-square-card-button" type="button"><?php esc_html_e('Pay', 'mec-square'); ?></button>
        </form>
        <div id="mec-square-payment-status-container"></div>
        <div class="mec-gateway-message" id="mec_do_transaction_square_payment_message<?php echo esc_attr($transaction_id); ?>"><?php do_action('mec_extra_info_payment'); ?></div>
    <?php
    }

    public function do_transaction($transaction_id = NULL)
    {
        if (!trim($transaction_id)) $transaction_id = isset($_GET['transaction_id']) ? sanitize_text_field($_GET['transaction_id']) : 0;
        $square_token = isset($_GET['square_token']) ? sanitize_text_field($_GET['square_token']) : NULL;
        $verification_token = isset($_GET['verification_token']) ? sanitize_text_field($_GET['verification_token']) : NULL;

        // Verify that the nonce is valid.
        if (!wp_verify_nonce(sanitize_text_field($_GET['_wpnonce']), 'mec_transaction_form_' . $transaction_id)) {
            $this->response(array(
                'success' => 0,
                'code' => 'NONCE_IS_INVALID',
                'message' => esc_html__('Request is invalid!', 'mec-square'),
            ));
        }

        // Validate Ticket Availability
        $this->validate($transaction_id);

        $options = $this->options($transaction_id);
        $access_token = (isset($options['access_token']) && !empty($options['access_token'])) ? $options['access_token'] : '';
        $mode = $options['mode'];
        $location_id = isset($options['location_id']) ? $options['location_id'] : '';
        $transaction = $this->book->get_transaction($transaction_id);

        $main_attendee = isset($transaction['tickets'][0]) ? $transaction['tickets'][0] : array();
        $buyer_name = isset($main_attendee['name']) ? trim($main_attendee['name']) : '';
        $name_parts = explode(' ', $buyer_name, 2);
        $first_name = $name_parts[0];
        $last_name = isset($name_parts[1]) ? $name_parts[1] : '';

        $event_id = isset($transaction['event_id']) ? $transaction['event_id'] : 0;
        $title = get_the_title($event_id);
        $date = get_post_meta($event_id, 'mec_start_date', true);

        $currency =  $this->main->get_currency_code($event_id);
        $amount = isset($transaction['price']) ? number_format((float) esc_attr($transaction['price']) * 100, 0, '', '') : 0;

        $curl_handle = curl_init();
        curl_setopt($curl_handle, CURLOPT_URL, $this->get_square_endpoint($mode) . '/v2/payments');
        curl_setopt($curl_handle, CURLOPT_CONNECTTIMEOUT, 2);
        curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, 1);
        $headers = array(
            "Content-Type: application/json",
            "Authorization: Bearer " . $access_token
        );
        curl_setopt($curl_handle, CURLOPT_HTTPHEADER, $headers);

        $tickets_note = [];
        foreach ($transaction['tickets'] as $ticket) {
            $tickets_note[] = "{$ticket['name']} (x{$ticket['count']})";
        }
        $tickets_note_str = implode(', ', $tickets_note);

        $note = "Event: {$title} | Date: {$date} | Buyer: {$buyer_name} | Tickets: {$tickets_note_str}";

        $data = json_encode(array(
            'amount_money' => array(
                'currency' => $currency,
                'amount' => (int) $amount,
            ),
            'idempotency_key' => $transaction_id,
            'source_id' => $square_token,
            'verification_token' => $verification_token,
            'location_id' => $location_id,
            'billing_address' => array(
                'first_name' => $first_name,
                'last_name' => $last_name,
            ),
            'note' => $note,
        ));

        curl_setopt($curl_handle, CURLOPT_POSTFIELDS, $data);

        $buffer = curl_exec($curl_handle);
        $result = json_decode($buffer, TRUE); // parse json string to array

        curl_close($curl_handle);

        if (empty($buffer)) {
            $this->response(array(
                'success' => 0,
                'code' => 'ORDER_IS_INVALID',
                'message' => esc_html__('Nothing returned from url.', 'mec-square'),
            ));
        } else {
            if (isset($result['errors'])) {
                // print_r($result['errors'][0]['detail']);
                $this->response(array(
                    'success' => 0,
                    'code' => 'ORDER_IS_INVALID',
                    'message' => esc_html__($result['errors'][0]['detail']),
                ));
            } elseif ($result['payment']) {
                // print_r($result['payment']);
                $transaction = $this->book->get_transaction($transaction_id);
                $attendees = isset($transaction['tickets']) ? $transaction['tickets'] : array();

                $attention_date = isset($transaction['date']) ? $transaction['date'] : '';
                $attention_times = explode(':', $attention_date);
                $date = date('Y-m-d H:i:s', trim($attention_times[0]));

                // Is there any attendee?
                if (!count($attendees)) {
                    $this->response(array(
                        'success' => 0,
                        'code' => 'NO_TICKET',
                        'message' => esc_html__('There is no attendee for booking!', 'mec-square'),
                    ));
                }

                $main_attendee = isset($attendees[0]) ? $attendees[0] : array();
                $name = isset($main_attendee['name']) ? $main_attendee['name'] : '';

                $ticket_ids = '';
                $attendees_info = array();

                foreach ($attendees as $i => $attendee) {
                    if (!is_numeric($i)) continue;

                    $ticket_ids .= $attendee['id'] . ',';
                    if (!array_key_exists($attendee['email'], $attendees_info)) $attendees_info[$attendee['email']] = array('count' => $attendee['count']);
                    else $attendees_info[$attendee['email']]['count'] = ($attendees_info[$attendee['email']]['count'] + $attendee['count']);
                }

                $ticket_ids = ',' . trim($ticket_ids, ', ') . ',';
                $user_id = $this->register_user($main_attendee, $transaction);

                // Remove Sensitive Data
                if (isset($transaction['username']) or isset($transaction['password'])) {
                    unset($transaction['username']);
                    unset($transaction['password']);
                }


                $this->book->update_transaction($transaction_id, $transaction);

                // MEC User
                $u = $this->getUser();

                $book_subject = $name . ' - ' . (isset($main_attendee['email']) ? $main_attendee['email'] : $u->get($user_id)->user_email);
                $book_id = $this->book->add(
                    array(
                        'post_author' => $user_id,
                        'post_type' => $this->PT,
                        'post_title' => $book_subject,
                        'post_date' => $date,
                        'attendees_info' => $attendees_info,
                        'mec_attendees' => $attendees,
                        'mec_gateway' => 'MEC_gateway_square',
                        'mec_gateway_label' => $this->title()
                    ),
                    $transaction_id,
                    $ticket_ids
                );
                if (isset($result['payment']['id'])) {
                    update_post_meta($book_id, 'mec_square_payment_id', $result['payment']['id']);
                    update_post_meta($book_id, 'mec_square_payment_type', 'payment');
                }
                // Assign User
                $u->assign($book_id, $user_id);

                // Fires after completely creating a new booking
                do_action('mec_booking_completed', $book_id);

                $event_id = (isset($transaction['event_id']) ? $transaction['event_id'] : 0);
                $redirect_to = '';

                $thankyou_page_id = $this->main->get_thankyou_page_id($event_id);
                if ($thankyou_page_id) $redirect_to = $this->book->get_thankyou_page($thankyou_page_id, $transaction_id);

                // Invoice Link
                $mec_confirmed = get_post_meta($book_id, 'mec_confirmed', true);
                $invoice_link = (!$mec_confirmed) ? '' : $this->book->get_invoice_link($transaction_id);
                $dl_file_link = (!$mec_confirmed) ? '' : $this->book->get_dl_file_link($book_id);

                $extra_info = apply_filters('MEC_extra_info_gateways', '', $this->book->get_event_id_by_transaction_id($transaction_id), $book_id);

                $this->response(
                    array(
                        'success' => 1,
                        'message' => stripslashes($this->main->m('book_success_message', esc_html__('Thank you for booking. Your tickets are booked, booking verification might be needed, please check your email.', 'mec-square'))),
                        'data' => array(
                            'book_id' => $book_id,
                            'redirect_to' => $redirect_to,
                            'invoice_link' => $invoice_link,
                            'dl_file_link' => $dl_file_link,
                            'extra_info' => $extra_info,
                        ),
                    )
                );
            }
        }
    }

    public function cart_checkout_form($cart_id, $params = array())
    {
        // Get Options Compatible with Organizer Payment
        $options = $this->options();

        // Cart Library
        $c = $this->getCart();
        $cart = $c->get_cart($cart_id);

        $currency =  $this->main->get_currency_code();

        $application_id = (isset($options['application_id']) && !empty($options['application_id'])) ? $options['application_id'] : '';
        $location_id = (isset($options['location_id']) && !empty($options['location_id'])) ? $options['location_id'] : '';
        $payable_total = ceil($c->get_payable($cart));
        $verify_amount = number_format($payable_total, 2, '.', '');

        $email = isset($cart[0]['tickets'][0]['email']) ? $cart[0]['tickets'][0]['email'] : 'example@example.com';

    ?>
        <script type="text/javascript" src="<?php echo $this->get_square_endpoint_js($options['mode']); ?>"></script>
        <script>
            const mec_verify_amount = "<?php echo esc_js($verify_amount); ?>";
            const mec_verify_currency = "<?php echo esc_js($currency); ?>";
            const mec_verify_email = "<?php echo esc_js($email); ?>";
            if (document.readyState !== 'loading') {

                console.log('document is already ready, just execute code here');
                var mec_cart_square_interval = setInterval(function() {
                    if (!window.Square) {
                        throw new Error('Square.js failed to load properly');
                    } else {
                        async function createPayment(token, squareVerificationToken) {
                            let sToken = token.token;
                            let vToken = squareVerificationToken
                            jQuery.ajax({
                                type: "GET",
                                url: "<?php echo admin_url('admin-ajax.php', NULL); ?>",
                                data: 'action=mec_do_cart_square_payment&_wpnonce=<?php echo wp_create_nonce('mec_cart_form_' . $cart_id); ?>&cart_id=<?php echo esc_attr($cart_id); ?>&verification_token=' + vToken + '&square_token=' + sToken,
                                dataType: "JSON",
                                success: function(data) {
                                    jQuery("#mec-square-loader").hide();
                                    if (data.success === 1) {
                                        jQuery(".mec-checkout-form-gateway-label").remove();
                                        jQuery(".mec-gateway-comment").hide();
                                        jQuery("#mec-square-payment-form").remove();

                                        jQuery("#mec_do_transaction_square_payment_message<?php echo esc_attr($cart_id); ?>").addClass("mec-success").html(data.message).show();

                                        // Redirect to thank you page
                                        if (typeof data.data.redirect_to !== "undefined" && data.data.redirect_to !== "") {
                                            setTimeout(function() {
                                                window.location.href = data.data.redirect_to;
                                            }, <?php echo absint($this->main->get_thankyou_page_time()); ?>);
                                        }
                                    } else {
                                        jQuery("#mec_do_transaction_square_payment_message<?php echo esc_attr($cart_id); ?>").addClass("mec-error").html(data.message).show();
                                    }
                                },
                                error: function(jqXHR, textStatus, errorThrown) {
                                    jQuery("#mec-square-loader").hide();
                                    jQuery("#mec_do_transaction_square_payment_message<?php echo esc_attr($cart_id); ?>").addClass("mec-error").html("<?php echo esc_js(esc_html__('Something went wrong!', 'mec-square')); ?>").show();
                                }
                            });
                        }

                        async function verifyBuyer(payments, token) {
                            const verificationDetails = {
                                currencyCode: '<?php echo $currency; ?>',
                                intent: 'CHARGE',
                            };

                            const verificationResults = await payments.verifyBuyer(
                                token,
                                verificationDetails
                            );
                            return verificationResults.token;
                        }
                        const loader = document.querySelector('#mec-square-loader')
                        async function main() {
                            const appId = '<?php echo $application_id; ?>';
                            const locationId = '<?php echo $location_id; ?>';
                            const payments = window.Square.payments(appId, locationId);
                            loader.style.display = 'block'
                            const card = await payments.card();
                            loader.style.display = 'none'
                            await card.attach('#mec-square-card-container');
                            async function eventHandler(event) {
                                event.preventDefault();
                                loader.style.display = 'block'
                                try {
                                    const result = await card.tokenize();
                                    const paymentVerificationToken = await verifyBuyer(payments, result.token);
                                    const paymentResults = await createPayment(result, paymentVerificationToken);
                                } catch (e) {
                                    console.error(e);
                                }
                            };
                            const cardButton = document.getElementById('mec-square-card-button');
                            cardButton.addEventListener('click', eventHandler);
                        }
                        main();
                    }
                    clearInterval(mec_cart_square_interval);
                }, 100);
            } else {
                document.addEventListener('DOMContentLoaded', function() {
                    console.log('document was not ready, place code here');
                    var mec_cart_square_interval = setInterval(function() {
                        if (!window.Square) {
                            throw new Error('Square.js failed to load properly');
                        } else {
                            async function createPayment(token, squareVerificationToken) {
                                let sToken = token.token;
                                let vToken = squareVerificationToken
                                jQuery.ajax({
                                    type: "GET",
                                    url: "<?php echo admin_url('admin-ajax.php', NULL); ?>",
                                    data: 'action=mec_do_cart_square_payment&_wpnonce=<?php echo wp_create_nonce('mec_cart_form_' . $cart_id); ?>&cart_id=<?php echo esc_attr($cart_id); ?>&verification_token=' + vToken + '&square_token=' + sToken,
                                    dataType: "JSON",
                                    success: function(data) {
                                        jQuery("#mec-square-loader").hide();
                                        if (data.success === 1) {
                                            jQuery(".mec-checkout-form-gateway-label").remove();
                                            jQuery(".mec-gateway-comment").hide();
                                            jQuery("#mec-square-payment-form").remove();

                                            jQuery("#mec_do_transaction_square_payment_message<?php echo esc_attr($cart_id); ?>").addClass("mec-success").html(data.message).show();

                                            // Redirect to thank you page
                                            if (typeof data.data.redirect_to !== "undefined" && data.data.redirect_to !== "") {
                                                setTimeout(function() {
                                                    window.location.href = data.data.redirect_to;
                                                }, <?php echo absint($this->main->get_thankyou_page_time()); ?>);
                                            }
                                        } else {
                                            jQuery("#mec_do_transaction_square_payment_message<?php echo esc_attr($cart_id); ?>").addClass("mec-error").html(data.message).show();
                                        }
                                    },
                                    error: function(jqXHR, textStatus, errorThrown) {
                                        jQuery("#mec-square-loader").hide();
                                        jQuery("#mec_do_transaction_square_payment_message<?php echo esc_attr($cart_id); ?>").addClass("mec-error").html("<?php echo esc_js(esc_html__('Something went wrong!', 'mec-square')); ?>").show();
                                    }
                                });
                            }

                            async function verifyBuyer(payments, token) {
                                const verificationDetails = {
                                    amount: mec_verify_amount,
                                    currencyCode: mec_verify_currency,
                                    intent: 'CHARGE',
                                    billingContact: {
                                        email: mec_verify_email,
                                    },
                                    customerInitiated: true,
                                    sellerKeyedIn: false
                                };
                                const verificationResults = await payments.verifyBuyer(token, verificationDetails);
                                return verificationResults.token;
                            }
                            const loader = document.querySelector('#mec-square-loader')
                            async function main() {
                                const appId = '<?php echo $application_id; ?>';
                                const locationId = '<?php echo $location_id; ?>';
                                const payments = window.Square.payments(appId, locationId);
                                loader.style.display = 'block'
                                const card = await payments.card();
                                loader.style.display = 'none'
                                await card.attach('#mec-square-card-container');
                                async function eventHandler(event) {
                                    event.preventDefault();
                                    loader.style.display = 'block'
                                    try {
                                        const result = await card.tokenize();
                                        const paymentVerificationToken = await verifyBuyer(payments, result.token);
                                        const paymentResults = await createPayment(result, paymentVerificationToken);
                                    } catch (e) {
                                        console.error(e);
                                    }
                                };
                                const cardButton = document.getElementById('mec-square-card-button');
                                cardButton.addEventListener('click', eventHandler);
                            }
                            main();
                        }
                        clearInterval(mec_cart_square_interval);
                    }, 300);
                });
            }
        </script>
        <form id="mec-square-payment-form" style="margin-top: 30px; position: relative">
            <div id="mec-square-loader" style="position: absolute;width: 100%;height: 100%;background: #ffffffa8;z-index: 9;">
                <div class="mec-loader" style="top: calc(50% - 35px);"></div>
            </div>
            <div id="mec-square-card-container"></div>
            <button id="mec-square-card-button" type="button"><?php esc_html_e('Pay', 'mec-square'); ?></button>
        </form>
        <div id="mec-square-payment-status-container"></div>
        <div class="mec-gateway-message" id="mec_do_transaction_square_payment_message<?php echo esc_attr($cart_id); ?>"><?php do_action('mec_extra_info_payment'); ?></div>
<?php
    }

    public function cart_do_transaction()
    {
        $cart_id = isset($_GET['cart_id']) ? sanitize_text_field($_GET['cart_id']) : NULL;
        $square_token = isset($_GET['square_token']) ? sanitize_text_field($_GET['square_token']) : NULL;
        $verification_token = isset($_GET['verification_token']) ? sanitize_text_field($_GET['verification_token']) : NULL;

        if (!wp_verify_nonce(sanitize_text_field($_GET['_wpnonce']), 'mec_cart_form_' . $cart_id)) {
            $this->response(array(
                'success' => 0,
                'code' => 'NONCE_IS_INVALID',
                'message' => esc_html__('Request is invalid!', 'mec-square'),
            ));
        }

        $this->cart_validate($cart_id);

        $c = $this->getCart();
        $cart = $c->get_cart($cart_id);
        $options = $this->options();
        $access_token = $options['access_token'] ?? '';
        $mode = $options['mode'];
        $location_id = $options['location_id'] ?? '';
        $currency = $this->main->get_currency_code();

        $amount = (int) bcmul((string) $c->get_payable($cart), '100', 0);

        $first_transaction = reset($cart);
        $transaction = $this->book->get_transaction($first_transaction);
        $attendees = $transaction['tickets'] ?? [];
        $main_attendee = $attendees[0] ?? [];

        $buyer_name = trim($main_attendee['name'] ?? '');
        $name_parts = explode(' ', $buyer_name, 2);
        $first_name = $name_parts[0] ?? '';
        $last_name = $name_parts[1] ?? '';
        $buyer_email = $main_attendee['email'] ?? '';

        $event_id = $transaction['event_id'] ?? 0;
        $event_title = get_the_title($event_id);
        $event_date = get_post_meta($event_id, 'mec_start_date', true);

        $tickets_note = [];
        foreach ($attendees as $ticket) {
            $tickets_note[] = "{$ticket['name']} (x{$ticket['count']})";
        }
        $tickets_note_str = implode(', ', $tickets_note);

        $note = "Event: {$event_title} | Date: {$event_date} | Buyer: {$buyer_name} | Tickets: {$tickets_note_str}";

        $transaction_id = uniqid('txn_' . $cart_id . '_');

        $data = json_encode(array(
            'amount_money' => array(
                'currency' => $currency,
                'amount' => $amount,
            ),
            'idempotency_key' => $transaction_id,
            'source_id' => $square_token,
            'verification_token' => $verification_token,
            'location_id' => $location_id,
            'buyer_email_address' => $buyer_email,
            'billing_address' => array(
                'first_name' => $first_name,
                'last_name' => $last_name,
            ),
            'note' => $note,
        ));

        $curl_handle = curl_init();
        curl_setopt($curl_handle, CURLOPT_URL, $this->get_square_endpoint($mode) . '/v2/payments');
        curl_setopt($curl_handle, CURLOPT_CONNECTTIMEOUT, 2);
        curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curl_handle, CURLOPT_HTTPHEADER, array(
            "Content-Type: application/json",
            "Authorization: Bearer " . $access_token
        ));
        curl_setopt($curl_handle, CURLOPT_POSTFIELDS, $data);

        $buffer = curl_exec($curl_handle);
        $result = json_decode($buffer, TRUE);
        curl_close($curl_handle);

        if (empty($buffer)) {
            $this->response(array(
                'success' => 0,
                'code' => 'ORDER_IS_INVALID',
                'message' => esc_html__('Nothing returned from url.', 'mec-square'),
            ));
        } else {
            if (isset($result['errors'])) {
                // print_r($result['errors'][0]['detail']);
                $this->response(array(
                    'success' => 0,
                    'code' => 'ORDER_IS_INVALID',
                    'message' => esc_html__($result['errors'][0]['detail']),
                ));
            } elseif ($result['payment']) {
                // print_r($result['payment']);
                $book_ids = array();
                foreach ($cart as $transaction_id) {
                    $transaction = $this->book->get_transaction($transaction_id);
                    $attendees = isset($transaction['tickets']) ? $transaction['tickets'] : array();

                    $attention_date = isset($transaction['date']) ? $transaction['date'] : '';
                    $attention_times = explode(':', $attention_date);
                    $date = date('Y-m-d H:i:s', trim($attention_times[0]));

                    $main_attendee = isset($attendees[0]) ? $attendees[0] : array();
                    $name = isset($main_attendee['name']) ? $main_attendee['name'] : '';

                    $ticket_ids = '';
                    $attendees_info = array();

                    foreach ($attendees as $i => $attendee) {
                        if (!is_numeric($i)) continue;

                        $ticket_ids .= $attendee['id'] . ',';

                        if (!array_key_exists($attendee['email'], $attendees_info)) $attendees_info[$attendee['email']] = array('count' => $attendee['count']);
                        else $attendees_info[$attendee['email']]['count'] = ($attendees_info[$attendee['email']]['count'] + $attendee['count']);
                    }

                    $ticket_ids = ',' . trim($ticket_ids, ', ') . ',';
                    $user_id = $this->register_user($main_attendee, $transaction);

                    // Remove Sensitive Data
                    if (isset($transaction['username']) or isset($transaction['password'])) {
                        unset($transaction['username']);
                        unset($transaction['password']);

                        $this->book->update_transaction($transaction_id, $transaction);
                    }

                    // MEC User
                    $u = $this->getUser();

                    $book_subject = $name . ' - ' . (isset($main_attendee['email']) ? $main_attendee['email'] : $u->get($user_id)->user_email);

                    $book_id = $this->book->add(
                        array(
                            'post_author' => $user_id,
                            'post_type' => $this->PT,
                            'post_title' => $book_subject,
                            'post_date' => $date,
                            'attendees_info' => $attendees_info,
                            'mec_attendees' => $attendees,
                            'mec_gateway' => 'MEC_gateway_square',
                            'mec_gateway_label' => $this->title()
                        ),
                        $transaction_id,
                        $ticket_ids
                    );

                    if (isset($result['payment']['id'])) {
                        update_post_meta($book_id, 'mec_square_payment_id', $result['payment']['id']);
                        update_post_meta($book_id, 'mec_square_payment_type', 'payment');
                    }


                    // Assign User
                    $u->assign($book_id, $user_id);

                    // Fires after completely creating a new booking
                    do_action('mec_booking_completed', $book_id);

                    $book_ids[] = $book_id;
                }

                $invoice_status = (isset($this->settings['mec_cart_invoice']) and $this->settings['mec_cart_invoice']);
                $invoice_link = (!$invoice_status) ? '' : $c->get_invoice_link($cart_id);

                $message = stripslashes($this->main->m('book_success_message', esc_html__('Thank you for booking. Your tickets are booked, booking verification might be needed, please check your email.', 'mec-square')));
                if (trim($invoice_link)) $message .= ' <a class="mec-invoice-download" target="_blank" href="' . esc_url($invoice_link) . '">' . esc_html__('Download Invoice', 'mec-square') . '</a>';

                $redirect_to = '';

                $thankyou_page_id = $this->main->get_thankyou_page_id();
                if ($thankyou_page_id) $redirect_to = $this->book->get_thankyou_page($thankyou_page_id, NULL, $cart_id);

                $this->remove_fees_if_disabled($cart_id);

                // Empty Cart
                $c->clear($cart_id);

                $this->response(array(
                    'success' => 1,
                    'message' => $message,
                    'data' => array(
                        'book_ids' => $book_ids,
                        'redirect_to' => $redirect_to,
                        'invoice_link' => $invoice_link,
                    ),
                ));
            }
        }
    }
    public function refundsquare($booking_id, $amount = null)
    {
        $payment_id = get_post_meta($booking_id, 'mec_square_payment_id', true);

        if (!$payment_id) {
            return false;
        }

        $access_token = $this->options['access_token'] ?? '';
        $mode = $this->options['mode'] ?? 'sandbox';
        if (!$access_token) {
            return false;
        }

        $base_url = ($mode === 'sandbox') ? 'https://connect.squareupsandbox.com' : 'https://connect.squareup.com';

        $get_payment_url = "$base_url/v2/payments/$payment_id";

        $get_payment = wp_remote_get($get_payment_url, [
            'headers' => [
                'Content-Type'  => 'application/json',
                'Authorization' => 'Bearer ' . $access_token,
            ],
            'timeout' => 20,
        ]);

        $payment_data = json_decode(wp_remote_retrieve_body($get_payment));

        if (!isset($payment_data->payment)) {
            return false;
        }

        $amount_money = $payment_data->payment->amount_money ?? null;
        if (!$amount_money || !isset($amount_money->amount) || !isset($amount_money->currency)) {
            return false;
        }

        $amount = $amount_money->amount;
        $currency = $amount_money->currency;
        $body = [
            'idempotency_key' => uniqid(),
            'payment_id' => $payment_id,
            'amount_money' => [
                'amount' => $amount,
                'currency' => $currency,
            ],
        ];

        $args = [
            'headers' => [
                'Content-Type'  => 'application/json',
                'Authorization' => 'Bearer ' . $access_token,
            ],
            'body'    => json_encode($body),
            'timeout' => 20,
        ];

        $refund_url = "$base_url/v2/refunds";
        $response = wp_remote_post($refund_url, $args);
        $code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $result = json_decode($response_body);

        if ($code === 200 && isset($result->refund->id)) {
            update_post_meta($booking_id, 'mec_refund_ref_id', $result->refund->id);
            return true;
        }

        return false;
    }

    public function get_square_endpoint($mode)
    {
        if ($mode === 'sandbox') $url = 'https://connect.squareupsandbox.com';
        else $url = 'https://connect.squareup.com';

        return $url;
    }

    public function get_square_endpoint_js($mode)
    {
        if ($mode === 'sandbox') $url = 'https://sandbox.web.squarecdn.com/v1/square.js';
        else $url = 'https://web.squarecdn.com/v1/square.js';

        return $url;
    }

    public function get_square_account_status($token, $mode)
    {
        if (empty($token) || is_null($token) || empty($mode) || is_null($mode)) return;
        $client = new SquareClient([
            'accessToken' => $token,
            'environment' => $mode,
            'numberOfRetries' => 2,
            'timeout' => 60,
        ]);

        try {

            $apiResponse = $client->getLocationsApi()->listLocations();

            if ($apiResponse->isSuccess()) {
                $result = $apiResponse->getResult();
                return '<div class="mec-success">' . __('Access token is correct!', 'mec-square') . '</div>';
            } else {
                $errors = $apiResponse->getErrors();
                foreach ($errors as $error) {
                    return '<div class="mec-error">' . sprintf(
                        __("%s<br/> %s<br/> %s", 'mec-square'),
                        $error->getCategory(),
                        $error->getCode(),
                        $error->getDetail()
                    ) . '</div>';
                }
            }
        } catch (ApiException $e) {
            return '<div class="mec-error">' . sprintf(
                __("ApiException occurred: %s", 'mec-square'),
                $e->getMessage()
            ) . '</div>';
        }
    }
} //Init

add_filter(
    'MEC_register_gateways',
    function ($gateways) {
        $gateways['MEC_gateway_square_payment'] = \MEC_Square\Core\Gateway\Init::instance();
        return $gateways;
    }
);

add_action(
    'mec_feature_gateways_init',
    function () {
        \MEC_Square\Core\Gateway\Init::instance();
    }
);
