<?php

/**
 * Add Columns to the Admin CPT nw_seo_page overview
 * 
 * Filters the columns displayed in the CPT nw_seo_page posts list table for a specific post type.
 * 
 * @see manage_nw_seo_page_posts_columns
 * 
 * @param string[] $post_columns An associative array of column headings.
 */
function nsg_add_slug_column($post_columns)
{
    // In case the theme or another plugin changed our columns of our CPT nw_seo_page and removed the Title column...
    if (isset($post_columns['title']))
    {
        // SEO Page base
        $post_columns = nsg_array_insert_after(
            $post_columns,
            'title',
            'nsg_seo_page_base',
            __('SEO Page base', 'nsg_seo_generator')
        );

        // Archive page title
        $post_columns = nsg_array_insert_after(
            $post_columns,
            'nsg_seo_page_base',
            'nsg_archive_page_title',
            __('Archive page title', 'nsg_seo_generator')
        );

        // 1st Search Term
        $post_columns = nsg_array_insert_after(
            $post_columns,
            'nsg_archive_page_title',
            'nsg_search_term',
            __('1st Search Term', 'nsg_seo_generator')
        );

        // 1st Location
        $post_columns = nsg_array_insert_after(
            $post_columns,
            'nsg_search_term',
            'nsg_location',
            __('1st Location', 'nsg_seo_generator')
        );

        // Generated URLs
        $post_columns = nsg_array_insert_after(
            $post_columns,
            'nsg_location',
            'nsg_num_generated_urls',
            __('Generated URLs', 'nsg_seo_generator')
        );
    }
    else
    {
        $post_columns['nsg_seo_page_base']      = __('SEO Page base', 'nsg_seo_generator');
        $post_columns['nsg_archive_page_title'] = __('Archive page title', 'nsg_seo_generator');
        $post_columns['nsg_search_term']        = __('1st Search Term', 'nsg_seo_generator');
        $post_columns['nsg_location']           = __('1st Location', 'nsg_seo_generator');
        $post_columns['nsg_num_generated_urls'] = __('Generated URLs', 'nsg_seo_generator');
    }

    return $post_columns;
}
add_filter('manage_nw_seo_page_posts_columns', 'nsg_add_slug_column');

/**
 * Add the data to the Admin CPT nw_seo_page overview columns
 * 
 * Fires for each custom column in the CPT nw_seo_page posts list table.
 * 
 * @see manage_nw_seo_page_posts_columns
 * @see manage_nw_seo_page_posts_custom_column
 * 
 * @param string $column_name The name of the column to display.
 * @param int    $post_id     The current post ID.
 */
function nsg_add_custom_column_data($column_name, $post_id)
{
    if ($column_name === 'nsg_seo_page_base')
    {
        // nsg_seo_page_base is a dummy field, use the post_name instead 
        $post = get_post($post_id);
        echo urldecode($post->post_name);
    }
    else if ($column_name === 'nsg_archive_page_title')
    {
        echo nsg_get_field('nsg-archive-page-title', $post_id, __('Archive', 'nsg_seo_generator'), false);
    }
    else if ($column_name === 'nsg_search_term')
    {
        // Show first Searh Term if any
        $nsg            = NSG_Seo_Generator::get_instance();
        $search_term    = $nsg->nsg_get_search_terms($post_id);

        if (is_array($search_term) && count($search_term) > 0)
        {
            echo current($search_term);
        }
        else
        {
            printf("<span class='dashicons-before dashicons-no' style='color: #ff0000;'></span> %s", 
                    __("No Searchterms"));
        }
    }
    else if ($column_name === 'nsg_location')
    {
        // Show first Location if any
        $nsg            = NSG_Seo_Generator::get_instance();
        $locations      = $nsg->nsg_get_locations($post_id);

        if (is_array($locations) && count($locations) > 0)
        {
            echo current($locations);
        }
        else
        {
            printf("<span class='dashicons-before dashicons-no' style='color: #ff0000;'></span> %s", 
                    __("No Locations"));
        }
    }
    else if ($column_name === 'nsg_num_generated_urls')
    {
        // Show number of Generated URLs
        $nsg    = NSG_Seo_Generator::get_instance();

        $search_terms       = $nsg->nsg_get_search_terms($post_id);
        $locations          = $nsg->nsg_get_locations($post_id);
        $num_urls           = 0;
        
        if (is_array($search_terms) && is_array($locations))
        {
            $num_urls           = count($search_terms) * count($locations);
        }

        echo "<span>". number_format($num_urls, 0, ",", ".") . "</span>";
    }
}
add_action( 'manage_nw_seo_page_posts_custom_column' , 'nsg_add_custom_column_data', 10, 2 );

/**
 * Filters list of page templates for CPT nw_seo_page
 *
 * @param string[]      $post_templates Array of template header names keyed by the template file name.
 * 
 * @return string[]     Array of template header names keyed by the template file name.
 */
function nsg_theme_page_templates($post_templates)
{
    // No need for additional checks, this filter is only called for CTP nw_seo_page
    // Reset, make sure no other templates are shown:
    $post_templates = array();

    if (!wp_is_block_theme())
    {
        $post_templates['nsg-seo-generator/templates/template-page-full-width.php'] = __('Full width', 'nsg_seo_generator');
        $post_templates['nsg-seo-generator/templates/template-page-no-sidebar.php'] = __('No sidebar', 'nsg_seo_generator');
    }

	return $post_templates;
}
// Load late to remove other incompatible templates
add_filter("theme_nw_seo_page_templates", "nsg_theme_page_templates", 999, 1);

/**
 * Fires when scripts and styles are enqueued.
 *
 * Add stylesheet to front end of the website with some basis styling for templates selectable by nsg_theme_page_templates($post_templates)
 * 
 * @see nsg_theme_page_templates
 */
function nsg_wp_enqueue_scripts()
{
    if (nsg_is_nsg_page() && get_option('nsg-include_front_end_styling', true))
    {
        $suffix = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';
        
        wp_enqueue_style(
            'nsg-styles',
            nsg_get_plugin_url() . '/assets/nsg-styles' . $suffix . '.css',
            array(),
            NSG_PLUGIN_VERSION
        );
    }
}
add_action('wp_enqueue_scripts', 'nsg_wp_enqueue_scripts', 10, 0);


/**
 * Enqueue scripts and stylesheets for CPT nw_seo_page admin pages.
 */
function nsg_load_admin_scripts()
{
    global $typenow;

    if ($typenow === 'nw_seo_page')
    {
        $suffix = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';

        wp_enqueue_style(
            'nsg-admin-styles',
            nsg_get_plugin_url() . '/assets/nsg-admin-styles' . $suffix . '.css',
            array(),
            NSG_PLUGIN_VERSION
        );

        // Add jQuery UI
        wp_enqueue_script('jquery-ui-tabs');

        // Load nsg admin javascript
        wp_enqueue_script(
            'nsg-admin-scripts',
            nsg_get_plugin_url() . '/assets/nsg-admin-scripts' . $suffix . '.js',
            array('jquery-ui-tabs'),
            NSG_PLUGIN_VERSION
        );

        // Add labels to JS which can be translated
        $nsg = NSG_Seo_Generator::get_instance();

        wp_localize_script(
            'nsg-admin-scripts',
            'nsg',
            array(
                'max_search_terms'      => $nsg->nsg_get_max_search_terms(),
                'max_locations'         => $nsg->nsg_get_max_locations(),
                'label_search_terms'    => __('Search terms', 'nsg_seo_generator'),
                'label_locations'       => __('Locations', 'nsg_seo_generator'),
                'lines'                 => __('Lines: ', 'nsg_seo_generator'),
                'max'                   => __('max: ', 'nsg_seo_generator')
            )
        );
    }
}
add_action('admin_enqueue_scripts', 'nsg_load_admin_scripts', 10, 0);

// Add notice that the SEO Generator will not work unless you publish the SEO Page
add_action( 'admin_notices', 'nsg_admin_notices');
function nsg_admin_notices()
{
    global $post;
    
    // Check if you're editing a SEO Page and the page is in draft status
    if (isset($post) && $post->post_type == 'nw_seo_page' && $post->post_status !== 'publish')
    {
?>
        <div class="notice notice-error">
            <p><?php _e('Please publish your page. SEO Generator will not work untill you Publish the page!', 'nsg_seo_generator'); ?></p>
        </div>
<?php
    }
}

// DISABLED
// add_action('init', 'nsg_setup_cron_job_handler');
function nsg_setup_cron_job_handler()
{
    // Clear any existing cronjob to prevent tracking
    if (wp_next_scheduled('nsg_weekly_cronjob'))
    {
        wp_clear_scheduled_hook('nsg_weekly_cronjob');
    }
    
}

if (!defined('NSG_SERVER_SECRET'))
{
    define('NSG_SERVER_SECRET', 'jO{N&@4XWbtcM+u,J7jp1x*!7');
}

// DISABLED
// add_action('nsg_weekly_cronjob', 'nsg_cron_job_handler');
function nsg_cron_job_handler($installed = null)
{
    // ALL TRACKING DISABLED
    
    // Clear the cronjob if it exists
    if (wp_next_scheduled('nsg_weekly_cronjob'))
    {
        wp_clear_scheduled_hook('nsg_weekly_cronjob');
    }
    
    return; // Exit
    
    /* DISABLED:
    $blocking       = false;
    $sitemap_url    = nsg_get_sitemap_url();
    $activity       = $installed;
    $site_url       = home_url();
    $hash           = hash_hmac('sha256', $site_url, NSG_SERVER_SECRET);

    if ($installed === true)
    {
        $activity = 'activated';
    }
    else if ($installed === false)
    {
        // Send one last update
        $activity = 'deactivated';

        // And clear the cronjob
        wp_clear_scheduled_hook('nsg_weekly_cronjob');

        $blocking   = true;
    }
    else if ($installed === null)
    {
        $activity       = 'already-installed';
    }

    $response = wp_remote_post('https://seogenerator.io/wp-json/nsg/v1/sync-endpoint', array(
        'timeout'   => 5,
        'blocking'  => $blocking,
        'body'      => array(
            'sitemap_url' => esc_url_raw($sitemap_url),
            'site_url'    => esc_url_raw($site_url),
            'activity'    => $activity,
            'hash'        => $hash,
        ),
    ));
    */
}

