<?php

/**
 * Determine whether the SEO Generator archive page should be set to noindex.
 *
 * This checks if the current request is for the SEO Generator archive page,
 * and whether the 'noindex archive page' setting is enabled.
 *
 * @return bool True if the archive page should be noindexed, false otherwise.
 */
function nsg_should_noindex_archive_page()
{
    if (!nsg_is_nsg_archive_page())
	{
		return false;
	}

    $seo_page_id                    = get_the_ID();
    $enable_archive_page            = nsg_get_field('nsg-enable_archive_page', $seo_page_id, true, false);

    $enable_noindex_archive_page    = false;
    if ($enable_archive_page)
    {
        $seo_page_id                    = nsg_get_seo_archive_page_id();
        $enable_noindex_archive_page    = (bool)nsg_get_field('nsg-enable_noindex_archive_page', $seo_page_id, true, false);
    }
    
    return $enable_noindex_archive_page;
}

/**
 * Outputs the SEO Generator archive page content.
 *
 * Used in shortcode [nsg-archive] and directly by the archive template.
 *
 * @return void
 */
function nsg_render_archive_listing()
{
    if (is_admin() || !nsg_is_nsg_archive_page())
    {
        return '[nsg-archive]';
    }

    // First fetch the $seo_page_id
    $seo_page_id = nsg_get_seo_archive_page_id();

	$post = get_post($seo_page_id);
	if (!$post)
	{
        // Should never occur, just in case
		echo '<p>' . __('Archive page not found.', 'nsg_seo_generator') . '</p>';
		return;
	}

    $enable_archive_page = nsg_get_field('nsg-enable_archive_page', $seo_page_id, true, false);

    if ($enable_archive_page)
    {
        $default_post_title_to_replace  = $post->post_title;
        $seo_page_base                  = $post->post_name;

        $lookup_table = nsg_get_search_terms_and_locations_lookup_table($seo_page_id);
        if (empty($lookup_table))
        {
            $lookup_table = nsg_regenerate_search_terms_and_locations_lookup_tables($seo_page_id);
        }

        if (empty($lookup_table))
        {
            // Should never occur
            echo "SAVE THE SEO GENERATOR PAGE FIRST!";
            exit;
        }

        if (is_array($lookup_table) && !empty($lookup_table))
        {
            $keys         = array_keys($lookup_table);

            $current_page  = max(1, get_query_var('paged'));
            $num_per_page  = apply_filters('nsg_archive_num_per_page', 10);
            $num_seo_pages = count($lookup_table);
            $max_pages     = (int)ceil($num_seo_pages / $num_per_page);
            $offset        = $num_per_page * ($current_page - 1);
            $nw_seo_pages  = array_slice($lookup_table, $offset, $num_per_page);

            $index = $offset;
            $orig_post_content  = $post->post_content;

            echo '<div class="nsg-archive-listing">';
            foreach ($nw_seo_pages as $nw_seo_page)
            {
                list($search_term, $location, $search_terms, $locations) = $nw_seo_page;

                $the_slug   = $keys[$index];

                $params     = array(
                    'the_post_id'			=> $seo_page_id,	
                    'search_term'			=> $search_term,
                    'location'				=> $location,
                    'search_term_plural'	=> $search_terms,
                    'location_plural'		=> $locations,
                    'the_slug'				=> $the_slug,
                    'spintax_offset'		=> $index,
                    'context'				=> 'title',
                );
                $the_title = nsg_get_seo_pages_replace_search_terms_and_locations($default_post_title_to_replace, $params);
                $the_url   = esc_url(trailingslashit(home_url($seo_page_base . '/' . strtolower(sanitize_title($the_slug)))));

                $post_excerpt = get_the_excerpt($seo_page_id);
                // Custom Excerpt
                if (!empty($post_excerpt) && get_option('nsg-enable_custom_excerpt_tag', false) && !get_option('nsg-enable_spintax', false))
                {
                    // Overwrite:
                    $params['context']          = 'excerpt';

                    $post_excerpt           = do_shortcode($post_excerpt);
                    $post_excerpt           = nsg_get_seo_pages_replace_search_terms_and_locations($post_excerpt, $params);
                    
                    $custom_excerpt_length  = (int)get_option('nsg-custom_excerpt_length', 55);
                    $post_excerpt           = wp_trim_words($post_excerpt, $custom_excerpt_length);
                }
                // For the Excerpt we need to first do our own search and replace AND spintax rotation
                // This is needed because if text A and B for instance are really long in { A | B | C },
                // Than the entire spintax tag is cut on in the excerpt causing the spintax to fail. { A | B is not a valid and closed spintax
                else if (get_option('nsg-enable_spintax', false))
                {
                    // Reset the post and content. If Spintax was already replaced, we cannot replace it again with different values for a diffent seach_term and location
                    $post_content   = $orig_post_content;

                    // Overwrite:
                    $params['context']          = 'excerpt';

                    // Replace search terms and locations. And do spintax rotation if enabled
                    if (get_option('nsg-enable_custom_excerpt_tag', false))
                    {
                        // Process the custom excerpt with search term replacements FIRST
                        $post_excerpt = nsg_get_seo_pages_replace_search_terms_and_locations($post_excerpt, $params);
                        
                        $custom_excerpt_length  = (int)get_option('nsg-custom_excerpt_length', 55);
                        $post_excerpt           = wp_trim_words($post_excerpt, $custom_excerpt_length);
                    }
                    else
                    {
                        $post_excerpt       = nsg_get_seo_pages_replace_search_terms_and_locations($post_content, $params);

                        // Fake it:
                        $post->post_excerpt = '';
                        $post->post_content = $post_excerpt;

                        $my_excerpt_more_filter = function($more) use ($the_slug, $params) {
                            return nsg_archive_post_link($more, $the_slug, $params);
                        };
                        add_filter('excerpt_more', $my_excerpt_more_filter, 9999, 1);

                        // Now finally call the default wordpress excerpt to make sure filters like excerpt_length are also working
                        $post_excerpt = get_the_excerpt($post);

                        remove_filter('excerpt_more', $my_excerpt_more_filter, 9999);
                    }
                }
                else
                {
                    $post_excerpt = get_the_excerpt();
                }

                // Overwrite:
                $params['context']          = 'excerpt';
                
                // Remove [nsg-toc] shortcode from excerpt content
                $post_excerpt = str_replace('[nsg-toc]', '', $post_excerpt);
                $post_excerpt = trim($post_excerpt);
                
                echo "<div class='nsg-seo-page'>";
                echo sprintf("<h2><a class='nsg-seo-page' href='%s'>%s</a></h2>", $the_url, $the_title);
                echo "<p>" . nsg_get_seo_pages_replace_search_terms_and_locations($post_excerpt, $params) . "</p>";
                echo "</div>";

                $index++;
            }
            echo '</div>';

            echo '<nav class="nsg-pagination">';
            echo paginate_links(array(
                'total'     => $max_pages,
                'current'   => $current_page,
                'add_args'  => false,
                'type'      => 'list',
                'end_size'  => 3,
                'mid_size'  => 3,
                'show_all'  => false,
                'prev_next' => false
            ));
            echo '</nav>';
        }
    }
}

function nsg_archive_post_link($more, $the_slug, $params)
{
	$dom = new DOMDocument();
	libxml_use_internal_errors(true);

    // Deprecated: mb_convert_encoding(): Handling HTML entities via mbstring is deprecated
    // $dom->loadHTML(mb_convert_encoding($more, 'HTML-ENTITIES', 'UTF-8'));
    // Fix: Use htmlentities instead of mb_convert_encoding for HTML entities
    $dom->loadHTML(htmlentities($more, ENT_QUOTES | ENT_HTML5, 'UTF-8', false));

	$links = $dom->getElementsByTagName('a');
	foreach ($links as $link)
	{
        $params['context'] = 'title';
        
        $title = nsg_get_seo_pages_replace_search_terms_and_locations(get_the_title($params['the_post_id']), $params);
		$link->setAttribute('title', $title);
		$link->setAttribute('href', trailingslashit(trailingslashit(get_permalink($params['the_post_id'])) . $the_slug));
		$link->setAttribute('aria-label', $title);
	}

	if ($links->length > 0)
	{
		$more = ' ... ' . $dom->saveHTML($links->item(0));
	}

	return $more;
}

add_shortcode('nsg-archive', 'nsg_archive_shortcode_handler');
function nsg_archive_shortcode_handler($atts)
{
    $result = '[nsg-archive]';
    if (!is_admin())
    {
        // Buffer output to make sure the result of the [nsg-archive] is actually printed in the correct location where the shortcode is added to the page
        ob_start();
        nsg_render_archive_listing();
        $result =  ob_get_clean();
    }

    return $result;
}
