<?php

/**
 * Holds NSG_Meta_Field fields to store HTML documentation how to use the plugin
 */
class NSG_Documentation
{
	/**
	 * List of tabs labels
	 * 
	 * @var array 
	 */
	private $tabs	= array();

	/**
	 * List of tab panels with NSG_Meta_Field fields
	 * 
	 * @var array 
	 */
	private $blocks = array();

	/**
	 * Nonce value that was used for verification, usually via a form field.
	 * 
	 * @var string
	 */
	private $nonce_name;

	/**
	 * Metabox ID
	 * 
	 * @var string
	 */
	private $nsg_documentation_page_id = 'nw_seo_page_page_nsg-documentation';

	/**
	 * Creates an instance of NSG_Documentation
	 */
	public function __construct()
	{
		$meta_box_config = array(
			'id'	=> 'nsg-meta-box-documentation',
			'title'	=> __('SEO Generator documentation', 'nsg_seo_generator'),
			'context'	=> 'normal',
			'priority'	=> 'core',
			'screen'	=> 'nw_seo_page_page_documentation'
		);
		
		$this->nonce_name 		= $meta_box_config['id'] . '_nonce';

		add_action('admin_menu', array($this, 'nsg_admin_menu'));

		if (nsg_is_documentation_page())
		{
			add_action('add_meta_boxes', array($this, 'add_meta_boxes'));

			// Load the JavaScript needed for the documentation screen
			add_action('admin_enqueue_scripts', array($this, 'nsg_options_page_enqueue_scripts'));
		}
	}

	/**
	 * Adds new submenu page with tabs and fields to hold documentation
	 */
	public function nsg_admin_menu()
	{
		// Add documentation menu page
        $documentation_page = add_submenu_page(
			'edit.php?post_type=nw_seo_page',
			__('Documentation', 'nsg_seo_generator'),
			__('Documentation', 'nsg_seo_generator'),
			'edit_posts',
			'nsg-docs',
			array($this, 'nsg_docs_page'),
			null
		);

        if ($documentation_page)
		{
			$nsg = NSG_Seo_Generator::get_instance();

			$html_field = new NSG_Meta_HTML_Field(
				array(
					'attr'          => array(
						'type'      => 'html',
						'id'    	=> 'documentation',
					),
					'wrapper_class' => array('nsg-small-12'),
					'is_option'     => true
				)
			);

			$documentation_block = new NSG_Meta_Block();
			$documentation_block->add_field($html_field);
			
            $this->add_block("nsg-documentation-documentation", __('Documentation', 'nsg_seo_generator'), $documentation_block);
		}
	}

	/**
	 * Loads script needed for the Metabox
	 * 
	 * @param string $hook_suffix The current admin page.
	 */
	public function nsg_options_page_enqueue_scripts($hook_suffix)
	{
		$page_hook_id = $this->nsg_documentation_page_id;

		if ($hook_suffix == $page_hook_id)
		{
			wp_enqueue_script( 'common' );
			wp_enqueue_script( 'wp-lists' );
			wp_enqueue_script( 'postbox' );
		}
	}

	/**
	 * Documentation Page Callback
	 */
	function nsg_docs_page()
	{
		$hook_suffix = $this->nsg_documentation_page_id;
		
		// enable add_meta_boxes function in this page.
		do_action('add_meta_boxes', $hook_suffix, false );
	?>
		<div class="wrap">
			<h2><?php _e('SEO Generator Documentation', 'nsg_seo_generator'); ?></h2>

			<?php settings_errors(); ?>

			<div class="nsg-documentation-meta-box-wrap">

				<form id="nsg-form" method="post" action="options.php">

					<?php settings_fields( 'nsg' ); // options group  ?>
					<?php wp_nonce_field( 'closedpostboxes', 'closedpostboxesnonce', false ); ?>
					<?php wp_nonce_field( 'meta-box-order', 'meta-box-order-nonce', false ); ?>

					<div id="poststuff">

						<div id="post-body" class="metabox-holder columns-<?php echo 1 == get_current_screen()->get_columns() ? '1' : '2'; ?>">

							<div id="postbox-container-1" class="postbox-container">
								<?php do_meta_boxes( $hook_suffix, 'side', null ); ?>
							</div>

							<div id="postbox-container-2" class="postbox-container">
								<?php do_meta_boxes( $hook_suffix, 'normal', null ); ?>
								<?php do_meta_boxes( $hook_suffix, 'advanced', null ); ?>
							</div>

						</div>
						<br class="clear">
					</div>
				</form>
			</div>
		</div>
	<?php
	}

	/**
	 * Adds meta boxes
	 */
	public function add_meta_boxes()
	{
		$page_hook_id = $this->nsg_documentation_page_id;

		// Save Options page meta box on the right side:
		add_meta_box(
			'submitdiv',               
			__('Save options', 'nsg_seo_generator'),
			array($this, 'nsg_submit_meta_box'),
			$page_hook_id,
			'side',
			'high'
		);

		add_meta_box(
			'nsg-documentation',
			__('Documentation', 'nsg_seo_generator'),
			array($this, 'show'),
			$page_hook_id,
			'normal',
			'default'
		);
	}

	/**
	 * Submit meta box callback
	 */
	public function nsg_submit_meta_box()
	{
	?>
	<div id="submitpost" class="submitbox">
		<div id="major-publishing-actions">
			<div id="publishing-action">
				<span class="spinner"></span>
				<?php submit_button( esc_attr( 'Save' ), 'primary', 'submit', false );?>
			</div>
			<div class="clear"></div>
		</div>
	</div>

	<?php
	}

	/**
	 * Adds NSG_Meta_Block
	 * 
	 * @param int				$tab_id		The tab ID
	 * @param string			$tab_title	The tab title
	 * @param NSG_Meta_Block 	$block		The tab pabel content with a NSG_Meta_Block object
	 */
	public function add_block($tab_id, $tab_title, $block)
	{
		if (apply_filters('nsg-add-block', true, $tab_id, $tab_title, $block))
		{
			$this->blocks[]	= $block;
		}
	}

	/**
	 * Shows fake meta box with tabs and tab panels with custom NSG_Meta_Field fields
	 */
	public function show()
	{
		wp_nonce_field(basename(__FILE__), $this->nonce_name);

		echo "\r\n<div id='nsg-documentation' class='ui-helper-clearfix'>\r\n";

		foreach ($this->blocks as $block)
		{
			$block->show();
		}

		echo "</div>\r\n";
	}
}

add_action('init', 'init_nsg_documentation');
function init_nsg_documentation()
{
    if (is_admin())
	{
		new NSG_Documentation();
	}
}
