<?php

class NSG_Spintax
{
	private $text               = '';
	private $choices            = array();
	private $total_combinations = -1;

	public function __construct($text)
	{
		$this->text                 = $text;
		$this->choices              = $this->extract_choices($text);
		$this->total_combinations   = $this->calculatetotal_combinations($this->choices);
	}

	/**
	 * Return true if the given inner text contains a top-level pipe (|) outside nested braces.
	 */
	private function has_top_level_pipe($inner)
	{
		$depth = 0;
		$len = mb_strlen($inner, 'UTF-8');

		for($i = 0; $i < $len; $i++)
		{
			$ch = mb_substr($inner, $i, 1, 'UTF-8');

			if($ch === '{')
			{
				$depth++;
			}
			else if($ch === '}')
			{
				if($depth > 0)
				{
					$depth--;
				}
			}
			else if($ch === '|' && $depth === 0)
			{
				return true;
			}
		}

		return false;
	}

	/**
	 * Split by unescaped top-level pipes. Keeps inner whitespace as-is.
	 */
	private function split_choices($inner)
	{
		$parts = array();
		$buf = '';
		$depth = 0;
		$len = mb_strlen($inner, 'UTF-8');
		$escape = false;

		for($i = 0; $i < $len; $i++)
		{
			$ch = mb_substr($inner, $i, 1, 'UTF-8');

			if($escape)
			{
				$buf .= $ch;
				$escape = false;
				continue;
			}

			if($ch === '\\')
			{
				$buf .= $ch;
				$escape = true;
				continue;
			}

			if($ch === '{')
			{
				$depth++;
				$buf .= $ch;
				continue;
			}

			if($ch === '}')
			{
				if($depth > 0)
				{
					$depth--;
				}
				$buf .= $ch;
				continue;
			}

			if($ch === '|' && $depth === 0)
			{
				$parts[] = $buf;
				$buf = '';
				continue;
			}

			$buf .= $ch;
		}

		$parts[] = $buf;

		// Trim each choice with a Unicode-aware trim, but do NOT collapse internal whitespace.
		$custom_trim = function($value)
		{
			return preg_replace('/^[\p{Z}\p{C}]+|[\p{Z}\p{C}]+$/u', '', $value);
		};

		return array_map($custom_trim, $parts);
	}

	private function extract_choices($text)
	{
		$choices = array();

		if(!NSG_Dom_Node::contains_spintax($text))
		{
			return $choices;
		}

		if(!empty($text))
		{
            // Prevent memory exhaustion and DoS attacks by limiting text size to 2MB
            if (strlen($text) > 2097152)
            {
                return array();
            }

			// Match nested {…} blocks using recursive regex pattern
			preg_match_all('/\{(((?>[^\{\}]+)|(?R))*)\}/s', $text, $matches);

			foreach($matches[1] as $inner)
			{
				// Only treat as spintax if there is a top-level pipe.
				if($this->has_top_level_pipe($inner))
				{
					$choices[] = $this->split_choices($inner);
				}
			}
		}

		return $choices;
	}

	private function calculatetotal_combinations($choices)
	{
		$num_combinations = 1;
		if(!empty($choices))
		{
			$num_combinations = array_product(array_map('count', $choices));
		}
		return $num_combinations;
	}

	public function get_combination($index_offset = 0, $context = false)
	{
		$result = $this->text;

		// If pure JSON or no valid spintax choices, return as-is.
		if(NSG_Dom_Node::is_json($result) || empty($this->choices))
		{
			return $result;
		}

		$context = nsg_get_grouped_context($context);

		$nsg     = NSG_Seo_Generator::get_instance();
		$post_id = get_the_ID();
		global $wp_query;
		if($wp_query->original_seo_page_id !== null)
		{
			$post_id = $wp_query->original_seo_page_id;
		}

		$index              = $nsg->nsg_get_lookup_table_slug_index() + $index_offset;
		$sequential_index   = $index;
		$seo_page_index     = $index;
		$combinations       = $this->total_combinations;

		if($combinations === 0)
		{
			$combinations = 1;
		}
		if($combinations > PHP_INT_MAX)
		{
			$combinations = PHP_INT_MAX;
		}

		$sequential_index = $sequential_index % $combinations;

		$mode = intval(get_option('nsg-randomize_spintax', 0));

		$persistent_choices = array();
		$lookup_table = nsg_get_search_terms_and_locations_lookup_table($post_id);

		if(empty($lookup_table))
		{
			nsg_regenerate_search_terms_and_locations_lookup_tables($post_id);
			$lookup_table = nsg_get_search_terms_and_locations_lookup_table($post_id);
		}

		if(empty($lookup_table))
		{
			echo "SAVE THE SEO GENERATOR PAGE FIRST!";
			exit;
		}

		$slug_keys    = array_keys($lookup_table);
		$current_slug = isset($slug_keys[$seo_page_index]) ? $slug_keys[$seo_page_index] : '';

		$do_save_persistent_choices = false;
		if($mode === 2 && $current_slug)
		{
			if(isset($lookup_table[$current_slug]['_spintax_choices']) && !empty($lookup_table[$current_slug]['_spintax_choices']))
			{
				$persistent_choices = $lookup_table[$current_slug]['_spintax_choices'];
			}
			else
			{
				$do_save_persistent_choices = true;
			}
		}

		// Precompute the chosen index for each valid spintax block.
		$selected_indices = array();
		if(!empty($this->choices))
		{
			foreach($this->choices as $i => $block_choices)
			{
				if(empty($block_choices))
				{
					$selected_indices[$i] = 0;
					continue;
				}

				$block_hash = md5(serialize($block_choices) . $current_slug . $context);

				if($mode === 1)
				{
					$selected_indices[$i] = array_rand($block_choices);
				}
				else if($mode === 2)
				{
					if(!isset($persistent_choices[$block_hash]))
					{
						$persistent_choices[$block_hash] = array();
					}

					if(isset($persistent_choices[$block_hash][$i]))
					{
						$selected_indices[$i] = $persistent_choices[$block_hash][$i];
					}
					else
					{
						$selected_indices[$i] = array_rand($block_choices);
						$persistent_choices[$block_hash][$i] = $selected_indices[$i];
						$do_save_persistent_choices = true;
					}
				}
				else
				{
					$block_count = count($block_choices);
					if($block_count === 0)
					{
						$block_count = 1;
					}
					$selected_indices[$i] = $sequential_index % $block_count;
					$sequential_index = intdiv($sequential_index, $block_count);
				}
			}
		}

		// Replace only true spintax blocks via callback; preserve other {…} blocks intact.
		$choice_cursor = 0;
		$self = $this;

		$result = preg_replace_callback('/\{(((?>[^\{\}]+)|(?R))*)\}/u', function($m) use (&$choice_cursor, $self, $selected_indices)
		{
			$inner = $m[1];

			if(!$self->has_top_level_pipe($inner))
			{
				return $m[0];
			}

			if(!isset($self->choices[$choice_cursor]) || empty($self->choices[$choice_cursor]))
			{
				$choice_cursor++;
				return $m[0];
			}

			$idx = isset($selected_indices[$choice_cursor]) ? $selected_indices[$choice_cursor] : 0;
			$choice_cursor++;

			return $self->choices[$choice_cursor - 1][$idx];
		}, $result);

		if($mode === 2 && $current_slug)
		{
			if($do_save_persistent_choices)
			{
				$lookup_table[$current_slug]['_spintax_choices'] = $persistent_choices;
				update_post_meta($post_id, 'nsg-search-word-and-location-for-slugs', $lookup_table);
			}
		}

		return $result;
	}

	public function get_total_combinations()
	{
		return $this->total_combinations;
	}
}
