<?php

/**
 * NSG Table of Contents Class
 * Integrated Table of Contents functionality for NSG SEO Generator
 */
class NSG_Table_Of_Contents
{
    private static $instance = null;
    private $shortcode_used = false;

    /**
     * Get singleton instance
     */
    public static function get_instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Initialize the TOC functionality
     */
    public function __construct()
    {
        add_shortcode('nsg-toc', array($this, 'render_toc_shortcode'));
        // TOC is now added via nsg_maybe_add_toc_to_content_late() at priority 99999999+3
        // This ensures spintax is already resolved before TOC is generated
        add_action('wp_enqueue_scripts', array($this, 'enqueue_toc_scripts'));
    }

    /**
     * Enqueue TOC scripts and styles
     */
    public function enqueue_toc_scripts()
    {
        if (!nsg_is_nsg_page()) {
            return;
        }

        $post_id = nsg_get_seo_archive_page_id();
        $enable_toc = get_post_meta($post_id, 'nsg-enable_toc', true);
        
        if ($enable_toc) {
            $suffix = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';
            
            wp_enqueue_style(
                'nsg-toc-style',
                nsg_get_plugin_url() . '/assets/nsg-toc-style' . $suffix . '.css',
                array(),
                NSG_PLUGIN_VERSION
            );
            
            wp_enqueue_script(
                'nsg-toc-script',
                nsg_get_plugin_url() . '/assets/nsg-toc-script' . $suffix . '.js',
                array(),
                NSG_PLUGIN_VERSION,
                true
            );
        }
    }

    /**
     * Maybe add TOC to content automatically (LATE - after spintax resolved)
     */
    public function maybe_add_toc_to_content_late($content)
    {
        // Note: We removed is_main_query() and in_the_loop() checks because at this late priority,
        // some themes/plugins may have already closed the loop, causing false negatives.
        // The nsg_maybe_add_toc_to_content_late() wrapper function already checks nsg_is_nsg_page()
        
        if (!nsg_is_nsg_page()) {
            return $content;
        }

        // Don't add TOC to archive pages, only to individual SEO pages
        if ((int)get_query_var('nsg_seo_page_archive')) {
            return $content;
        }

        $post_id = nsg_get_seo_archive_page_id();
        $enable_toc = get_post_meta($post_id, 'nsg-enable_toc', true);

        if (!$enable_toc) {
            return $content;
        }

        $toc_html = $this->generate_toc_late($content, $post_id);
        
        if (empty($toc_html)) {
            return $content;
        }

        // If shortcode placeholder exists, replace it with actual TOC
        if ($this->shortcode_used && strpos($content, '<!--NSG_TOC_PLACEHOLDER-->') !== false) {
            return str_replace('<!--NSG_TOC_PLACEHOLDER-->', $toc_html, $content);
        }

        // Otherwise add TOC before content (automatic placement)
        return $toc_html . $content;
    }

    /**
     * Render TOC shortcode
     */
    public function render_toc_shortcode($atts = array())
    {
        // Set flag that shortcode is used
        $this->shortcode_used = true;
        
        if (!nsg_is_nsg_page()) {
            return '';
        }

        // Don't show TOC shortcode on archive pages
        if ((int)get_query_var('nsg_seo_page_archive')) {
            return '';
        }

        $post_id = nsg_get_seo_archive_page_id();
        $enable_toc = get_post_meta($post_id, 'nsg-enable_toc', true);

        if (!$enable_toc) {
            return '';
        }

        // Return placeholder that will be replaced with actual TOC after spintax is resolved
        return '<!--NSG_TOC_PLACEHOLDER-->';
    }

    /**
     * Generate Table of Contents HTML
     */
    public function generate_toc($content, $post_id)
    {
        $headings = $this->extract_headings($content, $post_id);
        
        if (empty($headings)) {
            return '';
        }

        return $this->generate_toc_html($headings, $post_id);
    }

    /**
     * Generate Table of Contents HTML (LATE - content already has spintax resolved)
     */
    public function generate_toc_late($content, $post_id)
    {
        $headings = $this->extract_headings_late($content, $post_id);
        
        if (empty($headings)) {
            return '';
        }

        return $this->generate_toc_html($headings, $post_id);
    }

    /**
     * Extract headings from content
     */
    private function extract_headings($content, $post_id)
    {
        // First process the content through NSG filters to handle Spintax and placeholders
        $params = array('context' => 'heading');
        $processed_content = nsg_get_seo_pages_replace_search_terms_and_locations($content, $params);
        
        return $this->extract_headings_from_processed_content($processed_content, $post_id);
    }

    /**
     * Extract headings from content that already has spintax resolved
     */
    private function extract_headings_late($content, $post_id)
    {
        // Content already has spintax resolved, use as-is
        return $this->extract_headings_from_processed_content($content, $post_id);
    }

    /**
     * Extract headings from processed content (internal)
     */
    private function extract_headings_from_processed_content($processed_content, $post_id)
    {
        
        $include_h3 = get_post_meta($post_id, 'nsg-toc_include_h3', true);
        
        if ($include_h3) {
            $pattern = '/<h([23])[^>]*>(.*?)<\/h[23]>/i';
        } else {
            $pattern = '/<h2[^>]*>(.*?)<\/h2>/i';
        }

        preg_match_all($pattern, $processed_content, $matches, PREG_SET_ORDER);

        $headings = array();
        $h2_counter = 0;
        $h3_counter = 0;

        foreach ($matches as $match) {
            $level = isset($match[2]) && $match[2] ? intval($match[1]) : 2;
            $text = isset($match[2]) ? strip_tags($match[2]) : strip_tags($match[1]);
            $text = trim($text);
            
            if (empty($text)) {
                continue;
            }

            if ($level == 2) {
                $h2_counter++;
                $h3_counter = 0;
                $number = $h2_counter;
            } else {
                $h3_counter++;
                $number = $h2_counter . '.' . $h3_counter;
            }

            $anchor = sanitize_title($text);
            $anchor = 'toc-' . $anchor;

            $headings[] = array(
                'text' => $text,
                'anchor' => $anchor,
                'level' => $level,
                'number' => $number
            );
        }

        return $headings;
    }

    /**
     * Generate TOC HTML structure
     */
    private function generate_toc_html($headings, $post_id)
    {
        $toc_title = get_post_meta($post_id, 'nsg-toc_title', true);
        $toc_numbered = get_post_meta($post_id, 'nsg-toc_numbered', true);
        $toc_toggle = get_post_meta($post_id, 'nsg-toc_toggle', true);

        if (empty($toc_title)) {
            $toc_title = __('Table of Contents', 'nsg_seo_generator');
        }

        $toc_html = '<div class="nsg-table-of-contents" id="nsg-toc">';
        
        // TOC Header
        $toc_html .= '<div class="nsg-toc-header">';
        $toc_html .= '<h3 class="nsg-toc-title">' . esc_html($toc_title) . '</h3>';
        
        if ($toc_toggle) {
            $toc_html .= '<button class="nsg-toc-toggle" type="button">';
            $toc_html .= '<span class="nsg-toc-toggle-text">[' . __('hide', 'nsg_seo_generator') . ']</span>';
            $toc_html .= '</button>';
        }
        
        $toc_html .= '</div>';

        // TOC Content
        $toc_html .= '<div class="nsg-toc-content">';
        $toc_html .= $toc_numbered ? '<ol class="nsg-toc-list">' : '<ul class="nsg-toc-list">';

        $current_level = 2;
        $open_sublists = 0;

        foreach ($headings as $heading) {
            $level = $heading['level'];
            
            if ($level > $current_level) {
                // Opening sublevel
                $toc_html .= $toc_numbered ? '<ol>' : '<ul>';
                $open_sublists++;
            } elseif ($level < $current_level && $open_sublists > 0) {
                // Closing sublevel
                $toc_html .= $toc_numbered ? '</ol>' : '</ul>';
                $toc_html .= '</li>';
                $open_sublists--;
            } elseif ($level == $current_level && $current_level > 2) {
                $toc_html .= '</li>';
            }

            if ($toc_numbered) {
                // Use <ol> automatic numbering
                $toc_html .= '<li><a href="#' . esc_attr($heading['anchor']) . '">' . esc_html($heading['text']) . '</a>';
            } else {
                // Manual numbering with <ul>
                $toc_html .= '<li>' . esc_html($heading['number']) . '. <a href="#' . esc_attr($heading['anchor']) . '">' . esc_html($heading['text']) . '</a>';
            }

            $current_level = $level;
        }

        // Close any remaining open sublists
        while ($open_sublists > 0) {
            $toc_html .= $toc_numbered ? '</ol>' : '</ul>';
            $toc_html .= '</li>';
            $open_sublists--;
        }

        if ($current_level >= 2) {
            $toc_html .= '</li>';
        }

        $toc_html .= $toc_numbered ? '</ol>' : '</ul>';
        $toc_html .= '</div></div>';

        return $toc_html;
    }

    /**
     * Process content and add anchor links to headings
     */
    public function process_content_add_anchors($content)
    {
        if (!nsg_is_nsg_page()) {
            return $content;
        }

        // Don't add anchors to archive pages
        if (nsg_is_nsg_archive_page())
        {
            return $content;
        }

        $post_id = nsg_get_seo_archive_page_id();
        $enable_toc = get_post_meta($post_id, 'nsg-enable_toc', true);

        if (!$enable_toc) {
            return $content;
        }

        $include_h3 = get_post_meta($post_id, 'nsg-toc_include_h3', true);
        
        if ($include_h3) {
            $pattern = '/<h([23])([^>]*)>(.*?)<\/h[23]>/i';
        } else {
            $pattern = '/<h2([^>]*)>(.*?)<\/h2>/i';
        }

        $content = preg_replace_callback($pattern, function($matches) {
            $level = isset($matches[3]) ? intval($matches[1]) : 2;
            $attributes = isset($matches[3]) ? $matches[2] : $matches[1];
            $text = isset($matches[3]) ? $matches[3] : $matches[2];
            
            // Process the heading text through NSG filters to handle Spintax and placeholders
            $params = array('context' => 'heading');
            $processed_text = nsg_get_seo_pages_replace_search_terms_and_locations($text, $params);
            
            $anchor = sanitize_title(strip_tags($processed_text));
            $anchor = 'toc-' . $anchor;

            // Check if ID already exists in attributes
            if (!preg_match('/id\s*=/', $attributes)) {
                $attributes .= ' id="' . esc_attr($anchor) . '"';
            }

            return '<h' . $level . $attributes . '>' . $text . '</h' . $level . '>';
        }, $content);

        return $content;
    }
}