<?php

if (!is_admin())
{
    // Pagebuilder Support
    if (!nsg_template_redirect_pagebuilder_support())
    {
        return;
    }
	
    add_action('template_redirect', 'nsg_start_output_buffer', -99);
    add_filter('the_content', 'nsg_add_adjacent_post_links', 99999999 + 1, 1);
    add_filter('the_content', 'nsg_get_seo_pages_replace_search_terms_and_locations', 99999999, 2);
    add_filter('the_content', 'nsg_add_toc_anchors_to_content', 99999999 + 2, 1);
    add_filter('the_content', 'nsg_maybe_add_toc_to_content_late', 99999999 + 3, 1);
}

function nsg_start_output_buffer()
{
    if (!nsg_is_nsg_page() || !nsg_is_html_output())
	{
		return;
	}
	ob_start('nsg_process_output');
}

function nsg_is_html_output()
{
	if (!function_exists('headers_list')) {
		return true;
	}

	$headers = headers_list();
	foreach ($headers as $header)
	{
		if (stripos($header, 'Content-Type:') !== false && stripos($header, 'text/html') !== false)
		{
			return true;
		}
	}
	return false;
}

function nsg_process_output($html)
{
	if (empty($html) || !nsg_is_nsg_page())
	{
		return $html;
	}

	$dom = new DOMDocument();
	libxml_use_internal_errors(true);
	$dom->loadHTML('<?xml encoding="UTF-8">' . $html, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
	libxml_clear_errors();

    // Process all DOM nodes (spintax, shortcodes, etc.)
	nsg_process_dom_node($dom, $dom);

    // Get final HTML
    $html = $dom->saveHTML();

    // Now post-process JSON-LD script tag with raw replacement
    $html = nsg_replace_ld_json_script($html);

    return $html;
}

function nsg_replace_ld_json_script($html)
{
    return preg_replace_callback(
        '#<script[^>]+type=["\']application/ld\+json["\'][^>]*>.*?</script>#is',
        function ($matches)
        {
            $original_script = $matches[0];

            // Extract the JSON string inside the script tag
            if (preg_match('#<script[^>]*>(.*?)</script>#is', $original_script, $inner)) {
                $json_raw = html_entity_decode($inner[1], ENT_QUOTES | ENT_HTML5, 'UTF-8');

                $decoded = json_decode($json_raw, true);
                if (is_array($decoded)) {
                    $processed = nsg_recursive_process_schema($decoded);
                    $json = json_encode($processed, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

                    // Return the new <script> block with raw UTF-8 JSON
                    return '<script type="application/ld+json">' . $json . '</script>';
                }
            }

            // If anything fails, return original
            return $original_script;
        },
        $html
    );
}

function nsg_safe_html($value)
{
    return htmlspecialchars(html_entity_decode($value, ENT_QUOTES | ENT_HTML5, 'UTF-8'), ENT_QUOTES | ENT_HTML5, 'UTF-8');
}

function nsg_process_dom_node(DOMDocument $dom, DOMNode $node, $context = '')
{
	foreach ($node->childNodes as $child)
	{
		$nsg_node = new NSG_Dom_Node($child);

		if ($nsg_node->is_text_node() && $child->nodeName !== 'script')
		{
			$node_value = trim($child->nodeValue);
            if (!empty($node_value) && (NSG_Dom_Node::contains_shortcodes($node_value) || NSG_Dom_Node::contains_spintax($node_value)))
            {
                // Give these the same context to match spintax choices
                if ($child->parentNode && ($child->parentNode->nodeName === 'title' || $child->parentNode->nodeName === 'h1'))
                {
                    $context = 'title';
                }
                else if ($child->parentNode && in_array($child->parentNode->nodeName, array('h2', 'h3')))
                {
                    $context = 'heading';
                }
                else if ($child->parentNode && stristr($child->parentNode->getAttribute('class'), 'breadcrumb'))
                {
                    $context = 'title';
                }
                $child->nodeValue = html_entity_decode(nsg_process_placeholders_and_spintax($node_value, $context));
			}
			continue;
		}

		if (!$nsg_node->is_element_node())
		{
			continue;
		}

        if ($child->nodeName === 'script' && $child->getAttribute('type') === 'application/ld+json')
		{
			$node_value = trim($child->nodeValue);

			if (!empty($node_value) && (NSG_Dom_Node::contains_shortcodes($node_value) || NSG_Dom_Node::contains_spintax($node_value)))
			{
				$json = json_decode($node_value, true);
				if (is_array($json))
				{
					$json             = nsg_recursive_process_schema($json);
					// $child->nodeValue = json_encode($json, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

                    $jsonLd = json_encode($json, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

                    // Create a new text node without escaping
                    $newScriptContent = $dom->createTextNode($jsonLd);

                    // Remove old content
                    while ($child->hasChildNodes()) {
                        $child->removeChild($child->firstChild);
                    }

                    // Append raw JSON
                    $child->appendChild($newScriptContent);
				}
			}
            
		}

		if (strtolower($child->nodeName) === 'meta')
		{
			$name     = strtolower($child->getAttribute('name'));
			$property = strtolower($child->getAttribute('property'));

			if (stristr($name, 'desc') || stristr($property, 'desc'))
			{
				$meta_context   = 'description';
				$content        = $child->getAttribute('content');
				if (!empty($content) && (NSG_Dom_Node::contains_shortcodes($content) || NSG_Dom_Node::contains_spintax($content)))
				{
                    $replaced = nsg_process_placeholders_and_spintax($content, $meta_context);
                    $child->setAttribute('content', html_entity_decode($replaced, ENT_QUOTES | ENT_HTML5, 'UTF-8'));
				}
			}
            else if (stristr($name, 'title') || stristr($property, 'title'))
			{
				$meta_context   = 'title';
				$content        = $child->getAttribute('content');
				if (!empty($content) && (NSG_Dom_Node::contains_shortcodes($content) || NSG_Dom_Node::contains_spintax($content)))
				{
                    $replaced = nsg_process_placeholders_and_spintax($content, $meta_context);
                    $child->setAttribute('content', html_entity_decode($replaced, ENT_QUOTES | ENT_HTML5, 'UTF-8'));
				}
			}
            else if ($name === 'robots')
            {
                if (nsg_should_noindex_archive_page())
                {
                    $child->setAttribute('content', 'noindex, follow');
                    $child->setAttribute('class', 'nsg-noindex-archive-page');
                }
            }
		}
        else if (strtolower($child->nodeName) === 'link')
        {
            $rel     = strtolower($child->getAttribute('rel'));
            if ($rel === 'canonical')
            {
                $child->setAttribute('href', nsg_get_canonical_url());
                $child->setAttribute('class', 'nsg-canonical');
            }
        }

		foreach ($child->attributes as $attr)
		{
			$original_value = trim($attr->nodeValue);
			$decoded_check = html_entity_decode($original_value, ENT_QUOTES | ENT_HTML5);

			if (!NSG_Dom_Node::contains_shortcodes($decoded_check) && !NSG_Dom_Node::contains_spintax($decoded_check))
			{
				continue;
			}

			if ($nsg_node->is_json($decoded_check))
			{
				$decoded = @json_decode($decoded_check, true);
				if (is_array($decoded))
				{
					$decoded = nsg_recursive_process_schema($decoded);
					$attr->nodeValue = htmlspecialchars(json_encode($decoded, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE), ENT_QUOTES | ENT_HTML5);
					continue;
				}
			}
			elseif ($nsg_node->is_serialized($decoded_check))
			{
				$unserialized = @unserialize($decoded_check);
				if (is_array($unserialized))
				{
					$unserialized = nsg_recursive_process_schema($unserialized);
					$attr->nodeValue = htmlspecialchars(serialize($unserialized), ENT_QUOTES | ENT_HTML5);
					continue;
				}
			}
			else
			{
				$replaced = nsg_process_placeholders_and_spintax($decoded_check, $context);
				$attr->nodeValue = htmlspecialchars($replaced, ENT_QUOTES | ENT_HTML5);
			}
		}

		nsg_process_dom_node($dom, $child, $context);
	}
}

function nsg_recursive_process_schema(array $data, $parentKey = '')
{
	foreach ($data as $key => &$value)
	{
		$currentKey = $parentKey ? $parentKey . '.' . $key : $key;

		if (is_array($value))
		{
			$value = nsg_recursive_process_schema($value, $currentKey);
		}
		elseif (is_string($value))
		{
            $value = nsg_process_placeholders_and_spintax($value, $currentKey);
		}
	}
	return $data;
}

function nsg_process_placeholders_and_spintax($content, $current_key = null)
{
    $params = array();
    if ($current_key !== null)
    {
        $params = array(
            'context'       => $current_key
        );
    }
	$content = nsg_get_seo_pages_replace_search_terms_and_locations($content, $params);
	return $content;
}

function nsg_resolve_spintax($text)
{
	return preg_replace_callback('/\{([^{}]+)\}/', function ($matches)
	{
		$options = explode('|', $matches[1]);
		return $options[array_rand($options)];
	}, $text);
}

class NSG_Dom_Node
{
	private $node;
	private static $shortcodes = array();

	public function __construct(DOMNode $node)
	{
		$this->node = $node;

        $nsg            = NSG_Seo_Generator::get_instance();
        $search_term_placeholder_single = $nsg->get_search_term_single_placeholder();
		$search_term_placeholder_plural = $nsg->get_search_term_plural_placeholder();

		$location_placeholder_single    = $nsg->get_location_single_placeholder();
		$location_placeholder_plural    = $nsg->get_location_plural_placeholder();

        self::$shortcodes[] = $search_term_placeholder_single;
        self::$shortcodes[] = $search_term_placeholder_plural;
        self::$shortcodes[] = $location_placeholder_single;
        self::$shortcodes[] = $location_placeholder_plural;

        if (get_option('nsg-enable_search_term_case_sensitivity', true))
        {
            // Lowercase: replace [search_term] with lowercase value
            $search_term_placeholder_single_lowercase   = strtolower($search_term_placeholder_single);
            $search_term_placeholder_plural_lowercase   = strtolower($search_term_placeholder_plural);

            // UCFirst: replace [Search_term] with ucfirst value
            $search_term_placeholder_single_ucfirst     = $nsg->nsg_get_ucfirst_placeholder($search_term_placeholder_single);
            $search_term_placeholder_plural_ucfirst     = $nsg->nsg_get_ucfirst_placeholder($search_term_placeholder_plural);

            self::$shortcodes[] = $search_term_placeholder_single_lowercase;
            self::$shortcodes[] = $search_term_placeholder_plural_lowercase;
            self::$shortcodes[] = $search_term_placeholder_single_ucfirst;
            self::$shortcodes[] = $search_term_placeholder_plural_ucfirst;

            // Lowercase: replace [location] with lowercase value
            $location_placeholder_single_lowercase   = strtolower($location_placeholder_single);
            $location_placeholder_plural_lowercase   = strtolower($location_placeholder_plural);

            // UCFirst: replace [Location] with ucfirst value
            $location_placeholder_single_ucfirst     = $nsg->nsg_get_ucfirst_placeholder($location_placeholder_single);
            $location_placeholder_plural_ucfirst     = $nsg->nsg_get_ucfirst_placeholder($location_placeholder_plural);

            self::$shortcodes[] = $location_placeholder_single_lowercase;
            self::$shortcodes[] = $location_placeholder_plural_lowercase;
            self::$shortcodes[] = $location_placeholder_single_ucfirst;
            self::$shortcodes[] = $location_placeholder_plural_ucfirst;
        }
	}

	public function is_text_node()
	{
		return $this->node->nodeType === XML_TEXT_NODE;
	}

	public function is_element_node()
	{
		return $this->node->nodeType === XML_ELEMENT_NODE;
	}

	public static function contains_shortcodes($string)
	{
		foreach (self::$shortcodes as $code)
		{
			if (strpos($string, $code) !== false)
			{
				return true;
			}
		}
		return false;
	}

	public static function contains_spintax($string)
	{
		// Ensure valid spintax pattern with a pipe separator inside curly braces
		return preg_match('/\{[^{}|]*\|[^{}]*\}/', $string);
	}

	public static function is_json($string)
	{
        // Step 1: Quick size check
        if (strlen($string) > 1048576) return false; // 1MB

        // Step 2: Quick format check (without decode)
        if (!preg_match('/^\s*[\{\[]/', trim($string))) return false;

        // Step 3: Safe decode only for small strings
        json_decode($string);
        return (json_last_error() === JSON_ERROR_NONE);
	}

	public static function is_serialized($string)
	{
		return ($string === serialize(false) || @unserialize($string) !== false);
	}
}

/**
 * Replace search term and location placeholders in text with actual search term and location for given SEO Page
 * 
 * @param string 	$text			    The text to replace the placeholders in with the actual values
 * @param bool|int	$post_id		    The Post ID to 
 * @param string	$search_term	    The Search term to replace. If omitted, find first search term for $post_id
 * @param string	$location		    The Location to replace. If omitted, find first location for $post_id
 * @param string	$search_term_plural	Plural form of Search term to replace. If omitted, the single search term will be used
 * @param string	$location_plural	Plural form of Location to replace. If omitted, the single location will be used
 * @param string	$the_slug	        Overrule the slug that is used on which the correct search terms and locations are searched in the lookup table
 * @param int       $spintax_offset     If Spintax is enabled, use this text combination defined by this offset value
 * 
 * @return string $text The text with the replacements made
 */
function nsg_get_seo_pages_replace_search_terms_and_locations($text, $params = array())
{
    $defaults = array(
		'the_post_id'        => false,
		'search_term'        => false,
		'location'           => false,
		'search_term_plural' => false,
		'location_plural'    => false,
		'the_slug'           => false,
		'spintax_offset'     => false,
		'context'            => false
	);

	extract(array_merge($defaults, $params));

    $post_id    = nsg_get_seo_archive_page_id(); 
    $context    = nsg_get_grouped_context($context);
        
	if (get_post_type($the_post_id) === 'nw_seo_page' && $the_post_id !== false && $post_id !== $the_post_id)
	{
		// We are being called by something like get_next_post_link / get_previous_post_link or something similar
		// Work with given $the_post_id, not from current page
		$post_id = $the_post_id;
	}

    if (get_post_type($post_id) === 'nw_seo_page')
	{
		$nsg    = NSG_Seo_Generator::get_instance();
        $slug   = $nsg->nsg_get_query_var_nsg_seo_page();

		if (empty($slug) && !empty($the_slug))
		{
			$slug = $the_slug;
		}

		// First find the search word and location in the url
		$nsg                		    = NSG_Seo_Generator::get_instance();

        if (!empty($slug))
		{
            if (get_option('nsg-enable_spintax', false) && NSG_Dom_Node::contains_spintax($text))
            {
                $spinner    = new NSG_Spintax($text);
                $text       = $spinner->get_combination($spintax_offset, $context);
            }
            
			$search_term_placeholder_single = $nsg->get_search_term_single_placeholder();
			$search_term_placeholder_plural	= $nsg->get_search_term_plural_placeholder();

			$location_placeholder_single    = $nsg->get_location_single_placeholder();
			$location_placeholder_plural	= $nsg->get_location_plural_placeholder();
			
			if ($search_term === false && $location === false)
			{
				$lookup_table					= nsg_get_search_terms_and_locations_lookup_table($post_id);

                if ($lookup_table !== false && isset($lookup_table[$slug]))
				{
					$search_term    	= $lookup_table[$slug][0];
					$location       	= $lookup_table[$slug][1];
					$search_term_plural = !empty($lookup_table[$slug][2]) ? $lookup_table[$slug][2] : $search_term;	// fallback to single
					$location_plural    = !empty($lookup_table[$slug][3]) ? $lookup_table[$slug][3] : $location;	// fallback to single
				}
			}

			// $search_term and $location have already been escaped with esc_html() in nsg_textarea_value_to_array()
            // Only replace placeholders if $search_term is not empty... otherwise just leave the placeholder
            if (!empty($search_term))
            {
                // First check if enabled
                if (get_option('nsg-enable_search_term_case_sensitivity', true))
                {
                    // Lowercase: replace [search_term] with lowercase value
                    $search_term_placeholder_single_lowercase   = strtolower($search_term_placeholder_single);
                    $search_term_placeholder_plural_lowercase   = strtolower($search_term_placeholder_plural);

                    $search_term_lowercase                      = strtolower($search_term);
                    $search_term_plural_lowercase               = strtolower($search_term_plural);

                    $text            = str_replace($search_term_placeholder_single_lowercase, $search_term_lowercase, $text);
                    $text            = str_replace($search_term_placeholder_plural_lowercase, $search_term_plural_lowercase, $text);

                    // UCFirst: replace [Search_term] with ucfirst value
                    $search_term_placeholder_single_ucfirst     = $nsg->nsg_get_ucfirst_placeholder($search_term_placeholder_single);
                    $search_term_placeholder_plural_ucfirst     = $nsg->nsg_get_ucfirst_placeholder($search_term_placeholder_plural);

                    $search_term_ucfirst                        = ucfirst($search_term);
                    $search_term_plural_ucfirst                 = ucfirst($search_term_plural);

                    $text            = str_replace($search_term_placeholder_single_ucfirst, $search_term_ucfirst, $text);
                    $text            = str_replace($search_term_placeholder_plural_ucfirst, $search_term_plural_ucfirst, $text);
                }
                else
                {
                    // str_ireplace case insensitive replace will replace all placeholders anyway they are written
                    if ($text !== null)
                    {
                        $text            = str_ireplace($search_term_placeholder_single, $search_term, $text);
                        $text            = str_ireplace($search_term_placeholder_plural, $search_term_plural, $text);
                    }
                }
            }

			// Only replace placeholders if $location is not empty... otherwise just leave the placeholder
            if (!empty($location))
			{
                // First check if enabled
                if (get_option('nsg-enable_location_case_sensitivity', true))
                {
                    // Lowercase: replace [location] with lowercase value
                    $location_placeholder_single_lowercase      = strtolower($location_placeholder_single);
                    $location_placeholder_plural_lowercase      = strtolower($location_placeholder_plural);

                    $location_lowercase                         = strtolower($location);
                    $location_plural_lowercase                  = strtolower($location_plural);

                    $text            = str_replace($location_placeholder_single_lowercase, $location_lowercase, $text);
                    $text            = str_replace($location_placeholder_plural_lowercase, $location_plural_lowercase, $text);

                    // UCFirst: replace [Location] with ucfirst value
                    if (!empty($location))
                    {
                        $location_placeholder_single_ucfirst        = $nsg->nsg_get_ucfirst_placeholder($location_placeholder_single);
                        $location_placeholder_plural_ucfirst        = $nsg->nsg_get_ucfirst_placeholder($location_placeholder_plural);

                        $location_ucfirst                           = ucfirst($location);
                        $location_plural_ucfirst                    = ucfirst($location_plural);

                        $text            = str_replace($location_placeholder_single_ucfirst, $location_ucfirst, $text);
                        $text            = str_replace($location_placeholder_plural_ucfirst, $location_plural_ucfirst, $text);
                    }
                }
                else
                {
                    // str_ireplace case insensitive replace will replace all placeholders anyway they are written
                    if ($text !== null)
                    {
                        $text            = str_ireplace($location_placeholder_single, $location, $text);
                        $text            = str_ireplace($location_placeholder_plural, $location_plural, $text);
                    }   
                }
			}
		}
		else
		{
			if ((int)get_query_var('nsg_seo_page_archive') && !isset($_GET["elementor-preview"]))
			{
                if ($context === 'title')
                {
                    $text = nsg_get_field('nsg-archive-page-title', $post_id, __('Archive', 'nsg_seo_generator'), false);
                }
                else if ($context === 'description')
                {
                    $text = nsg_get_archive_page_meta_description();
                }
                else
                {
                    $text = $text;
                }
			}
		}
	}

    return $text;
}

function nsg_get_archive_page_meta_description()
{
	$nsg               	= NSG_Seo_Generator::get_instance();
	$post_id			= get_the_ID();
	$meta_description	= '';
	$archive_page_title	= nsg_get_field('nsg-archive-page-title', $post_id, __('Archive', 'nsg_seo_generator'), false);
	$search_terms 		= $nsg->nsg_get_search_terms($post_id);
	$locations			= $nsg->nsg_get_locations($post_id);

	if (is_array($search_terms))
	{
		$search_terms = array_slice($search_terms, 0, 3);
		$search_terms = array_map('nsg_get_plural_value', $search_terms);
	}

	if (is_array($locations))
	{
		$locations = array_slice($locations, 0, 3);
		$locations = array_map('nsg_get_single_value', $locations);
	}

	if (is_array($search_terms) && is_array($locations))
	{
		$meta_description = sprintf(
			__("Archive for %s and more in %s and other locations - %s", 'nsg_seo_generator'),
			implode(", ", $search_terms),
			implode(", ", $locations),
			$archive_page_title
		);
	}

	return $meta_description;
}

/**
 * Add previous / index / next links to improve internal linking
 */
function nsg_add_adjacent_post_links($content)
{
	if (get_option('nsg-enable_adjacent_seo_pages_links', true))
	{
		$post_id 		= get_the_ID();

		if ($post_id > 0 && get_post_type($post_id) === 'nw_seo_page' && get_post_status($post_id) === 'publish')
		{
			$nsg    = NSG_Seo_Generator::get_instance();
            $slug   = $nsg->nsg_get_query_var_nsg_seo_page();
			
			if (!empty($slug))
			{
				$post			= get_post($post_id);
				$post_title		= $post->post_title;	// fetch unfiltered title
				$lookup_table	= nsg_get_search_terms_and_locations_lookup_table($post_id);

                if (isset($lookup_table[$slug]))
                {
                    $keys			= array_keys($lookup_table);
                    $index	    	= array_search($slug, $keys);
                    $seo_page_base 	= $lookup_table[$slug][5];

                    $content		.= "<div class='nsg-adjacent-links'>";
                    
                    // Previous link with correct title
                    if ($index - 1 >= 0)
                    {
                        $prev_slug 	= $keys[$index - 1];
                        $prev		= $lookup_table[$prev_slug];
                        
                        $prev_post_title    = $post_title;

                        if (get_option('nsg-enable_spintax', false))
                        {
                            $index_offset       = -1;
                            $spinner            = new NSG_Spintax($post_title);
                            $prev_post_title    = $spinner->get_combination($index_offset, 'title');
                        }

                        $params     = array(
                            'the_post_id'			=> $post_id,	
                            'search_term'			=> $prev[0],
                            'location'				=> $prev[1],
                            'context'				=> 'title',
                        );
                        
                        $prev_title	= nsg_get_seo_pages_replace_search_terms_and_locations($prev_post_title, $params);

                        $prev_url	= esc_url( trailingslashit(home_url($seo_page_base . '/' . strtolower(sanitize_title($prev_slug)))));
                        
                        $content	.= sprintf("<a class='nsg-prev-seo-page' href='%s'>%s</a>",
                                        $prev_url,
                                        $prev_title);
                    }

                    // Link to overview of all SEO Pages for this $seo_page_base
                    if (nsg_get_field('nsg-enable_archive_page', $post_id, true, false))
                    {
                        $content	.= sprintf("<a class='nsg-overview-seo-pages' href='%s'>%s</a>",
                                            esc_url(trailingslashit(home_url($seo_page_base))),
                                            nsg_get_field('nsg-overview-label', $post_id, __('Overview', 'nsg_seo_generator'), false));

                    }
                    
                    // Next link with correct title
                    if ($index + 1 < count($keys))
                    {
                        $next_slug 	= $keys[$index + 1];
                        $next		= $lookup_table[$next_slug];

                        $next_post_title    = $post_title;

                        if (get_option('nsg-enable_spintax', false))
                        {
                            $index_offset   = 1;
                            $spinner        = new NSG_Spintax($post_title);
                            $next_post_title = $spinner->get_combination($index_offset, 'title');
                        }

                        $params     = array(
                            'the_post_id'			=> $post_id,	
                            'search_term'			=> $next[0],
                            'location'				=> $next[1],
                            'context'				=> 'title',
                        );

                        $next_title	= nsg_get_seo_pages_replace_search_terms_and_locations($next_post_title, $params);

                        $next_url  	= esc_url( trailingslashit( home_url($seo_page_base . '/' . strtolower(sanitize_title($next_slug))) ) );

                        $content	.= sprintf("<a class='nsg-next-seo-page' href='%s'>%s</a><br>",
                                        $next_url,
                                        $next_title);
                    }

                    $content .= "</div>";
                }
			}
		}
	}

	return $content;
}

// Update NW SEO Page titles in admin nav menus and replace placeholders
add_filter('wp_setup_nav_menu_item', 'nsg_wp_setup_nav_menu_item', 10, 1);
function nsg_wp_setup_nav_menu_item($menu_item)
{
	if (is_admin() && $menu_item->object === 'nw_seo_page')
	{
		$post_id 		= $menu_item->object_id;

		$nsg = NSG_Seo_Generator::get_instance();

		// $menu_item->title and $menu_item->label are used in the box to select the SEO Page, on the accordion title and inside the menu item text field

		if ($nsg->has_any_placeholder($menu_item->title) || $menu_item->title === '')
		{
			$menu_item->title = nsg_get_field('nsg-archive-page-title', $post_id, __('Archive', 'nsg_seo_generator'), false);
		}

		if ($nsg->has_any_placeholder($menu_item->label) || $menu_item->label === '')
		{
			$menu_item->label = $menu_item->title;
		}
	}

	return $menu_item;
}

add_filter('the_title', 'nsg_maybe_hide_page_title', 10, 1);
function nsg_maybe_hide_page_title($title)
{
    // Only run on the main query and single pages

    if (!is_admin() && nsg_is_nsg_page() && in_the_loop() && is_main_query())
    {
        if ((bool)get_option('nsg-disable_page_title', false) === true)
        {
            // Return empty string to hide title
            return ''; 
        }
    }

    return $title;
}

/**
 * Add anchor IDs to headings for TOC functionality
 */
function nsg_add_toc_anchors_to_content($content)
{
    if (!nsg_is_nsg_page()) {
        return $content;
    }

    $toc = NSG_Table_Of_Contents::get_instance();
    return $toc->process_content_add_anchors($content);
}

/**
 * Add TOC to content after spintax has been resolved
 */
function nsg_maybe_add_toc_to_content_late($content)
{
    if (!nsg_is_nsg_page()) {
        return $content;
    }

    $toc = NSG_Table_Of_Contents::get_instance();
    return $toc->maybe_add_toc_to_content_late($content);
}
