<?php

/**
 * Fires once the field is shown
 * 
 * HTML fields have no value stored in DB. Show content by applying a filter
 * 
 * @param string  			$html  		HTML to show
 * @param NSG_Meta_Field 	$this 		The current NSG_Meta_Field object
 */
function nsg_acf_load_field_information($html, $field)
{
	// Display the first 20 links generated from the combiation of the Search Term and Locations lists.
	// so users can see the pages generated by this plugin
	if ($field->get_id() === 'nsg-explanation')
	{
		global $post;

		$nsg                		= NSG_Seo_Generator::get_instance();

		$post_id            		= (int)get_the_ID();
		$post_name          		= $post->post_name;
		$post_status				= $post->post_status;
		$urls               		= $nsg->nsg_get_seo_page_urls($post_name, $post_id, 20);
		$search_terms       		= $nsg->nsg_get_search_terms($post_id);
		$locations          		= $nsg->nsg_get_locations($post_id);
		$search_term_placeholder 	= $nsg->get_search_term_single_placeholder();
		$location_placeholder 		= $nsg->get_location_single_placeholder();

		if (is_array($search_terms) && is_array($locations))
		{
			$num_urls           = count($search_terms) * count($locations);
			$nums_urls_cap      = $num_urls > 20 ? __("The first 20 URLs are listed below.") : "";
		}

		if (is_array($urls) && count($urls) > 0)
		{
			$explanation 		= sprintf(
								__("A unique URL is generated for each combination of %s and %s.", 'nsg_seo_generator'),
								$search_term_placeholder,
								$location_placeholder
							);

			$num_generated		= sprintf(__('%s URLs generated.', 'nsg_seo_generator'), number_format($num_urls, 0, ",", "."));

			$html = "<div class='nsg-label'><p>{$explanation}</p>";
			
			$html .= "<p>{$num_generated} {$nums_urls_cap}</p></div><br>";

            $html .= "<div style='overflow-y: scroll; max-height: 100px;'>";

            $href = false;
            if ($post_status !== 'publish')
            {
                $href = get_home_url() . "?post_type=nw_seo_page&p={$post_id}&preview=true";
            }

            $index = 0;
            foreach ($urls as $url)
            {
                $style = 'display: block; padding: 8px;';
                if ($index % 2 == 0)
                {
                    $style .= ' background-color: #f0f0f0;';
                }
                if ($post_status === 'publish')
                {
                    $href = $url;
                }

                $url    = urldecode($url);

                $html .= "<a href='{$href}' target='_blank' style='{$style}'>{$url}</a>";

                $index++;
            }

			$html .= "</div>";
		}
		else
		{
			$no_urls = __('Add Search terms and Locations.', 'nsg_seo_generator');
			$html = "<div class='nsg-label'><p><span class='dashicons-before dashicons-no' style='color: #ff0000;'></span> {$no_urls}</p></div>";
		}
	}
	// Shows the sitemap link
	else if ($field->get_id() === 'nsg-sitemap')
	{
		$sitemap_url = nsg_get_sitemap_url();

		$html = __('Add this sitemap to the Google Search Console Sitemaps:', 'nsg_seo_generator');

        $sitemap_name = nsg_get_sitemap_name();
		
		$html .= sprintf(
			'<br><br><a href="%s" target="_blank">%s_index.xml</a>',
			$sitemap_url,
            $sitemap_name
		);			
	}
    // Shows the plugin documentation html
	else if ($field->get_id() === 'nsg-documentation')
	{
        $nsg_seo_generator_dir = nsg_get_plugin_dir();

        ob_start();
            require_once "{$nsg_seo_generator_dir}assets/documentation.php";
		    $html = ob_get_contents();
        ob_end_clean();
	}
	// Shows the actual URL structure created by SEO Page Base and Slug fields
	else if ($field->get_id() === 'nsg-url-structure')
	{
		global $post;

		$nsg = NSG_Seo_Generator::get_instance();
		$site_url			= get_home_url();
		$seo_page_base		= urldecode('/' . $post->post_name);
		$slug_placeholder	= $nsg->nsg_get_slug_placeholder($post->ID);

        

		$html = "<p class='nsg-url-structure'><span class='nsg-url-structure-site-url'>{$site_url}</span><label class='nsg-url-structure-seo-page-base' for='nsg-seo-page-base'>{$seo_page_base}</label>/<label class='nsg-url-structure-base-slug' for='nsg-slug-placeholder'>{$slug_placeholder}</span></p>";
	}
	else if ($field->get_id() === 'nsg-placeholders-warning')
	{
		$nsg        = NSG_Seo_Generator::get_instance();
		$pages      = $nsg->nsg_get_seo_pages('any');

		if ($pages)
		{
			$html = sprintf("<span class='error'>%s</span>",
				__('When you change these placeholders, do not forget to update the placeholders on your SEO Pages', 'nsg_seo_generator')
			);
		}
	}
    else if ($field->get_id() === 'nsg-shortcodes-information')
	{
		$nsg            = NSG_Seo_Generator::get_instance();
		$pages          = $nsg->nsg_get_seo_pages('any');

        $warning        = __("Please note, values inside the shortcodes are case sensitive and should match exactly with your given search terms and locations!", 'nsg_seo_generator');
        $examples       = __("Shortcode expamples:", 'nsg_seo_generator');
        
        $label_1        = __("Show content only for a single search term:", 'nsg_seo_generator');
        $label_2        = __("Show content only for multiple search terms:", 'nsg_seo_generator');
        $label_3        = __("Show content only for a single location:", 'nsg_seo_generator');
        $label_4        = __("Show content only for multiple locations:", 'nsg_seo_generator');
        $label_5        = __("Show content only for a single search term and location:", 'nsg_seo_generator');
        $label_6        = __("Show content only for multiple search terms and locations:", 'nsg_seo_generator');

		if ($pages)
		{
            $html = sprintf("<span class='error'>%s</span><br><br>%s", $warning, $examples);

			$html .= sprintf('<br><br><strong>%s</strong><br><br>
                <code style="display: block;">[nsg_seo_page search_term="Search Term 1"]<br>Content only shown for Search Term 1<br>[/nsg_seo_page]</code>',
                $label_1
			);

            $html .= sprintf('<br><br><strong>%s</strong><br><br>
                <code style="display: block;">[nsg_seo_page search_term="Search Term 1 | Search Term 2"]<br>Content only shown for Search Term 1 OR  Search Term 2<br>[/nsg_seo_page]</code>',
                $label_2
			);

            $html .= sprintf('<br><br><strong>%s</strong><br><br>
                <code style="display: block;">[nsg_seo_page location="Location 1"]<br>Content only shown for Location 1<br>[/nsg_seo_page]</code>',
                $label_3
			);

            $html .= sprintf('<br><br><strong>%s</strong><br><br>
                <code style="display: block;">[nsg_seo_page location="Location 1 | Location 2"]<br>Content only shown for Location 1 OR Location 2<br>[/nsg_seo_page]</code>',
                $label_4
			);

            $html .= sprintf('<br><br><strong>%s</strong><br><br>
                <code style="display: block;">[nsg_seo_page search_term="Search Term 1" location="Location 1"]<br>Content only shown for Search Term 1 AND Location 1<br>[/nsg_seo_page]</code>',
                $label_5
			);

            $html .= sprintf('<br><br><strong>%s</strong><br><br>
                <code style="display: block;">[nsg_seo_page search_term="Search Term 1 | Search Term 2" location="Location 1 | Location 2"]<br>Content only shown for Search Term 1 OR Search Term 2 AND Location 1 OR Location 2<br>[/nsg_seo_page]</code>',
                $label_6
			);
		}
	}

    return $html;
}
add_filter('nsg-show-field-html', 'nsg_acf_load_field_information', 10, 2);

/**
 * Sanitizes a string from user input or from the database.
 *
 * @param string $value String to sanitize.
 * 
 * @return string Sanitized string.
 */
function nsg_meta_block_field_sanitize_value($value, $field)
{
	if ($field->get_id() === 'nsg-slug-placeholder')
	{
		$nsg						= NSG_Seo_Generator::get_instance();

		// Only work with the Single values, not the Plural values!
		$search_term_placeholder 	= $nsg->get_search_term_single_placeholder();
		$location_placeholder 		= $nsg->get_location_single_placeholder();
		
		$has_both_placeholders		= $nsg->has_any_placeholder($value, true);

		if ($has_both_placeholders)
		{
			// Relocation the placeholders to something that is safe for urls and will not be removed by sanitize_title
			$value = str_replace($search_term_placeholder, '___aa1234aa___', $value);
			$value = str_replace($location_placeholder, '___bb1234bb___', $value);
		}

		// Use the custom sanitizer instead of sanitize_title()
        // But make sure non-Latin characters etc are preserved (like Hebrew)
        $value = nsg_sanitize_slug($value);

		if ($has_both_placeholders)
		{
			// And replace placeholders back in
			$value = str_replace('___aa1234aa___', $search_term_placeholder, $value);
			$value = str_replace('___bb1234bb___', $location_placeholder, $value);
		}
		else
		{
			if (!nsg_is_empty($value))
			{
				$value .= "-" . $field->args['default'];
			}
		}
	}
	else if ($field->get_id() === 'nsg-search-terms')
	{
		$nsg    = NSG_Seo_Generator::get_instance();
		$value	= nsg_limit_max_lines($value, $nsg->nsg_get_max_search_terms());
		$value	= strip_tags($value);
	}
	else if ($field->get_id() === 'nsg-locations')
	{
		$nsg    = NSG_Seo_Generator::get_instance();
		$value	= nsg_limit_max_lines($value, $nsg->nsg_get_max_locations());
		$value	= strip_tags($value);
	}
	

    return $value;
}
add_filter('nsg-meta-block-field-sanitize-value', 'nsg_meta_block_field_sanitize_value', 10, 2);

/**
 * Custom slug sanitizer that preserves non-Latin characters and diacritics.
 *
 * @param string $slug The slug to sanitize.
 * @return string The sanitized slug.
 */
function nsg_sanitize_slug($slug)
{
    // Remove accents but KEEP diacritics and non-Latin characters
    $slug = remove_accents($slug);

    // Preserve letters, numbers, diacritics (accents), dashes, underscores, and placeholders
    $slug = preg_replace('/[^\p{L}\p{N}\p{M}\-_\[\]]+/u', '-', $slug);

    // Remove duplicate dashes
    $slug = preg_replace('/-+/', '-', $slug);

    // Trim dashes
    return trim($slug, '-');
}

/**
 * Fires once the field 'nsg-seo-page-base' has been saved
 * 
 * Update Post Name (Slug) when SEO Page base is saved. $value has already been sanitized
 * 
 * @param string  			$value  	Field value
 * @param int				$post->ID  	Post ID
 */
function nsg_update_post_name($value, $post_id)
{	
	// unhook this function so it doesn't loop infinitely
	remove_action('nsg-save-field-nsg-seo-page-base', 'nsg_update_post_name');

	if ((int)$post_id > 0)
	{
		$post	= get_post($post_id);

		

		if (empty($value))
		{
			// If we do not have a SEO Page base yet, use Post Title instead
			$value 	= sanitize_title($post->post_title);
		}

		// Make sure the SEO Page base adheres to the rule that $post->post_name must be unique!
		$value = wp_unique_post_slug($value, $post_id, $post->post_status, $post->post_type, $post->post_parent);

		if ($post->post_name !== $value)
		{
			// And update the post_name field
			$args	= array(
				'ID'		=> $post_id,
				'post_name'	=> $value
			);
		
			wp_update_post($args);
		}
	}

	// Workaround to flush the rewrite rules:
	update_option( 'nsg-flush-rewrite-rules', 1 );
}
add_action('nsg-save-field-nsg-seo-page-base', 'nsg_update_post_name', 10, 2);

/**
 * Fires once an existing post has been updated.
 *
 * Flush rewrite rules once a SEO Page is saved to make sure the new dynamic urls can be found
 *
 * @param int     $post_ID      Post ID.
 * @param WP_Post $post_after   Post object following the update.
 * @param WP_Post $post_before  Post object before the update.
 */
function nsg_post_updated($post_ID, $post_after, $post_before)
{
	remove_action("post_updated", "nsg_post_updated");

	if ($post_after->post_type === 'nw_seo_page' && $post_before->post_name !== $post_after->post_name)
	{
		// TLDR; You can't flush_rewrite_rules(); on save_post action hook.
		// https://wordpress.stackexchange.com/questions/182798/flush-rewrite-rules-on-save-post-does-not-work-on-first-post-save

		// Workaround to flush the rewrite rules:
		update_option( 'nsg-flush-rewrite-rules', 1 );
	}
}
add_action("post_updated", "nsg_post_updated", 10, 3);

/**
 * Fires once the field value has been retrieved from the database
 * 
 * nsg-seo-page-base is just a dummy field, use the post_name instead!
 * 
 * @param string  			$value  	Field value
 * @param int				$post->ID  	Post ID
 */
function nsg_get_value_seo_page_base($value, $post_id)
{
	$post = get_post($post_id);
    return urldecode($post->post_name);
}
add_filter('nsg-get-value-nsg-seo-page-base', 'nsg_get_value_seo_page_base', 10, 2);

function nsg_after_save($option_name, $old_value, $new_value)
{
    if ($option_name === 'nsg-sitemap_name')
    {
        flush_rewrite_rules();

        if ($old_value !== $new_value)
        {
            nsg_cron_job_handler('updated-sitemap-name');
        }
        
        // Workaround to flush the rewrite rules:
		update_option( 'nsg-flush-rewrite-rules', 1 );
    }
}
add_action('update_option', 'nsg_after_save', 10, 3);

// Make sure the sitemap name will never become empty!
add_filter('pre_update_option', 'nsg_pre_update_option_sitemap_name', 10, 3);
function nsg_pre_update_option_sitemap_name($value, $option, $old_value) 
{
    // Check if the new value is empty and replace it with a default value
    if ($option === 'nsg-sitemap_name')
    {
        // Sitemap name is part of the url!
        $value = strtolower(sanitize_title($value));

        if (empty($value))
        {
            $value = 'seo_generator_sitemap';
        }

        flush_rewrite_rules();

        // Workaround to flush the rewrite rules:
		update_option( 'nsg-flush-rewrite-rules', 1 );
    }
    return $value;
}
