<?php

function nsg_regenerate_search_terms_and_locations_lookup_tables($seo_page_id = false)
{
	$nsg            = NSG_Seo_Generator::get_instance();
    $seo_pages      = $nsg->nsg_get_seo_pages();
    $lookup_table   = false;
    // Regenerate lookup tables
    if (is_array($seo_pages) && count($seo_pages) > 0)
    {
        foreach ($seo_pages as $seo_page)
        {
            if ($seo_page_id !== false && $seo_page_id !== $seo_page->ID)
            {
                // We only want to regenerate the lookup table for $seo_page_id, not the other SEO Pages
                continue;
            }
            $lookup_table = $nsg->nsg_store_search_word_and_location_for_slugs($seo_page->ID);
        }
    }

    if ($seo_page_id !== false && $lookup_table !== false)
    {
        return $lookup_table;
    }
}

function nsg_get_search_terms_and_locations_lookup_tables()
{
	$meta_key = 'nsg-search-word-and-location-for-slugs';

    $posts = get_posts(
        array(
            'post_type' 		=> 'nw_seo_page',
            'meta_key' 			=> $meta_key,
            'posts_per_page' 	=> -1,
			'post_status'		=> 'publish'
        )
    );

    $meta_values = array();
    foreach( $posts as $post)
	{
		$post_meta = get_post_meta($post->ID, $meta_key, true);

		$meta_values[] = $post_meta;
    }

    return $meta_values;
}

function nsg_get_search_terms_and_locations_lookup_table($post_id)
{
	$lookup_table = array();
	if ($post_id > 0)
	{
		$lookup_table = get_post_meta($post_id, 'nsg-search-word-and-location-for-slugs', true);
	}
    
    return $lookup_table;
}

/**
 * Returns a value from the post_meta table with the search term and location placeholders already replaced
 * 
 * @param string 	$key     		The meta key to retrieve
 * @param int    	$post_id 		Post ID.
 * @param string	$default_value 	The default value to return if value is empty / not found
 * @param boolean	$do_replace		Replace the placeholders or not
 * 
 * @return string	$value		The value with the search term and location placeholders already replaced
 */
function nsg_get_field($key, $post_id, $default_value, $do_replace)
{
	$post_id	    = nsg_get_valid_post_id($post_id);
	$meta_values 	= get_post_meta($post_id, $key);
    
    if (empty($meta_values))
    {
        // Key does not exist
        $value = $default_value;
    }
    else
    {
        // Key exists, even if the value is '', 0 or false
        $value = $meta_values[0];
    }

	if ($do_replace)
	{
		$value		= nsg_get_seo_pages_replace_search_terms_and_locations($value);
	}

    if ($value === '0')
    {
        $value = false;
    }
    else if ($value === '1')
    {
        $value = true;
    }

	return $value;
}

/**
 *  This function will return a valid post_id based on the current screen / parameter
 * 
 *  @param	$post_id (mixed)
 * 
 *  @return	$post_id (mixed)
 */
function nsg_get_valid_post_id($post_id = 0)
{
	// if not $post_id, load queried object
	if( !$post_id )
	{
		// try for global post (needed for setup_postdata)
		$post_id = (int) get_the_ID();

		// try for current screen
		if( !$post_id )
		{
			$post_id = get_queried_object();
		}
	}

	if( is_object($post_id) )
	{
		// post
		if( isset($post_id->post_type, $post_id->ID) )
		{
			$post_id = $post_id->ID;

		// user
		} elseif( isset($post_id->roles, $post_id->ID) ) {

			$post_id = 'user_' . $post_id->ID;

		// term
		} elseif( isset($post_id->taxonomy, $post_id->term_id) ) {

			$post_id = 'term_' . $post_id->term_id;

		// comment
		} elseif( isset($post_id->comment_ID) ) {

			$post_id = 'comment_' . $post_id->comment_ID;

		// default
		} else {

			$post_id = 0;
		}
	}

	// And add NSG logic:
	if (is_admin())
	{
		global $pagenow;

		if ($pagenow === 'post.php' && isset($_GET['action']) && $_GET['action'] === 'edit')
		{
			$post_id = (int)$_GET['post'];
		}

		if (!$post_id && !empty($_POST)
			&& isset($_POST['action']) && $_POST['action'] === 'editpost'
			&& isset($_POST['post_type']) && $_POST['post_type'] === 'nw_seo_page'
		)
		{
			$post_id = (int)$_POST['post_ID'];
		}
	}

	// return
	return $post_id;
}

// Return the single value if the string contains single | plural in 1 line
function nsg_get_single_value($value)
{
	if (strstr($value, "|"))
	{
		$values = explode("|", $value);

		if (is_array($values) && count($values) === 2)
		{
			$value = trim($values[0]);
		}
	}

	return $value;
}

// Return the plural value if the string contains "single | plural" in 1 line
function nsg_get_plural_value($value)
{
	if (strstr($value, "|"))
	{
		$values = explode("|", $value);

		if (is_array($values) && count($values) === 2)
		{
			$value = trim($values[1]);
		}
	}

	return $value;
}

/**
 * Flush rewrite rules when SEO page is saved
 * 
 * Workaround to automatically flush rewrite rules when needed
 */
function nsg_late_init_flush_rewrite_rules()
{
    if (!$option = get_option( 'nsg-flush-rewrite-rules'))
    {
        return false;
    }

    if ( $option == 1 )
    {
        flush_rewrite_rules();
        update_option( 'nsg-flush-rewrite-rules', 0 );
    }

    return true;
}
add_action('init', 'nsg_late_init_flush_rewrite_rules', 999999);

/**
 * Limits max number of allowed lines by stripping the lines over the limit
 * 
 * @param string	$string		The multiline string to trim
 * @param int		$max_lines	The max number of lines allowed
 * 
 * @return string	The multiline string maximum set lines
 */
function nsg_limit_max_lines($string, $max_lines)
{
	
    if (!empty($string))
	{
		$lines  = nsg_textarea_value_to_array($string);
		$lines  = array_slice($lines, 0, $max_lines);
		$string = implode(PHP_EOL, $lines);
	}
    return $string;
}

/**
 * Explodes textarea lines to php arrray
 * 
 * @param string 	$textarea_value The multiline string
 * 
 * @return array	$array			The multiline string where each line is added to an array
 */
function nsg_textarea_value_to_array($textarea_value)
{
	$array = false;
    if (!empty($textarea_value))
    {
        $array                 = array_values(array_filter(explode(PHP_EOL, $textarea_value))); // transform textarea lines to array
        $array                 = array_map('trim', $array);                                     // trim spaces from each line
        $array                 = array_unique($array);                                          // Remove duplicates
        $array                 = array_map('esc_html', $array);                                 // Escaping: Securing Output
    }

    return $array;
}

/**
 * Recursive find and replace
 * 
 * @param string		$find
 * @param string 		$replace
 * @param array|string 	$array
 * 
 * @return array
 */
function nsg_recursive_array_replace ($find, $replace, $array)
{
    if (! is_array($array)) {
        return str_replace($find, $replace, $array);
    }

    $newArray = array();

    foreach ($array as $key => $value) {
        $newArray[$key] = nsg_recursive_array_replace($find, $replace, $value);
    }

    return $newArray;
}

/*
 * nsg_is_empty
 *
 * Returns true if the value provided is considered "empty". Allows numbers such as 0.
 *
 * @param	mixed $var The value to check.
 * 
 * @return	bool
 */
function nsg_is_empty( $var )
{
	return ( !$var && !is_numeric($var) );
}

/**
 * Filters the robots.txt output.
 * 
 * Adds Sitemap to robots.txt
 *
 * @param string $output The robots.txt output.
 * @param bool   $public Whether the site is considered "public".
 * 
 * @return string Robots.txt content with sitemap added
 */
function nsg_add_sitemap_to_robots($output, $public)
{
    $output .= "\r\nSitemap: " . nsg_get_sitemap_url();

	return $output;
}
add_filter('robots_txt', 'nsg_add_sitemap_to_robots', 10, 2);

/**
 * Array helper function to easily insert a value in an associative array after given needle
 * 
 * @see https://stackoverflow.com/questions/21335852/move-an-associative-array-key-within-an-array
 * 
 * @param string[] 	$array  		Associative array
 * @param string 	$insert_after 	Key (needle) to inser new key / value pair after
 * @param string 	$key 			New key to insert
 * @param string 	$value 			New value for given key
 * 
 * @return array Array with value inserted after given position
 */
function nsg_array_insert_after($array, $insert_after, $key, $new)
{
    $pos = (int) array_search($insert_after, array_keys($array)) + 1;
    return array_merge(
        array_slice($array, 0, $pos),
        array($key => $new),
        array_slice($array, $pos)
    );
}

/**
 * Checks if given option exists in options table
 * 
 * Will return true if the option exists, but the value is empty, 0, null, false
 * 
 * @global $wpdb 
 * 
 * @param string $option The option to check
 * 
 * @return bool Whether the option exists or not
 */
function nsg_option_exists($option)
{
    global $wpdb;

	$table	= $wpdb->prefix . "options";
	$sql	= $wpdb->prepare("SELECT * FROM {$table} WHERE option_name LIKE %s", $option);
	$result = (bool)$wpdb->query($sql);

	return $result;
}

add_filter( "previous_post_link", "nsg_get_adjacent_post_link", 10, 5);
add_filter( "next_post_link", "nsg_get_adjacent_post_link", 10, 5);
function nsg_get_adjacent_post_link($output, $format, $link, $post, $adjacent)
{
	if (!empty($output) && get_post_type($post) === 'nw_seo_page')
	{
		libxml_use_internal_errors(true);	// disable warnings about imperfect HTML markup
		$html = new DOMDocument();
		$html->loadHTML($output);

		foreach($html->getElementsByTagName('a') as $link)
		{
			$nsg 		= NSG_Seo_Generator::get_instance();

			// Fetch new link
			$new_link 	= $nsg->nsg_get_seo_page_urls($post->post_name, $post->ID, 1);

			// Replace first search term and first location in the title of given post
			$search_terms    = $nsg->nsg_get_search_terms($post->ID);

			if (is_array($search_terms) && count($search_terms) > 0)
			{
				$search_term_single = current($search_terms);
			}

			$locations      = $nsg->nsg_get_locations($post->ID);

			if (is_array($locations) && count($locations) > 0)
			{
				$location_single = current($locations);
			}
			
            $params     = array(
                'the_post_id'			=> $post->ID,	
                'search_term'			=> $search_term_single,
                'location'				=> $location_single,
                'context'				=> 'title',
            );
            
			$title	= nsg_get_seo_pages_replace_search_terms_and_locations($post->post_title, $params);

			// Change the <a href>-tag:
			$link->setAttribute('href', $new_link[0]);			
			$link->nodeValue = $title;
		}

		$output = $html->saveHtml();
	}

	 return $output;
}

/**
 * Formats the given timestamp to the needed format.
 *
 * @param string $date 		The date to use for the formatting.
 * @param string $offset    Change date with this offset, example: "+30 minutes"
 *
 * @return string The formatted date.
 */
function nsg_format_timestamp($date, $offset = false)
{
	$date = new DateTime($date);

	if ($offset !== false)
	{
		$date->modify($offset); //or whatever value you want
	}
	return $date->format('c');
}

/**
 * Get the Canonical URL for the SEO Generator archive page and all generated URLs
 */
function nsg_get_canonical_url ($canonical_url = false, $post = false)
{
    $post_id = (int)nsg_get_valid_post_id();

    $nsg    = NSG_Seo_Generator::get_instance();
    $slug   = $nsg->nsg_get_query_var_nsg_seo_page();

	if ((int)get_query_var('nsg_seo_page_archive') && nsg_is_nsg_archive_page())
    {
        
		// Fix canonical page for (paginated) archive pages
        global $wp_query;
        if (get_post_type($post_id) !== 'nw_seo_page' && $wp_query->original_seo_page_id !== null)
        {
            $canonical_url  = trailingslashit(get_permalink($wp_query->original_seo_page_id));
        }
        else
        {
            $canonical_url  = trailingslashit(get_permalink($post_id));
        }

        $current_page   = max(1, get_query_var('paged'));
        if($current_page > 1)
        {
            $canonical_url  = $canonical_url . 'page/' . $current_page . '/';
        }
    }
	else if ($post_id > 0 && get_post_type($post_id) === 'nw_seo_page' && get_post_status($post_id) === 'publish' && $slug)
	{
		remove_filter('post_type_link', 'nsg_remove_slug', 10, 3);

		$permalink      = trailingslashit(get_permalink($post_id));
		$canonical_url  = $permalink . $slug;

		if (strstr($canonical_url, '/nw_seo_page/'))
		{
			$canonical_url = str_replace( '/nw_seo_page/', '/', $canonical_url );
		}

		$canonical_url 	= trailingslashit($canonical_url);
	}

	return $canonical_url;
}

// Remove <link rel='shortlink' href='http://localhost/lite/klanten/connectr/seogenerator/?p={$post_id}' /> from head
add_filter('get_shortlink', 'nsg_get_shortlink', 10, 2);
function nsg_get_shortlink($shortlink, $post_id)
{
	if ((int)$post_id === 0)
	{
		$post_id = nsg_get_valid_post_id();
	}
	
	if (($post_id > 0 && get_post_type($post_id) === 'nw_seo_page') || (int)get_query_var('nsg_seo_page_archive'))
	{
		$shortlink = false;
		
	}
	
	return $shortlink;
}

// Remove:
// <link rel="alternate" type="application/json+oembed"
// <link rel="alternate" type="text/xml+oembed
add_filter( 'oembed_discovery_links', 'nsg_oembed_discovery_links', 10, 1);
function nsg_oembed_discovery_links($output)
{
	$post_id = nsg_get_valid_post_id();

	if (($post_id > 0 && get_post_type($post_id) === 'nw_seo_page') || (int)get_query_var('nsg_seo_page_archive'))
	{
		$output = false;
	}

	return $output;
}

add_action('init', 'nsg_cleanup_head');
function nsg_cleanup_head()
{
	remove_action( 'wp_head', 'feed_links_extra', 3 );				// Display the links to the extra feeds such as category feeds
	remove_action( 'wp_head', 'feed_links', 2 ); 					// Display the links to the general feeds: Post and Comment Feed
	remove_action( 'wp_head', 'index_rel_link' ); 					// index link
	remove_action( 'wp_head', 'parent_post_rel_link', 10, 0 ); 		// prev link
	remove_action( 'wp_head', 'start_post_rel_link', 10, 0 ); 		// start link
	remove_action( 'wp_head', 'adjacent_posts_rel_link', 10, 0 ); 	// Display relational links for the posts adjacent to the current post.
}

/**
 * Checks if we are currently on the settings page
 * 
 * @global $pagenow
 * 
 * @return Wheter or not we are on the settings page
 */
function nsg_is_settings_page()
{
	global $pagenow;

	return $pagenow === 'edit.php' && isset($_GET['post_type']) && $_GET['post_type'] === 'nw_seo_page' && isset($_GET['page']) && $_GET['page'] === 'nsg-settings';
}

/**
 * Checks if we are currently on the documentation page
 * 
 * @global $pagenow
 * 
 * @return Wheter or not we are on the documentation page
 */
function nsg_is_documentation_page()
{
	global $pagenow;

	return $pagenow === 'edit.php' && isset($_GET['post_type']) && $_GET['post_type'] === 'nw_seo_page' && isset($_GET['page']) && $_GET['page'] === 'nsg-docs';
}

/**
 * Filters the list of action links displayed for a specific plugin in the Plugins list table.
 *
 * The dynamic portion of the hook name, `$plugin_file`, refers to the path
 * to the plugin file, relative to the plugins directory.
 *
 * @param string[] $actions     An array of plugin action links. By default this can include 'activate',
 *                              'deactivate', and 'delete'. With Multisite active this can also include
 *                              'network_active' and 'network_only' items.
 */
function nsg_plugin_settings_link($links)
{ 
	$settings_page_url	= menu_page_url('nsg-settings', false);
	$settings_link		= sprintf('<a href="%s">%s</a>',
		esc_attr($settings_page_url),
		esc_html(__('Settings', 'nsg-seo-generator'))
	);

	array_unshift($links, $settings_link); 

	return $links; 
}
$nsg_seo_generator_basename = nsg_get_plugin_basename();
add_filter("plugin_action_links_{$nsg_seo_generator_basename}", 'nsg_plugin_settings_link');

// Check if we are on an archive or single nw_seo_page page in the frontend
function nsg_is_nsg_page()
{
	$is_nsg_page = false;

    $post_id = nsg_get_valid_post_id();

    if ((int)$post_id > 0 && get_post_type($post_id) !== 'nw_seo_page')
    {
        $post_id = nsg_get_seo_archive_page_id();
    }

    if (!is_admin() && ($post_id > 0 && get_post_type($post_id) === 'nw_seo_page') || (int)get_query_var('nsg_seo_page_archive'))
	{
		$is_nsg_page = true;
	}

	return $is_nsg_page;
}

// Check if we are on the archive in the frontend
function nsg_is_nsg_archive_page()
{
	$is_nsg_archive_page = false;

    $seo_page_id    = nsg_get_seo_archive_page_id();

	if (!is_admin() && ($seo_page_id > 0 && get_post_type($seo_page_id) === 'nw_seo_page') && (int)get_query_var('nsg_seo_page_archive'))
	{
		$is_nsg_archive_page = true;
	}

	return $is_nsg_archive_page;
}

/**
 * Returns the actual SEO Generator archive post ID.
 *
 * If a custom replacement page is active and this is that page,
 * retrieves the original archive post ID from the query var.
 * Otherwise returns get_the_ID().
 * 
 * @return int
 */
function nsg_get_seo_archive_page_id()
{
    // If no custom Archive page is set, the SEO Page itself is also the Archive page
    $seo_archive_page_id = get_the_ID();

    if (get_post_type($seo_archive_page_id) !== 'nw_seo_page')
    {
        global $wp_query;
        if ($wp_query->original_seo_page_id !== null)
        {
            $seo_archive_page_id = $wp_query->original_seo_page_id;
        }
        else
        {
            $seo_archive_page_id = false;
        }
    }

    return $seo_archive_page_id;
}

// Locate the template to be loaded from either the plugin or overridden and load from theme
function nsg_get_template($post_id)
{
    // Default template used by Wordpress: get_page_template_slug() returns empty string and falls back to single.php (or whatever default template in theme falls back on)

    // If a custom template is selected, make sure we include this and not fall back to page.php to show a regular page
    $custom_template = get_page_template_slug($post_id);
    
    if (strstr($custom_template, 'nsg-seo-generator/templates/'))
    {
        // Give theme authors option to copy templates to the theme directory and load those templates instead
        $path_parts 		= pathinfo($custom_template);
        $template_filename	= $path_parts['basename'];
        $template 			= locate_template($template_filename);
        
        // No template was found in the theme, load template from plugin
        if ($template === '')
        {
            $template = WP_PLUGIN_DIR . '/' . $custom_template;
        }
    }
    else
    {
        $template = locate_template(array('page.php', 'single.php', 'singular.php', 'index.php'));
    }

    return $template;
}

function nsg_get_sitemap_name()
{
    $sitemap_name = get_option('nsg-sitemap_name', 'seo_generator_sitemap');
    
    if (empty($sitemap_name))
    {
        $sitemap_name =  'seo_generator_sitemap';
    }
    return $sitemap_name;
}

function nsg_get_grouped_context($broad_context)
{
    $context = $broad_context;
    if ($broad_context === false)
    {
        $context = 'content';
    }
    else if (stristr($broad_context, 'name'))
    {
        $context = 'title';
    }
    else if (stristr($broad_context, 'desc'))
    {
        $context = 'description';
    }
    else if (stristr($broad_context, 'heading'))
    {
        $context = 'heading';
    }

    return $context;
}

// Create a demo page to showcase the plugin functionalities
function nsg_create_seo_gen_example_page()
{
    $post_content = 
'<h2>Professional [search_term] in [location]</h2>
Find a [search_term] in [location]. Lorem ipsum dolor sit amet, consectetur adipiscing elit. Aliquam varius nec ex fermentum vehicula. Cras sodales est nec gravida pretium. Integer libero arcu, pulvinar vitae tempus eget, convallis ut nulla.
<h2>Meet our [search_terms] at our salon</h2>
Our [search_terms] are the best [search_term] in [location]. Meet our team now!

<h2>Get a haircut from one of our [search_terms] in [location]</h2>
Donec ac tortor vitae purus cursus tempor. Duis non hendrerit augue, ut consectetur erat. Suspendisse [search_term], magna at lobortis pharetra, massa orci lacinia massa, non tempus turpis nisl nec libero. Sed sed enim lorem. Cras et orci sapien.
<p style="text-align: center;"><a class="button button-primary" href="https://seogenerator.io/">Book now</a></p>

<h2>Wide variety of [search_term] services in [location]</h2>
Suspendisse fermentum lacus vitae tristique consectetur. Nunc luctus volutpat arcu, eu tempor odio pulvinar in. Sed urna nulla, finibus ut lorem [location], consectetur finibus orci.
<ul>
        <li>Hair coloring in [location]</li>
        <li>Eyebrow styling in [location]</li>
        <li>Washing hair in [location]</li>
</ul>
<h2>Book a [search_term] in [location]</h2>
Nulla sagittis urna ultrices tortor viverra hendrerit. Phasellus sit amet luctus mauris, eu finibus mauris. Sed blandit nulla in diam porta, ac viverra arcu pretium. Nulla facilisi. Suspendisse ut nisl consequat, maximus enim ut, congue augue. Vivamus eget vehicula augue. Aenean eget ligula sed lacus mollis congue.
<p style="text-align: center;"><a class="button button-primary" href="https://seogenerator.io/">Book now</a></p>';


    $new_page_id    = wp_insert_post(array(
        'post_title'	    => 'The best [search_term] in [location]',
        'post_status'	    => 'draft',
        'post_type'		    => 'nw_seo_page',
        'post_author'		=> 1,
        'post_content'		=> $post_content,
        'comment_status'	=> 'closed',
        'post_name'         => 'hairdressers'
    ));
    
    // And set page template
    update_post_meta($new_page_id,	'_wp_page_template', 'nsg-seo-generator/templates/template-page-no-sidebar.php');

    // Update Custom Fields
    update_post_meta($new_page_id, 'nsg-seo-page-base', 'hairdressers');
    update_post_meta($new_page_id, 'nsg-slug-placeholder', '[search_term]-in-[location]');
    update_post_meta($new_page_id, 'nsg-archive-page-title', 'Archive');
    update_post_meta($new_page_id, 'nsg-overview-label', '');   // Start empty

    update_post_meta($new_page_id, 'nsg-search-terms',
'Hairdresser | Hairdressers
Stylist | Stylists
Barber | Barbers
Beautician | Beauticians
Coiffeur
Cosmetologist
Friseur
Hair stylist | Hair stylists');

    update_post_meta($new_page_id, 'nsg-locations',
'Aberdeen
Airway Heights
Albany
Algona
Amsterdam
Anacortes
Arlington
Asotin
Auburn
Bainbridge Island
Batavia
Battle Ground
Beacon
Bellevue
Bellingham
Benton City
Bingen
Binghamton
Black Diamond
Blaine
Bonney Lake
Bothell
Bremerton
Brewster
Bridgeport
Brier
Buckley
Buffalo
Burien
Burlington
Camas
Canandaigua
Carnation
Cashmere
Castle Rock
Centralia
Chehalis
Chelan
Cheney
Chewelah
Clarkston
Cle Elum
Clyde Hill
Cohoes
Colfax
College Place
Colville
Connell
Corning
Cortland
Cosmopolis
Covington
Davenport
Dayton
Deer Park
Des Moines
Dunkirk
DuPont
Duvall
East Wenatchee
Edgewood
Edmonds
Electric City
Ellensburg
Elma
Elmira
Entiat
Enumclaw
Ephrata
Everett
Everson
Federal Way
Ferndale
Fife
Fircrest
Forks
Fulton
Geneva
George
Gig Harbor
Glen Cove
Glens Falls
Gloversville
Gold Bar
Goldendale
Grand Coulee
Grandview
Granger
Granite Falls
Harrington
Hoquiam
Hornell
Hudson
Ilwaco
Issaquah
Ithaca
Jamestown
Johnstown
Kahlotus
Kalama
Kelso
Kenmore
Kennewick
Kent
Kettle Falls
Kingston
Kirkland
Kittitas
La Center
Lacey
Lackawanna
Lake Forest Park
Lake Stevens
Lakewood
Langley
Leavenworth
Liberty Lake
Little Falls
Lockport
Long Beach
Longview
Lynden
Lynnwood
Mabton
Maple Valley
Marysville
Mattawa
McCleary
Mechanicville
Medical Lake
Medina
Mercer Island
Mesa
Middletown
Mill Creek
Millwood
Milton
Monroe
Montesano
Morton
Moses Lake
Mossyrock
Mount Vernon
Mountlake Terrace
Moxee
Mukilteo
Napavine
New Rochelle
New York
Newburgh
Newcastle
Newport
Niagara Falls
Nooksack
Normandy Park
North Bend
North Bonneville
North Tonawanda
Norwich
Oak Harbor
Oakville
Ocean Shores
Ogdensburg
Okanogan
Olean
Olympia
Omak
Oneida
Oneonta
Oroville
Orting
Oswego
Othello
Pacific
Palouse
Pasco
Pateros
Peekskill
Plattsburgh
Pomeroy
Port Angeles
Port Jervis
Port Orchard
Port Townsend
Poughkeepsie
Poulsbo
Prescott
Prosser
Pullman
Puyallup
Quincy
Rainier
Raymond
Redmond
Rensselaer
Renton
Republic
Richland
Ridgefield
Ritzville
Rochester
Rock Island
Rome
Roslyn
Roy
Royal City
Ruston
Rye
Salamanca
Sammamish
Saratoga Springs
Schenectady
SeaTac
Seattle
Sedro-Woolley
Selah
Sequim
Shelton
Sherrill
Shoreline
Snohomish
Snoqualmie
Soap Lake
South Bend
Spangle
Spokane
Spokane Valley
Sprague
Stanwood
Stevenson
Sultan
Sumas
Sumner
Sunnyside
Syracuse
Tacoma
Tekoa
Tenino
Tieton
Toledo
Tonasket
Tonawanda
Toppenish
Troy
Tukwila
Tumwater
Union Gap
University Place
Utica
Vader
Vancouver
Waitsburg
Walla Walla
Wapato
Warden
Washougal
Watertown
Watervliet
Wenatchee
West Richland
Westport
White Plains
White Salmon
Winlock
Woodinville
Woodland
Woodway
Yakima
Yelm
Yonkers
Zillah'
);
    
    // Workaround to flush the rewrite rules:
    update_option( 'nsgd-flush-rewrite-rules', 1 );
}
