<?php

// SEO pages are regular pages of the CPT nw_seo_page
// We have removed the CPT base
// Now these pages have a regular slug for the url
// We add the search terms and locations after this slug part

// Examples:
// domain.com/webdevelopment/webdesign-in-location/
// /webdevelopment/ is the slug of the actual CPT page
// /webdesign-in-location/ is the combination of [search_term]-in-[location]
// and the dynamic part is stored in a query_var: nsg_seo_page

/**
 * Filters the query variables allowed before processing.
 *
 * Adds query vars used by the plugin for the dynamiccaly generated urls
 *
 * @param string[] $public_query_vars The array of allowed query variable names.
 * 
 * @return array
 */
function nsg_add_query_vars($public_query_vars)
{
    // Use to generate all unique slugs that come after the url of the SEO Pages
    $public_query_vars[] = 'nsg_seo_page';
    $public_query_vars[] = 'nsg_seo_page_archive';
    
    // Used for XML sitemap
    $public_query_vars[] = 'nsg_sitemap';
    $public_query_vars[] = 'nsg_sitemap_number';

    return $public_query_vars;
}
add_filter('query_vars', 'nsg_add_query_vars', 0, 1);

/**
 * Fires after WordPress has finished loading but before any headers are sent.
 *
 * Adds rewrite rules
 */
function nsg_add_rewrite_rules()
{
    $nsg        = NSG_Seo_Generator::get_instance();

    $pages      = $nsg->nsg_get_seo_pages();

    $page_names = array();

    if (is_array($pages) && count($pages) > 0)
    {
        foreach ($pages as $page)
        {
            $page_names[$page->ID] = $page->post_name;
        }

        // /post-1/[search_term-1]-in-[location-1]
        // /post-1/[search_term-2]-in-[location-1]
        // /post-1/[search_term-1]-in-[location-2]
        // /post-1/[search_term-2]-in-[location-2]

        // /post-2/[search_term-1]-in-[location-1]
        // /post-2/[search_term-2]-in-[location-1]
        // /post-2/[search_term-1]-in-[location-2]
        // /post-2/[search_term-2]-in-[location-2]

        // etc

        global $wp_rewrite;
        
        // Create a separate rewrite rule for each CPT nw_seo_page so we can add the slug after these pages
        foreach ($page_names as $post_id => $page_name)
        {
            // ^hairdressers\/([\p{L}0-9-_]+)\/?$


            // [(.?.+?)/page/?([0-9]{1,})/?$]       index.php?pagename=$matches[1]&paged=$matches[2]
            add_rewrite_rule('^' . $page_name . '/' . $wp_rewrite->pagination_base . '/?([0-9]{1,})/?$',
                'index.php?post_type=nsg_seo_page&name=' . $page_name . '&paged=$matches[1]&nsg_seo_page_archive=1&p=' . (int)$post_id,
                'top');

            // index.php?&paged=$matches[1]
            $pattern = preg_quote($page_name, '') . '/([^/]+)(?:/(.*))?/?$';

            // IMPORTANT!!! 2nd PARAM MUST BE IN SINGLE QUOTES
            add_rewrite_rule(
                $pattern,
                'index.php?post_type=nw_seo_page&name=' . $page_name . '&nsg_seo_page=$matches[1]',
                'top'
            );
            
            // Create a rewrite rule for the fake archive page
            add_rewrite_rule(
                "^{$page_name}/?$",
                'index.php?post_type=nw_seo_page&name=' . $page_name . '&nsg_seo_page_archive=1',
                'top'
            );
        } 
    }
}
add_action('init', 'nsg_add_rewrite_rules', 999);

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                      PAGINATION - START
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

/**
 * Filters the canonical redirect URL.
 *
 * Returning false to this filter will cancel the redirect.
 *
 * @since 2.3.0
 *
 * @param string $redirect_url  The redirect URL.
 * @param string $requested_url The requested URL.
 */
add_filter('redirect_canonical', 'nsg_redirect_canonical_paged', 10, 2);
function nsg_redirect_canonical_paged($redirect_url, $requested_url)
{
    if ((int)get_query_var('nsg_seo_page_archive') && (int)get_query_var('paged') > 1)
    {
        $post_id        = (int)get_query_var('p');
        $lookup_table   = nsg_get_search_terms_and_locations_lookup_table($post_id);

        // Check if the current paged URL is in range
        $num_seo_pages  = count($lookup_table);
        $num_per_page   = apply_filters('nsg_archive_num_per_page', 10);
        $current_page   = (int)get_query_var('paged');
        $max_pages      = (int)ceil($num_seo_pages / $num_per_page);

        // We are still within range, keep user on /page/X/ and load the archive.php template
        if ($current_page > 0 && $current_page <= $max_pages)
        {
            $redirect_url = false;
        }
    }

    return $redirect_url;
}

add_action('pre_get_posts', 'nsg_pre_get_posts', 10, 1);
function nsg_pre_get_posts ($query)
{
    // Wordpress thinks we are querying a Post because the URL is paged
    if (!is_admin() && $query->is_main_query())
    {
        if ((int)get_query_var('nsg_seo_page_archive') && (int)get_query_var('paged') > 1)
        {
            $query->set('post_type', 'nw_seo_page');
        }
    }
    
    return $query;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                      PAGINATION - END
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

add_filter( 'rewrite_rules_array', 'nsg_rewrite_rules_array', 10, 1);
function nsg_rewrite_rules_array($rules)
{
    // Remove all other rewrite rules added by Wordpress for our CPT as we do not use these
    $rules = array_filter($rules, function($rule) 
    {
        return (strpos( $rule , 'nw_seo_page/' ) !== 0);
    }, ARRAY_FILTER_USE_KEY);

    return $rules;
}

/**
 * Filters the path of the current template before including it.
 *
 * Checks if we are on a SEO Page url with search term and location in the URL.
 * Will return selected template or 404 found
 * 
 * @param string $template The path of the template to include.
 * 
 * @return string The path of the template to include.
 */
function nsg_template_include($template)
{
    // Gradually increase checks. Easiest most lightweight first:
    if (is_singular())
    {
        $post = get_queried_object();
        
        if ($post->post_type === 'nw_seo_page')
        {
            $nsg    = NSG_Seo_Generator::get_instance();
            $slug   = $nsg->nsg_get_query_var_nsg_seo_page();

            if ($slug !== '')
            {
                $slugs  = $nsg->nsg_get_seo_page_slugs($post->ID);      // Get all slugs combinations for searchword and location for this specific seo page

                $slugs  = array_map('strtolower', $slugs);
                $slugs  = array_map('esc_html', $slugs);

                // We are actually on the url of a NW Seo Page!
                if ($slug !== false)
                {
                    // Can we find an existing slug with the correct searchword location pair?
                    if (is_array($slugs) && in_array($slug, $slugs))
                    {
						// Default template used by Wordpress: get_page_template_slug() returns empty string and falls back to single.php (or whatever default template in theme falls back on)
                        $template = nsg_get_template($post->ID);

                        // Make sure Block Themes fall back on a default template:
                        if (wp_is_block_theme())
                        {
                            $template = locate_block_template($template, 'page', array());
                        }
                    }
                    else
                    {
                        // Show 404 page when url = webdevelopment/webdevelopment-in-asd
                        // test / find out if template_include is correct location. maybe template redirect?
                        global $wp_query;
                        $wp_query->set_404();
                        status_header(404);
                        nocache_headers();

                        $template = locate_template('404.php');
                    }
                }
            }
        }
    }
    
    // Allow to override template
    $template = apply_filters('nsg_template_include', $template);
    
	return $template;
}

/**
 * Filters the path of the current template before including it.
 *
 * Checks if we are on a SEO Page archive url
 * Will return selected template or 404 found
 * 
 * @param string $template The path of the template to include.
 * 
 * @return string The path of the template to include.
 */
function nsg_template_include_archive($template)
{
    // Gradually increase checks. Easiest most lightweight first:
    if (is_singular())
    {
        $post = get_queried_object();
        
        if ($post->post_type === 'nw_seo_page')
        {
            if ((int)get_query_var('nsg_seo_page_archive') === 1)
            {
                // Elementor support
                if (isset($_REQUEST['elementor-preview']) && (int)$_REQUEST['elementor-preview'] === get_the_ID())
                {
                    $template = nsg_get_template($post->ID);
                }
                else
                {
                    // Give theme authors option to copy templates to the theme directory and load those templates instead
                    $template_filename  = 'nsg-archive.php';
                    $template 			= locate_template($template_filename);
                                
                    // No template was found in the theme, load template from plugin
                    if ($template === '')
                    {
                        $template = WP_PLUGIN_DIR . '/' . 'nsg-seo-generator/templates/' . $template_filename;
                    }
                }
            }
        }
    }
    
    // Allow to override template
    $template = apply_filters('nsg_template_include_archive', $template);
    
	return $template;
}

// Divi fix. Otherwise Divi pagebuilder will not load in the backend and show error message about "Incompatible Post Type"
// Divi will not load in the frontend anyway...
$nsg_template_include_archive_priority = 99999; // Default priority to make sure the Archive page loads the default template
if (isset($_GET['et_fb']) && (int)$_GET['et_fb'] === 1)
{
    // Low priority to make sure Divi pagebuilder works when editing a SEO Page
    $nsg_template_include_archive_priority = -10;
}
add_filter('template_include', 'nsg_template_include', 10, 1);
add_filter('template_include', 'nsg_template_include_archive', $nsg_template_include_archive_priority, 1);


/**
 * Remove trailing slash from seo_generator_sitemap_index.xml
 * 
 * @param string $redirect The redirect URL currently determined.
 * 
 * @see https://wordpress.stackexchange.com/questions/281140/help-to-remove-last-trailing-slash-using-add-rewrite-rule
 */
function redirect_canonical_callback($redirect)
{
    $nsg_sitemap_query_var = get_query_var('nsg_sitemap', false);
    if ($nsg_sitemap_query_var === 'nsg_show_sitemap_index' || $nsg_sitemap_query_var === 'nsg_show_sitemap')
    {
        return false;
    }

    return $redirect;
}
add_filter('redirect_canonical', 'redirect_canonical_callback', 100, 1);

/**
 * Filters the permalink for a post of a custom post type.
 *
 * Removes CPT Slug from urls
 *
 * @param string  $post_link The post's permalink.
 * @param WP_Post $post      The post in question.
 * @param bool    $leavename Whether to keep the post name.
 * 
 * @return string The post's permalink.
 */
function nsg_remove_slug($post_link, $post, $leavename)
{
    if ($post->post_type === 'nw_seo_page' && ($post->post_status === 'publish' || $post->post_status === 'draft'))
    {
        $nsg    = NSG_Seo_Generator::get_instance();
        $slug   = $nsg->nsg_get_query_var_nsg_seo_page();   // Safely gets get_query_var('nsg_seo_page') with non-latin characters decoded properly
            
        // Check $slug for the page we are currently on
        // Check strstr($post_link, '/nw_seo_page/') used by next_post_link() / previous_post_link() which use post_type_link()
        if ($slug === '' || strstr($post_link, '/nw_seo_page/'))
        {
            // https://gist.github.com/kellenmace/f8a3393385f01ee226a087ff19cc6056
            // Remove /nw_seo_page/ from the url    
            if (strstr($post_link, '/nw_seo_page/'))
            {
                $post_link = str_replace( '/nw_seo_page/', '/', $post_link );
            }
        }
    }

    return $post_link;
}
add_filter('post_type_link', 'nsg_remove_slug', 10, 3);

// Change links added by the Wordpress menu
function filter_wp_nav_menu_objects( $sorted_menu_items, $args)
{ 
    foreach ($sorted_menu_items as $item)
    {
        if ($item->object === 'nw_seo_page' && strstr($item->url, '/nw_seo_page/'))
        {
            $item->url = str_replace( '/nw_seo_page/', '/', $item->url);
        }
    }

    return $sorted_menu_items; 
}; 
add_filter( 'wp_nav_menu_objects', 'filter_wp_nav_menu_objects', 10, 2 ); 

////////////////////////////////////////////////////////////////////////////////////////////////////////
//                               SITEMAP URL REWRITES / REDIRECTS / ETC
////////////////////////////////////////////////////////////////////////////////////////////////////////

/**
 * Fires after WordPress has finished loading but before any headers are sent.
 *
 * Adds rewrite rules for the sitemap index and sitemap pages
 */
function nsg_seo_generator_sitemap_xml_add_rewrite_rule()
{
    $sitemap_name = nsg_get_sitemap_name();

    // SITEMAP: listen to <SITE_URL>/seo_generator_sitemap_index.xml
    add_rewrite_rule(
        $sitemap_name . '_index.xml',
        'index.php?nsg_sitemap=nsg_show_sitemap_index',
        'top'
    );

    // SITEMAP: listen to <SITE_URL>/seo_generator_sitemap_<pagenumber>.xml
    add_rewrite_rule(
        $sitemap_name . '_([0-9]+)?\.xml',
        'index.php?nsg_sitemap=nsg_show_sitemap&nsg_sitemap_number=$matches[1]', 'top',
        'top'
    );
}
add_action( 'init', 'nsg_seo_generator_sitemap_xml_add_rewrite_rule' );

/**
 * Filters whether to short-circuit default header status handling.
 * 
 * Do not return 404 headers for existing sitemap urls
 * 
 * @param bool     $preempt  Whether to short-circuit default header status handling. Default false.
 */
add_filter( 'pre_handle_404', 'nsg_pre_handle_404', 10, 1);
function nsg_pre_handle_404 ($preempt)
{
    $nsg_sitemap_query_var = get_query_var('nsg_sitemap', false);
    if ($nsg_sitemap_query_var === 'nsg_show_sitemap_index' || $nsg_sitemap_query_var === 'nsg_show_sitemap')
    {
        $preempt = true;
    }
    
    return $preempt;
}

/**
 * Fires once the WordPress environment has been set up.
 * 
 * Adds actions to redirect sitemap urls and show correct content
 */
function nsg_template_redirect_sitemap()
{
    $nsg_sitemap = get_query_var('nsg_sitemap', false);
    
    if ($nsg_sitemap === 'nsg_show_sitemap_index')
    {
        add_action('template_redirect', 'nsg_show_sitemap_index');
    }
    else if ($nsg_sitemap === 'nsg_show_sitemap')
    {
        add_action('template_redirect', 'nsg_show_sitemap');
    }
}
add_action('wp', 'nsg_template_redirect_sitemap');

/**
 * Shows sitemap index with all sitemap pages
 */
function nsg_show_sitemap_index()
{
    global $nsg_seo_generator_dir;

    require_once "{$nsg_seo_generator_dir}/sitemap/sitemap-functions.php";
    require_once "{$nsg_seo_generator_dir}/sitemap/xml-sitemap-index.php";

    exit;
}

/**
 * Shows sitemap page with dynamically generated SEO pages
 */
function nsg_show_sitemap()
{
    global $nsg_seo_generator_dir;

    require_once "{$nsg_seo_generator_dir}/sitemap/sitemap-functions.php";
    require_once "{$nsg_seo_generator_dir}/sitemap/xml-sitemap.php";

    exit;
}

/**
 * Gets sitemap URL
 * 
 * @param bool|int  $current_page Index of current page to retrieve. Or return sitemap index url if false
 * 
 * @return string   The sitemap url
 */
function nsg_get_sitemap_url($current_page = false)
{
    $sitemap_url    = '';
    $sitemap_name   = nsg_get_sitemap_name();

    if ($current_page !== false)
    {
        $current_page   = (int)$current_page;
        $sitemap_url    = home_url() . "/" . $sitemap_name . "_{$current_page}.xml";
    }
    else
    {
        $sitemap_url = home_url() . "/" . $sitemap_name . "_index.xml";
    }

    return $sitemap_url;
}

function nsg_template_redirect_pagebuilder_support()
{
    $continue = true;

    // Elementor support
    // Elementor loads the front end of the page in a iframe when editing the page in the backend
    // <iframe id="elementor-preview-iframe" src="https://domain.com/se-page-base/?elementor-preview=16&amp;ver=1748590704" title="Preview" allowfullscreen="1"></iframe>
    if (isset($_REQUEST['elementor-preview']) && (int)$_REQUEST['elementor-preview'] === get_the_ID())
    {
        $continue = false;
    }
    // Divi support
    else if (isset($_REQUEST['et_fb']) && (int)$_REQUEST['et_fb'] === 1)
    {
        $continue = false;
    }
    // Bricks support
    // Bricks loads the front end of the page in a iframe when editing the page in the backend
    // <iframe id="bricks-builder-iframe" src="https://domain.com/se-page-base/?bricks=run&amp;brickspreview"></iframe>
    else if (isset($_REQUEST['bricks']) && $_REQUEST['bricks'] === 'run')
    {
        $continue = false;
    }

    return $continue;
}

/**
 * Conditionally disables the SEO Generator archive page.
 *
 * This function checks if the current request is for an SEO Generator archive page,
 * and if the "Disable Archive Page" setting is enabled.
 * Depending on the admin configuration, it will either:
 * - Redirect to a specified URL (301),
 * - Or trigger a 404 error page.
 *
 * The behavior is controlled by the following WordPress options:
 * - 'nsg-enable_archive_page' (boolean): Whether to enable the archive.
 * - 'nsg-archive_page_disabled_behavior' (string): 'redirect' or '404'.
 * - 'nsg-archive_page_redirect_url' (string): URL to redirect to if 'redirect' is selected.
 *
 * Hooked into 'template_redirect'.
 *
 * @return void
 */
add_action('template_redirect', 'nsg_handle_archive_page_template_redirect');
function nsg_handle_archive_page_template_redirect()
{
    // Pagebuilder Support
    if (!nsg_template_redirect_pagebuilder_support())
    {
        return;
    }

    $page_id = get_the_ID();

    // This might be a replacement archive page. If so, we need to redirect to the actual SEO Page archive url
    if (get_post_type($page_id) !== 'nw_seo_page')
    {
        // $page_id ID of the current page the visitor is viewing.
        // We need to find a SEO Page which has THIS page set as it's Custom Archive Page. If there is any...
        $seo_page_id    = false;
        $seo_query      = new WP_Query(array(
            'post_type'      => 'nw_seo_page',
            'posts_per_page' => 1,
            'meta_query'     => array(
                array(
                    'key'   => 'nsg-archive_page_custom_page_id',
                    'value' => $page_id,
                ),
            ),
        ));

        if ($seo_query->have_posts())
        {
            $seo_query->the_post();
            $seo_page_id = get_the_ID();
            wp_reset_postdata();
        }

        global $wp_query;

        if ($seo_page_id !== false && (int)$seo_page_id > 0 && !nsg_is_nsg_archive_page() && !property_exists($wp_query, 'original_seo_page_id'))
        {
            $enable_archive_page            = nsg_get_field('nsg-enable_archive_page', $seo_page_id, true, false);
            $enable_custom_archive_page     = nsg_get_field('nsg-enable_custom_archive_page', $seo_page_id, null, false);
            // Not needed, just to double check:
            $custom_page_id                 = (int)nsg_get_field('nsg-archive_page_custom_page_id', $seo_page_id, null, false);

            // Now check if in the original SEO page the archive is enabled and the custom archive page is enabled and THIS is that replacement page
            if ($enable_archive_page && $enable_custom_archive_page && $page_id === $custom_page_id)
            {
                // Let's redirect this other regular page to the actual SEO Page archive url
                wp_redirect(get_permalink($seo_page_id), 301);
                exit;
            }
        }
    }
    else if (get_post_type($page_id) === 'nw_seo_page')
    {
        // Let's check if the Archive Page is even enabled or not
        $seo_page_id            = $page_id;
        $enable_archive_page    = nsg_get_field('nsg-enable_archive_page', $seo_page_id, true, false);

        // First check if we are on the archive page and if the archive page is disabled
        if (!is_admin() && nsg_is_nsg_archive_page() && !$enable_archive_page)
        {
            // The archive page is disabled. Let's see if we need to show a 404 or redirect somewhere else
            $behavior = nsg_get_field('nsg-archive_page_disabled_behavior', $seo_page_id, '404', false);

            if ($behavior === 'redirect')
            {
                $archive_page_redirect_url  = trim(nsg_get_field('nsg-archive_page_redirect_url', $seo_page_id, null, false));

                if (!empty($archive_page_redirect_url))
                {
                    wp_redirect(esc_url_raw($archive_page_redirect_url), 301);
                    exit;
                }
            }
            else if ($behavior === '404')
            {
                // Default: show 404 or if the redirect url field is empty
                global $wp_query;
                $wp_query->set_404();
                status_header(404);
                nocache_headers();
                // Do NOT call exit here — allow WP to load 404.php
            }
        }
    }
}

add_action('wp', 'nsg_prepare_replacement_archive_page', 1);
function nsg_prepare_replacement_archive_page()
{
    // Pagebuilder Support
    if (!nsg_template_redirect_pagebuilder_support())
    {
        return;
    }

    if (is_admin())
    {
        return;
    }

    $page_id = get_the_ID();

    $custom_archive_page_id = null;

    // We might be on a Custom SEO Archive Page because the post type is something else
    // Check for get_query_var('nsg_seo_page_archive') because we cannot use nsg_is_nsg_archive_page() yet!
    if (get_post_type($page_id) === 'nw_seo_page' && (int)get_query_var('nsg_seo_page_archive'))
    {
        $enable_archive_page        = nsg_get_field('nsg-enable_archive_page', $page_id, true, false);
        $enable_custom_archive_page = nsg_get_field('nsg-enable_custom_archive_page', $page_id, null, false);
        $custom_archive_page_id     = (int)nsg_get_field('nsg-archive_page_custom_page_id', $page_id, null, false);
    }

    // Did we find a page ($seo_page_id) which has this ($seo_archive_page_id) as it's archive page?
    // Next we need to check of the archive page is even enabled and has the custom archive page setting enabled
    // We need to check this because the database value of nsg-archive_page_custom_page might still be set regardless of the other settings
    if ((int)$custom_archive_page_id > 0 && nsg_is_nsg_archive_page())
    {
        // Do we indeed want to show a custom archive page?
        if ($enable_archive_page && $enable_custom_archive_page)
        {
            // Override global post BEFORE SEO plugins run
            global $post, $wp_query;
            $post = get_post($custom_archive_page_id);  // Important that this is called $post

            if ($post->post_status === 'publish')
            {
                setup_postdata($post);

                $wp_query->is_single            = false;    // Make sure not the post template is loaded
                $wp_query->is_page              = true;     // But the page template is loaded as the selected Replacement Archive Page is from the post type = page
                $wp_query->post                 = $post;
                $wp_query->posts                = array($post);
                $wp_query->queried_object       = $post;
                $wp_query->queried_object_id    = $custom_archive_page_id;
                $wp_query->original_seo_page_id = $page_id; // Avoid infinite loop in nsg_handle_archive_page_template_redirect() because we change get_the_ID() here by overwriting the $wp_query
            }
        }
        else
        {
            // SHOW DEFAULT ARCHIVE PAGE
        }
    }
    else
    {
        // DISABLED ARCHIVE PAGE or just another page
    }
}

