<?php

namespace QuadLayers\PWB_PRO\Frontend;

use QuadLayers\PWB_PRO\Helpers;

class Load {

	protected static $instance;

	public function __construct() {
		add_action( 'pre_get_posts', array( $this, 'search_by_brand_name' ) );
		add_filter( 'woocommerce_get_breadcrumb', array( $this, 'add_woocommerce_breadcrumb_trail' ), 10, 2 );
		add_action( 'wp', array( $this, 'long_brand_desc_position' ) );

		add_action(
			'wp',
			function () {

				$current_page = self::get_current_page();

				if ( ! $current_page || ! is_string( $current_page ) ) {
					return;
				}

				remove_action( 'woocommerce_after_shop_loop_item_title', array( 'QuadLayers\\PWB\\WooCommerce', 'show_brands_in_loop' ), 9 );

				$action = 'after_shop_loop_item_title';
				$order  = 9;

				switch ( $current_page ) {
					case 'archive':
						$action = get_option( 'wc_pwb_admin_tab_brands_in_loop_hook' );
						$order  = get_option( 'wc_pwb_admin_tab_brands_in_loop_hook_order', 9 );
						break;
					case 'taxonomy-brand':
						$action = get_option( 'wc_pwb_admin_tab_archives_brand_in_loop_hook' );
						$order  = get_option( 'wc_pwb_admin_tab_archives_brand_in_loop_hook_order', 9 );
						break;
				}

				add_action( 'woocommerce_' . $action, array( 'QuadLayers\\PWB\\WooCommerce', 'show_brands_in_loop' ), $order );
			}
		);

		add_filter( 'option_woocommerce_catalog_columns', array( $this, 'loop_shop_columns' ), 10, 1 );

		add_filter( 'pwb_text_before_brands_links', array( $this, 'pwb_text_before_brands_links' ) );
		add_filter( 'woocommerce_show_page_title', array( $this, 'woocommerce_show_page_title' ) );
		add_filter( 'option_wc_pwb_admin_tab_brands_in_loop', array( $this, 'brands_in_loop' ) );
		add_filter( 'option_wc_pwb_admin_tab_brands_in_loop_separator', array( $this, 'brands_links_separator' ) );
	}

	/**
	 * Add brand to breadcrumb trail
	 *
	 * @param [type] $breadcrumbs
	 * @param [type] $breadcrumb_class
	 * @return array
	 */
	public function add_woocommerce_breadcrumb_trail( $breadcrumbs, $breadcrumb_class ) {

		if ( ! is_product() ) {
			return $breadcrumbs;
		}

		if ( 'no' === get_option( 'wc_pwb_admin_tab_brand_single_breadcrumbs', 'no' ) ) {
			return $breadcrumbs;
		}

		$brands = wp_get_post_terms( get_the_ID(), 'pwb-brand' );

		$brand = array_splice( $brands, 0, 1 );

		if ( ! isset( $brand[0]->term_id, $brand[0]->name ) ) {
			return $breadcrumbs;
		}

		$brand_link = get_term_link( $brand[0]->term_id );
		$brand_name = $brand[0]->name;

		if ( 'replace' === get_option( 'wc_pwb_admin_tab_brand_single_breadcrumbs', 'no' ) ) {
			Helpers::array_replace_position( $breadcrumbs, -1, array( array( $brand_name, $brand_link ) ) );
			return $breadcrumbs;
		}

		Helpers::array_push_to_position( $breadcrumbs, -1, array( array( $brand_name, $brand_link ) ) );

		return $breadcrumbs;
	}

	/**
	 *  Redirect if the search matchs with a brands name
	 *  Better search experience
	 *
	 * @param [type] $query
	 * @return void
	 */
	public function search_by_brand_name( $query ) {

		if ( wp_doing_ajax() ) {
			return;
		}

		if ( 'yes' !== get_option( 'wc_pwb_admin_tab_brands_search', 'no' ) ) {
			return;
		}

		if ( ! is_admin() && $query->is_main_query() && $query->is_search() ) {

			$brands = get_terms(
				array(
					'taxonomy' => 'pwb-brand',
					'fields'   => 'id=>name',
				)
			);

			$match = array_search( strtolower( trim( $query->get( 's' ) ) ), array_map( 'strtolower', $brands ) );

			if ( $match ) {
				wp_redirect( get_term_link( $match, 'pwb-brand' ) );
				exit;
			}
		}
	}

	/**
	 * Add separator to shop, brand and product pages
	 *
	 * @param [type] $value
	 * @return strng|html
	 */
	public function brands_links_separator( $value ) {

		$current_page = self::get_current_page();

		switch ( $current_page ) {
			case 'taxonomy-brand':
				$value = get_option( 'wc_pwb_admin_tab_archives_brand_in_loop_separator', $value );
				break;
			case 'product':
				$$value = get_option( 'wc_pwb_admin_tab_brands_in_single_separator', $value );
				break;
		}

		return $value;
	}

	/**
	 * Get current page
	 *
	 * @return string
	 */
	public static function get_current_page() {

		if ( is_tax( 'pwb-brand' ) ) {
			return 'taxonomy-brand';
		} elseif ( is_shop() || is_product_taxonomy() ) {
			return 'archive';
		} elseif ( is_product() ) {
			return 'product';
		}

		return false;
	}

	/**
	 * Hook WooCommerce brand archive columns
	 *
	 * @param [type] $columns
	 * @return number
	 */
	public function loop_shop_columns( $columns ) {
		$current_page = self::get_current_page();
		if ( 'taxonomy-brand' === $current_page && get_option( 'wc_pwb_admin_tab_brand_columns' ) ) {
			return get_option( 'wc_pwb_admin_tab_brand_columns' );
		}
		return $columns;
	}

	/**
	 * Change the label of the brand in the single product page
	 *
	 * @param [type] $label
	 * @return string
	 */
	public function pwb_text_before_brands_links( $label ) {
		$current_page = self::get_current_page();
		if ( 'product' === $current_page && null !== get_option( 'wc_pwb_admin_tab_brand_single_label', null ) ) {
			return get_option( 'wc_pwb_admin_tab_brand_single_label' );
		}
		return $label;
	}

	/**
	 * Allow/disallow the page title in the brand archive page
	 *
	 * @return bool
	 */
	public function woocommerce_show_page_title( $value ) {
		$current_page = self::get_current_page();
		if ( 'taxonomy-brand' === $current_page && 'no' === get_option( 'wc_pwb_admin_tab_brand_title', 'yes' ) ) {
			return false;
		}
		return $value;
	}

	/**
	 * Allow/disallow brands in loop
	 *
	 * @return bool
	 */
	public function brands_in_loop( $value ) {
		$current_page = self::get_current_page();
		if ( 'taxonomy-brand' === $current_page ) {
			return get_option( 'wc_pwb_admin_tab_archives_brand_in_loop', $value );
		}
		return $value;
	}

	/**
	 * Show/not show long brand description and set position
	 */
	public function long_brand_desc_position() {
		if ( is_tax( 'pwb-brand' ) && ! is_paged() ) {

			$show_desc = get_option( 'wc_pwb_admin_tab_long_brand_desc' );

			if ( 'yes' == $show_desc ) {
				/**
				 * Show second brand description before loop
				 */
				add_action( 'woocommerce_archive_description', array( $this, 'print_long_brand_description' ), 15 );
			} elseif ( 'yes_after_loop' == $show_desc ) {
				/**
				 * Show second brand description after loop
				 */
				add_action( 'woocommerce_after_main_content', array( $this, 'print_long_brand_description' ), 9 );
			}
		}
	}

	/**
	 * Print long brand description
	 */
	public function print_long_brand_description() {
		$queried_object  = get_queried_object();
		$show_desc       = get_option( 'wc_pwb_admin_tab_long_brand_desc' );
		$show_desc_class = ( ! $show_desc || 'yes' == $show_desc ) ? 'pwb-before-loop' : 'pwb-after-loop';
		$long_brand_desc = get_term_meta( $queried_object->term_id, 'pwb_long_brand_desc', true );
		ob_start();
		if ( '' != $long_brand_desc && 'no' !== $show_desc ) {
			echo '<div class="pwb-brand-banner-cont ' . esc_attr( $show_desc_class ) . '">';
			echo '<div class="pwb-brand-description ' . esc_attr( $show_desc_class ) . '">';
			echo do_shortcode( apply_filters( 'the_content', $long_brand_desc ) );
			echo '</div>';
			echo '</div>';
		}
	}

	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
}
