<?php

namespace PublishPress\FuturePro\Modules\Notifications\Controllers;

use PublishPress\Future\Core\HookableInterface;
use PublishPress\Future\Framework\InitializableInterface;
use PublishPress\FuturePro\Modules\Notifications\HooksAbstract;
use PublishPress\FuturePro\Modules\Notifications\Models\NotificationModel;
use PublishPress\FuturePro\Modules\Notifications\Models\NotificationsModel;

class RestApiV1 implements InitializableInterface
{
    /**
     * @var HookableInterface
     */
    private $hooks;

    /**
     * @var string
     */
    private $restApiV1BasePath;

    public function __construct(
        string $restApiV1BasePath,
        HookableInterface $hooks
    ) {
        $this->restApiV1BasePath = $restApiV1BasePath;
        $this->hooks = $hooks;
    }

    public function initialize()
    {
        $this->hooks->addFilter(
            HooksAbstract::FILTER_REGISTER_REST_ROUTES,
            [$this, 'registerRoutes']
        );
    }

    public function registerRoutes()
    {
        register_rest_route(
            $this->restApiV1BasePath,
            '/notifications/unread',
            [
                'methods' => 'GET',
                'callback' => [$this, 'getUnreadNotifications'],
                'permission_callback' => [$this, 'checkNotificationsPermission'],
            ]
        );

        register_rest_route(
            $this->restApiV1BasePath,
            '/notifications/execute-action-and-return-unread',
            [
                'methods' => 'POST',
                'callback' => [$this, 'executeActionAndReturnUnreadNotifications'],
                'permission_callback' => [$this, 'checkNotificationsPermission'],
                'args' => [
                    'notificationIds' => [
                        'type' => 'array',
                        'required' => true,
                    ],
                    'optionName' => [
                        'type' => 'string',
                        'required' => true,
                    ],
                ],
            ]
        );
    }

    public function getUnreadNotifications()
    {
        $currentUser = wp_get_current_user();

        $notificationsModel = new NotificationsModel();
        $notifications = $notificationsModel->getUnreadUserNotifications($currentUser->ID);
        $notifications = $notificationsModel->fillNotificationOptionsByEvent($notifications);

        return rest_ensure_response($notifications);
    }

    public function checkNotificationsPermission()
    {
        return is_user_logged_in();
    }

    public function executeActionAndReturnUnreadNotifications(\WP_REST_Request $request)
    {
        $notificationIds = $request->get_param('notificationIds');
        $optionName = $request->get_param('optionName');

        if (! is_array($notificationIds)) {
            return new \WP_Error('invalid_request', 'Notification IDs must be an array', ['status' => 400]);
        }

        /**
         * @since 4.7.0
         *
         * Action triggered when the notification option is executed.
         *
         * @param array $notificationIds The notification IDs.
         * @param string $optionName The option name.
         */
        $this->hooks->doAction(
            HooksAbstract::ACTION_EXECUTE_NOTIFICATION_OPTION,
            $notificationIds,
            $optionName
        );

        return $this->getUnreadNotifications();
    }
}
