<?php
namespace QuizPress\API\Controller;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class MagicLogin {
    private $namespace = '';
    public static function init() {
		$self            = new self();
		$self->namespace = QUIZPRESS_PLUGIN_SLUG . '/v1';
		add_action( 'rest_api_init', array( $self, 'register_routes' ) );
	}

    public function register_routes(){
        // magic/login/request
        register_rest_route($this->namespace , '/magic-login', [
            'methods' => 'POST',
            'callback' => [$this, 'handle_magic_login_request'],
            'permission_callback' => '__return_true',
            'args' => [
                'email' => [
                    'required' => true,
                    'validate_callback' => function($email){
                        if(! is_email($email)){
                            return new \WP_Error( 'invalid_email', 'The provided email address is not valid.', [ 'status' => 400 ] );
                        }
                        return true;
                    }
                ]
            ]
        ]);

        // magic/login/verify api
        register_rest_route($this->namespace , '/magic-login/verify', [
            'methods' => 'GET',
            'callback' => [$this, 'handle_magic_login_verify'],
            'permission_callback' => '__return_true',
            'args' => [
                'token'   => [ 'required' => true ],
                'user_id' => [ 
                    'required' => true, 
                    'sanitize_callback' => 'absint', // Ensure user_id is a positive integer
                ],
            ]
        ]);
    }

    public function handle_magic_login_request( $request ) {
        $email = sanitize_email( $request->get_param( 'email' ) );

        // Validate email format
        if ( ! is_email( $email ) ) {
            return new \WP_REST_Response(
                [ 'success' => false, 'message' => __( 'Invalid email format.', 'quizpress' ) ],
                400
            );
        }

        $user = get_user_by( 'email', $email );

        // 1. Create new user if not found
        if ( ! $user ) {
            $username = sanitize_user( current( explode( '@', $email ) ), true );

            // Generate a unique username if it already exists
            $base_username = $username;
            $i = 1;
            while ( username_exists( $username ) ) {
                $username = $base_username . $i;
                $i++;
            }

            $user_id = wp_insert_user( [
                'user_login' => $username,
                'user_email' => $email,
                'user_pass'  => wp_generate_password( 12, false ),
                'role'       => 'subscriber',
            ] );

            if ( is_wp_error( $user_id ) ) {
                return new \WP_REST_Response(
                    [ 'success' => false, 'message' => 'Error registering user: ' . esc_html( $user_id->get_error_message() ) ],
                    500
                );
            }

            $user = get_user_by( 'ID', $user_id );

        } else {
            $user_id = $user->ID;
        }

        // 2. Generate secure token and login link
        $token = wp_create_nonce( 'quizpress_magic_login_' . $user_id );

        $url = wp_get_referer(); // get URL where user came from
        $parsed_url = wp_parse_url($url);

        $clean_url = '';

        // Only append scheme and host if they exist
        if ( isset($parsed_url['scheme'], $parsed_url['host']) ) {
            $clean_url = $parsed_url['scheme'] . '://' . $parsed_url['host'];

            // Append port if exists
            if (isset($parsed_url['port'])) {
                $clean_url .= ':' . $parsed_url['port'];
            }

            // Append path if exists
            if (isset($parsed_url['path'])) {
                $clean_url .= $parsed_url['path'];
            }
        }

        $link = add_query_arg(
            [
                'token'        => $token,
                'user_id'      => $user_id,
                'redirect_url' => $clean_url,
            ],
            rest_url( trailingslashit( $this->namespace ) . 'magic-login/verify' )
        );

        $login_link = esc_url_raw( $link );
        // fire the hook after create magic link
        do_action('quizpress/after_create_magic_link', $login_link, $user);

        // Return REST success response
        return new \WP_REST_Response(
            [ 'success' => true, 'message' => __( 'Magic login link sent successfully.', 'quizpress' ) ],
            200
        );
    }

    public function handle_magic_login_verify($request) {
        $user_id = (int) $request->get_param( 'user_id' );
        $token   = sanitize_text_field( $request->get_param( 'token' ) );
        $redirect_url   = sanitize_text_field( $request->get_param( 'redirect_url' ) );
        
        if (!wp_verify_nonce($token, 'quizpress_magic_login_' . $user_id)) {
            return new \WP_REST_Response(['success' => false, 'message' => 'Invalid or expired token.'], 401);
        }

        $user = get_user_by('ID', $user_id);
        if (!$user) {
            return new \WP_REST_Response(['success' => false, 'message' => 'User not found.'], 404);
        }

        wp_set_current_user($user->ID);
        wp_set_auth_cookie($user->ID, true);
        wp_safe_redirect( esc_url_raw( urldecode( $redirect_url ) ) );
        exit;
    }
}