<?php
namespace QuizPress\API\Controller;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use QuizPress\API\Query\Questions as Query;
use QuizPress\API\Schema\QuestionSchema;


class Questions extends \WP_REST_Controller {

	use QuestionSchema;

	public static function init() {
		$self            = new self();
		$self->namespace = QUIZPRESS_PLUGIN_SLUG . '/v1';
		$self->rest_base = 'questions';
		add_action( 'rest_api_init', array( $self, 'register_routes' ) );
	}

	/**
	 * Register the routes for the objects of the controller.
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => '__return_true',
					'args'                => $this->get_collection_params(),
				),
				array(
					'methods'             => \WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_item_schema(),
				),
				'schema' => [ $this, 'get_public_item_schema' ],
			)
		);

		$get_item_args = array(
			'context' => $this->get_context_param( array( 'default' => 'view' ) ),
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				'args'   => array(
					'id' => array(
						'description' => esc_html__( 'Unique identifier for the object.', 'quizpress' ),
						'type'        => 'integer',
					),
				),
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => '__return_true',
					'args'                => $get_item_args,
				),
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'update_item' ),
					'permission_callback' => array( $this, 'update_item_permissions_check' ),
					'args'                => $this->get_item_schema(),
				),
				array(
					'methods'             => \WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_item' ),
					'permission_callback' => array( $this, 'delete_item_permissions_check' ),
					'args'                => array(
						'force' => array(
							'type'        => 'boolean',
							'default'     => false,
							'description' => esc_html__( 'Whether to bypass Trash and force deletion.', 'quizpress' ),
						),
					),
				),
				'schema' => [ $this, 'get_public_item_schema' ],
			)
		);

		register_rest_route( $this->namespace, '/render_quiz/(?P<id>[\d]+)', 
			array(
				'methods'  => 'GET',
				'callback' => array( $this, 'get_question_by_quiz' ),
				'permission_callback' => array( $this, 'render_quiz_item_permission_check' ),
				'args' => $get_item_args
			)
		);

	}

	public function create_item_permissions_check( $request ) {
		if ( ! current_user_can( 'edit_posts' ) ) {
			return new \WP_Error(
				'rest_forbidden_context',
				esc_html__( 'Sorry, you are not allowed to create quiz question', 'quizpress' ),
				array( 'status' => rest_authorization_required_code() )
			);
		}
		return true;
	}

	public function update_item_permissions_check( $request ) {
		if ( ! current_user_can( 'edit_posts' ) ) {
			return new \WP_Error(
				'rest_forbidden_context',
				esc_html__( 'Sorry, you are not allowed to update quiz question', 'quizpress' ),
				array( 'status' => rest_authorization_required_code() )
			);
		}
		return true;
	}

	public function delete_item_permissions_check( $request ) {
		if ( ! current_user_can( 'edit_posts' ) ) {
			return new \WP_Error(
				'rest_forbidden_context',
				esc_html__( 'Sorry, you are not allowed to delete quiz question', 'quizpress' ),
				array( 'status' => rest_authorization_required_code() )
			);
		}
		return true;
	}

	public function render_quiz_item_permission_check( $request ) {
		if ( ! is_user_logged_in() ) {
			return new \WP_Error(
				'rest_forbidden_context',
				esc_html__( 'Sorry, you are not allowed to play this quiz question', 'quizpress' ),
				array( 'status' => rest_authorization_required_code() )
			);
		}
		return true;
	}


	/**
	 * Retrieves a collection of posts.
	 *
	 * @since 4.7.0
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object on success, or \WP_Error object on failure.
	 */
	public function get_items( $request ) {
		$page     = max( 1, absint( $request->get_param( 'page' ) ) );
		$per_page = max( 1, absint( $request->get_param( 'per_page' ) ) );
		$keyword  = $request->get_param( 'keyword' );

		$args = array(
			'per_page' => $per_page,
			'offset'   => ( $page - 1 ) * $per_page,
			'status'   => $request->get_param( 'status' ),
			'keyword'  => $keyword,
		);
	
		$questions = Query::get_quiz_questions( $args );

		$data = array();
		if ( empty( $questions ) ) {
			$response = rest_ensure_response( $data );
			$response->header( 'x-wp-total', 0);
			return $response;
		}

		foreach ( $questions as $question ) {
			$response = $this->rest_prepare_item( $question, $request );
			$response['total_answers'] = count( Query::get_total_question_answers( $question->question_id ) );
			$answers = \QuizPress\API\Query\Answers::get_question_answers_by_question_id( $question->question_id );
			foreach ( $answers as &$answer ) {
				$answer = \QuizPress\API\Controller\Answers::prepare_answer_for_response( $answer );
			}
			unset( $answer );
			$response['answers'] = $answers;

			$data[] = $this->rest_prepare_for_collection( $response );
		}

		$response = rest_ensure_response( $data );
		$response->header( 'x-wp-total', Query::get_total_questions() );

		return $response;
	}

	public function get_item( $request ) {
		
		$id = absint( $request->get_param( 'id' ) );
	
		$question = Query::get_quiz_question( $id );

		if ( empty( $question ) ) {
			return rest_ensure_response( array() );
		}
		$response = $this->rest_prepare_for_collection(
			$this->rest_prepare_item( $question, $request )
		);
		
		$answers = \QuizPress\API\Query\Answers::get_question_answers_by_question_id( $id );
		foreach ( $answers as &$answer ) {
			$answer = \QuizPress\API\Controller\Answers::prepare_answer_for_response( $answer );
		}
		unset( $answer );
		$response['answers'] = $answers;
		
		return rest_ensure_response( $response );
	}

	public function get_question_by_quiz( $request ) {
		$quiz_id = absint( $request->get_param( 'id' ) );
		$data = get_post( $quiz_id );
		$raw_meta = get_post_meta( $quiz_id );

		$meta = array();
		foreach ( $raw_meta as $key => $value ) {
			$meta[ $key ] = maybe_unserialize( $value[0] );
		}
		$data->meta = (object) $meta;

		$data->meta->quizpress_quiz_time = (int) $data->meta->quizpress_quiz_time;
		$data->meta->quizpress_quiz_hide_quiz_time = (bool) $data->meta->quizpress_quiz_hide_quiz_time;
		$data->meta->quizpress_quiz_auto_start = (bool) $data->meta->quizpress_quiz_auto_start;
		$data->meta->quizpress_quiz_schedule_enabled = (bool) $data->meta->quizpress_quiz_schedule_enabled;
		$data->meta->quizpress_quiz_passing_grade = (int) $data->meta->quizpress_quiz_passing_grade;
		$data->meta->quizpress_quiz_max_attempts_allowed = (int) $data->meta->quizpress_quiz_max_attempts_allowed;
		$data->meta->quizpress_quiz_max_questions_for_answer = (int) $data->meta->quizpress_quiz_max_questions_for_answer;
		$data->meta->quizpress_quiz_hide_question_number = (bool) $data->meta->quizpress_quiz_hide_question_number;
		$data->meta->quizpress_quiz_short_answer_characters_limit = (int) $data->meta->quizpress_quiz_short_answer_characters_limit;
		$data->meta->quizpress_quiz_show_correct_answers = (bool) $data->meta->quizpress_quiz_show_correct_answers;
		$data->meta->quizpress_quiz_allow_review = (bool) $data->meta->quizpress_quiz_allow_review;
		$data->meta->quizpress_quiz_show_score = (bool) $data->meta->quizpress_quiz_show_score;

		$questions = Query::get_quiz_questions_by_quid_id( $quiz_id );

		if ( empty( $questions ) ) {
			return rest_ensure_response( $data );
		}

		foreach ( $questions as &$question ) {
			$question = (object) $this->rest_prepare_item($question);
			$question->answers = \QuizPress\API\Query\Answers::get_question_answers_by_question_id( $question->question_id );
			foreach ( $question->answers as &$answer ) {
				$answer = \QuizPress\API\Controller\Answers::prepare_answer_for_response( $answer );
			}
			unset( $answer );
		}
		unset( $question ); // break reference
		$data->questions = $questions;

		return rest_ensure_response( $data );
	}


	/**
	 * Creates a single post.
	 *
	 * @since 4.7.0
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object on success, or \WP_Error object on failure.
	 */
	public function create_item( $request ) {
		if ( ! empty( $request['id'] ) ) {
			return new \WP_Error(
				'rest_post_exists',
				esc_html__( 'Cannot create existing question.', 'quizpres' ),
				array( 'status' => 400 )
			);
		}

		$prepared_question = $this->prepare_item_for_database( $request );
		$question_id = Query::quiz_question_insert( wp_unslash( (array) $prepared_question ) );
		$question = Query::get_quiz_question( $question_id );
		$response = $this->rest_prepare_item( $question, $request );
		return rest_ensure_response( $response );
	}

	public function update_item( $request ) {
		$params = $request->get_params();
		if ( empty( $params['id'] ) ) {
			return new \WP_Error(
				'rest_question_id_not_exists',
				esc_html__( 'Cannot update existing question.', 'quizpress' ),
				array( 'status' => 400 )
			);
		}
		$prepared_question = $this->prepare_item_for_database( $request );
		$question_id = Query::quiz_question_insert( wp_unslash( (array) $prepared_question ) );
		$question = Query::get_quiz_question( $question_id );
		$response = $this->rest_prepare_item( $question, $request );
		$answers = \QuizPress\API\Query\Answers::get_question_answers_by_question_id( $question_id );
		foreach ( $answers as &$answer ) {
			$answer = \QuizPress\API\Controller\Answers::prepare_answer_for_response( $answer );
		}
		unset( $answer );
		$response['answers'] = $answers;
		return rest_ensure_response( $response );
	}

	public function delete_item( $request ) {
		$question_id = $request->get_param( 'id' );
		if ( ! $request['force'] ) {
			$question = Query::get_quiz_question( $question_id );
			$question->question_status = 'trash';
			$qid = Query::quiz_question_insert( (array) $question );
			if ( ! empty( $qid ) ) {
				return new \WP_REST_Response( [
					'trashed' => true
				], 200 );
			}
			return new \WP_REST_Response( [
				'trashed' => false
			], 500 );
		}
		$is_delete = Query::delete_question( $question_id );
		return new \WP_REST_Response( $is_delete, 200 );
	}


	protected function rest_prepare_item( $question ) {
		$data   = array();
		$schema = $this->get_public_item_schema();

		if ( isset( $schema['properties']['question_id'] ) ) {
			$data['question_id'] = (int) ($question->question_id ?? 0);
		}

		if ( isset( $schema['properties']['question_title'] ) ) {
			$data['question_title'] = $question->question_title ?? '';
		}

		if ( isset( $schema['properties']['question_content'] ) ) {
			$data['question_content'] = $question->question_content ?? '';
		}

		if ( isset( $schema['properties']['question_explanation'] ) ) {
			$data['question_explanation'] = $question->question_explanation ?? '';
		}

		if ( isset( $schema['properties']['question_status'] ) ) {
			$data['question_status'] = $question->question_status ?? '';
		}

		if ( isset( $schema['properties']['question_type'] ) ) {
			$data['question_type'] = $question->question_type ?? '';
		}

		if ( isset( $schema['properties']['answer_type'] ) ) {
			$data['answer_type'] = $question->answer_type ?? null;
		}

		if ( isset( $schema['properties']['question_name'] ) ) {
			$data['question_name'] = $question->question_name ?? '';
		}

		if ( isset( $schema['properties']['question_level'] ) ) {
			$data['question_level'] = $question->question_level ?? '';
		}

		if ( isset( $schema['properties']['question_order'] ) ) {
			$data['question_order'] = (int) ($question->question_order ?? 0);
		}

		if ( isset( $schema['properties']['question_score'] ) ) {
			$data['question_score'] = (float) ($question->question_score ?? 0);
		}

		if ( isset( $schema['properties']['question_negative_score'] ) ) {
			$data['question_negative_score'] = (float) ($question->question_negative_score ?? 0);
		}

		if ( isset( $schema['properties']['question_settings'] ) ) {
			$data['question_settings'] = json_decode( $question->question_settings ?? '{}', true );
		}

		if ( isset( $schema['properties']['question_created_at'] ) ) {
			$data['question_created_at'] = $question->question_created_at ?? '';
		}

		if ( isset( $schema['properties']['question_updated_at'] ) ) {
			$data['question_updated_at'] = $question->question_updated_at ?? '';
		}

		return $data;
	}

	protected function prepare_item_for_database( $request ) {
		$prepared_question  = new \stdClass();

		$schema = $this->get_item_schema();

		// Question Id.
		if ( ! empty( $schema['question_id'] ) && isset( $request['question_id'] ) ) {
			if ( is_numeric( $request['question_id'] ) ) {
				$prepared_question->question_id = $request['question_id'];
			}
		}

		// Question title.
		if ( ! empty( $schema['question_name'] ) && isset( $request['question_title'] ) ) {
			if ( is_string( $request['question_title'] ) ) {
				$prepared_question->question_title = $request['question_title'];
			}
		}

		// Question Content.
		if ( ! empty( $schema['question_content'] ) && isset( $request['question_content'] ) ) {
			if ( is_string( $request['question_content'] ) ) {
				$prepared_question->question_content = $request['question_content'];
			}
		}

		// Question Explanation.
		if ( ! empty( $schema['question_explanation'] ) && isset( $request['question_explanation'] ) ) {
			if ( is_string( $request['question_explanation'] ) ) {
				$prepared_question->question_explanation = $request['question_explanation'];
			}
		}

		// Question lavel.
		if ( ! empty( $schema['question_level'] ) && isset( $request['question_level'] ) ) {
			if ( is_string( $request['question_level'] ) ) {
				$prepared_question->question_level = $request['question_level'];
			}
		}

		// Question Type.
		if ( ! empty( $schema['question_type'] ) && isset( $request['question_type'] ) ) {
			if ( is_string( $request['question_type'] ) ) {
				$prepared_question->question_type = $request['question_type'];
			}
		}

		// Answer Type.
		if ( ! empty( $schema['answer_type'] ) && isset( $request['answer_type'] ) ) {
			if ( is_string( $request['answer_type'] ) ) {
				$prepared_question->answer_type = $request['answer_type'];
			}
		}

		// Question order.
		if ( ! empty( $schema['question_order'] ) && isset( $request['question_order'] ) ) {
			if ( is_numeric( $request['question_order'] ) ) {
				$prepared_question->question_order = $request['question_order'];
			}
		}

		// Question Score.
		if ( ! empty( $schema['question_score'] ) && isset( $request['question_score'] ) ) {
			if ( is_numeric( $request['question_score'] ) ) {
				$prepared_question->question_score = $request['question_score'];
			}
		}

		// Question Negative Score.
		if ( ! empty( $schema['question_negative_score'] ) && isset( $request['question_negative_score'] ) ) {
			if ( is_numeric( $request['question_negative_score'] ) ) {
				$prepared_question->question_negative_score = $request['question_negative_score'];
			}
		}

		// Question Negative Score.
		if ( ! empty( $schema['question_status'] ) && isset( $request['question_status'] ) ) {
			$prepared_question->question_status = $request['question_status'];
		}

		// Question Settings.
		if ( ! empty( $schema['question_settings'] ) && isset( $request['question_settings'] ) ) {
			if ( is_array( $request['question_settings'] ) ) {
				$prepared_question->question_settings = wp_json_encode( $request['question_settings'] );
			}
		}

		return apply_filters( 'quizpress/api/rest_pre_insert_quiz_question', $prepared_question, $request );
	}

	protected function rest_prepare_for_collection( $response ) {
		if ( ! ( $response instanceof \WP_REST_Response ) ) {
			return $response;
		}

		$data  = (array) $response->get_data();
		$server = rest_get_server();
		if ( method_exists( $server, 'get_compact_response_links' ) ) {
			$links = call_user_func( array( $server, 'get_compact_response_links' ), $response );
		} else {
			$links = call_user_func( array( $server, 'get_response_links' ), $response );
		}

		if ( ! empty( $links ) ) {
			$data['_links'] = $links;
		}

		return $data;
	}

}
