import React, { useEffect, useState } from 'react';
import PropTypes from 'prop-types';
import { __ } from '@wordpress/i18n';
import './styles.scss';
import { API, plugin_root_url, runUploader } from '@QPUtils/helper';
import { Button, Span, Text } from '@chakra-ui/react';
import Preloader from '@QPComponents/Loader/Preloader';

const propTypes = {
	name: PropTypes.string,
	value: PropTypes.number,
	saveValueHandler: PropTypes.func,
	imageSize: PropTypes.string,
	supportedFile: PropTypes.string,
};

export default function Attachment( {
	name = '',
	value = 0,
	saveValueHandler = () => {},
	imageSize = '700x430 pixels',
	supportedFile = '.jpg, .jpeg, .gif, .png, .pdf, or .docx',
} ) {
	const [ attachment, setAttachment ] = useState( {} );
	const [ isLoading, setLoading ] = useState( false );

	useEffect( () => {
		if ( value ) {
			setLoading( true );
			API.get( `wp/v2/media/${ value }` ).then( ( res ) => {
				const sourceURLArray = res.data?.source_url?.split( '.' );
				setAttachment( {
					url: res.data?.source_url,
					media_type: res.data?.media_type,
					title: res.data?.slug + '.' + sourceURLArray[ 1 ],
				} );
				setLoading( false );
			} );
		}
	}, [ value ] );

	const attachmentUploadHandler = ( event ) => {
		event.preventDefault();
		runUploader( {
			library: {
				type: 'application/pdf',
			},
			uploader: {
				type: 'application/pdf',
			},
		} ).then( ( res ) => {
			setAttachment( {
				url: res?.url,
				media_type: res?.type,
				title: res?.filename,
			} );
			saveValueHandler( name, res.id );
		} );
	};

	const removeAttachment = () => {
		saveValueHandler( name, 0 );
		setAttachment( {} );
	};

	const defaultImageURL =
		plugin_root_url + 'assets/images/primary-upload.svg';
	let uploadedAttachment = attachment.url;

	if (
		attachment?.media_type === 'file' ||
		attachment?.media_type === 'application'
	) {
		uploadedAttachment = plugin_root_url + 'assets/images/importFile.svg';
	}

	const getAttachmentClass = () => {
		const fileExtension = attachment?.title?.split( '.' ).pop();
		if ( fileExtension === 'pdf' || fileExtension === 'docx' ) {
			return 'quizpress-upload-attachment--left__uploaded__small';
		}
		return 'quizpress-upload-attachment--left__uploaded__large';
	};

	return (
		<React.Fragment>
			<div className="quizpress-upload-attachment">
				<div className="quizpress-upload-attachment--left">
					{ isLoading ? (
						<Preloader />
					) : (
						<>
							{ uploadedAttachment ? (
								<div className={ getAttachmentClass() }>
									<img
										src={ uploadedAttachment }
										alt="image"
										aria-hidden="true"
									/>
								</div>
							) : (
								<div className="quizpress-upload-attachment--left__linear">
									<div className="quizpress-upload-attachment--left__linear--img">
										<img
											src={ defaultImageURL }
											alt="image"
											aria-hidden="true"
										/>
									</div>
								</div>
							) }
						</>
					) }
				</div>
				<div className="quizpress-upload-attachment--right">
					<div className="quizpress-upload-attachment--right--items">
						{ attachment?.title ? (
							<h3
								className="quizpress-title"
								dangerouslySetInnerHTML={ {
									__html: attachment.title,
								} }
							></h3>
						) : (
							<Text className="quizpress-upload-label">
								<Span fontSize="0.9rem" fontWeight="500">
									{ __( 'Upload File', 'quizpress' ) }
								</Span>
								<br />
								<Span>
									{ __( 'File Support:', 'quizpress' ) }
								</Span>
								{ supportedFile }
							</Text>
						) }
						<p className="quizpress-upload-label">
							<strong>{ __( 'Size:', 'quizpress' ) }</strong>
							{ imageSize }
						</p>
						<Button
							variant="outline"
							onClick={ attachmentUploadHandler }
							size="sm"
							fontSize="0.625rem"
							padding="2px 10px"
						>
							{ Object.keys( attachment ).length > 0
								? __( 'Replace', 'quizpress' )
								: __( 'Upload', 'quizpress' ) }
						</Button>
					</div>

					{ /* {uploadedAttachment && (
							<Button
							variant='outline'
							onClick={attachmentUploadHandler}
							fontSize='0.625rem'
							padding='2px 10px'
							size="sm"
						>
							{
								Object.keys(attachment).length > 0
									? __('Replace', 'quizpress')
									: __('Upload', 'quizpress')
							}
						</Button>
					)} */ }
				</div>
			</div>
		</React.Fragment>
	);
}

Attachment.propTypes = propTypes;
