import React, { useState, useEffect } from 'react';
import { Box, Button, Flex, Text } from '@chakra-ui/react';
import { __ } from '@wordpress/i18n';
import { sortableContainer, sortableElement } from 'react-sortable-hoc';
import { arrayMoveImmutable } from 'array-move';
import ReactModal from '@QPComponents/Modal/ReactModal';

const SortableColumnItem = sortableElement(
	( { copyColumn, itemIndex, checkedChange } ) => {
		return (
			<div
				className="quizpress-table__filter-checked-item"
				key={ `column-${ itemIndex }` }
			>
				<span className="quizpress-table-filter-item-left">
					<span className="quizpress-icon quizpress-icon--move" />
					<span className="quizpress-table-filter-item-title">
						{ copyColumn.name }
					</span>
				</span>
				<input
					id={ copyColumn.name }
					type="checkbox"
					checked={ copyColumn.visible }
					onChange={ ( event ) =>
						checkedChange( {
							id: copyColumn.id,
							visible: event.target.checked,
						} )
					}
				/>
			</div>
		);
	}
);

const SortableColumnItemContainer = sortableContainer( ( props ) => {
	return (
		<ul className="quizpress-table__filter-items">{ props.children }</ul>
	);
} );

const SortableColumns = ( {
	setTempCopyColumns,
	tempCopyColumns,
	showColumnFilter,
	checkedChange,
	setCopyColumns,
	copyColumns,
	suffix,
} ) => {
	const [ modalOpen, isModalOpen ] = useState( false );
	const [ defaultColumns, setDefaultColumns ] = useState( [] );

	useEffect( () => {
		setDefaultColumns( [ ...tempCopyColumns ] );
	}, [] );

	const onRequestClose = () => {
		isModalOpen( false );
	};

	const onSortEnd = ( { oldIndex, newIndex } ) => {
		const sortedColumn = arrayMoveImmutable(
			tempCopyColumns,
			oldIndex,
			newIndex
		);
		setTempCopyColumns( sortedColumn );
	};

	const handleToggle = () => {
		isModalOpen( ! modalOpen );
	};

	const handleReset = () => {
		setTempCopyColumns( [ ...defaultColumns ] );
	};

	return (
		<React.Fragment>
			<div className="quizpress-table__sub-header-filter-icon">
				<Button
					color="var(--quizpress-font-color)"
					borderColor="var(--quizpress-border-color)"
					borderWidth="1px"
					bg="transparent"
					onClick={ handleToggle }
					_hover={ {
						bg: 'var(--quizpress-body-background)',
					} }
				>
					<span className="quizpress-icon quizpress-icon--columns" />
					{ __( 'Columns', 'quizpress' ) }
				</Button>
			</div>

			<ReactModal
				isOpen={ modalOpen }
				onRequestClose={ onRequestClose }
				title={ __( 'Columns', 'quizpress' ) }
				suffix="email"
				size="medium"
			>
				<Box
					minWidth="520px"
					p="16px"
					className="quizpress-table__filter-modal"
				>
					<Flex
						justifyContent="space-between"
						alignItems="center"
						gap="2"
					>
						<Text
							color="var(--quizpress-font-color)"
							fontSize="sm"
							fontWeight="medium"
						>
							{ __( 'Columns', 'quizpress' ) }
						</Text>

						<Button
							onClick={ handleReset }
							bg="transparent"
							padding="0"
							height="auto"
							color="var(--quizpress-font-color)"
						>
							{ __( 'Reset', 'quizpress' ) }
						</Button>
					</Flex>

					{ showColumnFilter && (
						<SortableColumnItemContainer onSortEnd={ onSortEnd }>
							{ tempCopyColumns?.map( ( item, index ) => (
								<SortableColumnItem
									key={ `column-${ index }` }
									index={ index }
									copyColumn={ item }
									itemIndex={ index }
									checkedChange={ checkedChange }
								/>
							) ) }
						</SortableColumnItemContainer>
					) }

					<Flex justifyContent="flex-end" gap="2">
						<Button
							color="var(--quizpress-font-color)"
							borderColor="var(--quizpress-border-color)"
							borderWidth="1px"
							bg="transparent"
							_hover={ {
								bg: 'var(--quizpress-body-background)',
							} }
							onClick={ () => {
								setTempCopyColumns( [ ...copyColumns ] );
								isModalOpen( false );
							} }
							type="button"
						>
							{ __( 'Cancel', 'quizpress' ) }
						</Button>

						<Button
							bg="var(--quizpress-primary)"
							onClick={ () => {
								setCopyColumns( tempCopyColumns );
								localStorage.setItem(
									suffix,
									JSON.stringify( tempCopyColumns )
								);
								isModalOpen( false );
							} }
						>
							{ __( 'Apply', 'quizpress' ) }
						</Button>
					</Flex>
				</Box>
			</ReactModal>
		</React.Fragment>
	);
};

export default SortableColumns;
