import React, { useEffect, useRef, useState } from 'react';
import PropTypes from 'prop-types';
import { __ } from '@wordpress/i18n';
import { createPortal } from 'react-dom';
import './styles.scss';

const propTypes = {
	children: PropTypes.object,
	value: PropTypes.string,
	options: PropTypes.object,
	onChangeHandler: PropTypes.func,
};

const StatusOptions = ( props ) => {
	const {
		value = 'pending',
		options = {},
		onChangeHandler = () => {},
		suffix = '',
	} = props;

	const [ isOpenDropdown, setOpenDropdown ] = useState( false );

	const statusLabel = {
		publish: __( 'Published', 'quizpress' ),
		pending: __( 'Pending', 'quizpress' ),
		draft: __( 'Draft', 'quizpress' ),
		trash: __( 'Trash', 'quizpress' ),
		completed: __( 'Approved', 'quizpress' ),
		approved: __( 'Approve', 'quizpress' ),
		processing: __( 'Processing', 'quizpress' ),
		cancel: __( 'Cancel', 'quizpress' ),
		private: __( 'Private', 'quizpress' ),
		future: __( 'Scheduled', 'quizpress' ),
	};

	const menuItemRef = useRef( null );
	const relativeTo = useRef( null );

	const handleClick = ( e ) => {
		if (
			menuItemRef?.current &&
			! menuItemRef?.current?.contains( e.target )
		) {
			setOpenDropdown( false );
		}
	};

	const handleMenuToggle = () => {
		setOpenDropdown( ! isOpenDropdown );
	};

	useEffect( () => {
		document.addEventListener( 'mousedown', handleClick );
		return () => document.removeEventListener( 'mousedown', handleClick );
	}, [] );

	useEffect( () => {
		if ( isOpenDropdown && relativeTo.current ) {
			const rect = relativeTo.current.getBoundingClientRect();
			const x = rect.left + window.pageXOffset;
			const y = rect.top + window.pageYOffset;
			const buttonHeight = relativeTo.current.offsetHeight;
			menuItemRef.current.style.position = 'absolute';
			menuItemRef.current.style.left = `${ x - 50 }px`;
			menuItemRef.current.style.top = `${ y + buttonHeight + 5 }px`;
			document.body.appendChild( menuItemRef.current );
		} else if (
			relativeTo.current &&
			relativeTo.current.parentNode === document.body
		) {
			document.body.removeChild( menuItemRef.current );
		}
	}, [ isOpenDropdown, relativeTo ] );

	return (
		<>
			<button
				className={ `quizpress-dropdown-option quizpress-dropdown-option--${ value } 
				${ suffix && 'quizpress-dropdown-option--' + suffix }` }
				type="button"
				ref={ relativeTo }
				onClick={ handleMenuToggle }
			>
				<span className="label">{ statusLabel[ value ] }</span>
				<span
					className={ `quizpress-icon quizpress-icon--arrow-down` }
				/>
			</button>
			{ createPortal(
				<div
					className="quizpress-dropdown-option__lists"
					ref={ menuItemRef }
				>
					{ isOpenDropdown && (
						<ul className="quizpress-more-status-options">
							{ options.items.map( ( item, index ) => (
								<li
									className={ `quizpress-more-status-options__item quizpress-more-status-options__item--${
										item.value
									} ${ item.value === value && 'active' }` }
									role="presentation"
									onClick={ () => {
										onChangeHandler( item.value );
										setOpenDropdown( false );
									} }
									key={ index }
								>
									{ item.label }
								</li>
							) ) }
						</ul>
					) }
				</div>,
				document.body
			) }
		</>
	);
};

StatusOptions.propTypes = propTypes;
export default StatusOptions;
