import React, { useState } from 'react';
import { __ } from '@wordpress/i18n';
import { Button, Flex, Input, Span, Text, Textarea } from '@chakra-ui/react';
import { Field, useFormikContext } from 'formik';
import QuestionSettings from './QuestionSettings';
import { questionTypeOptions } from '@QPUtils/helper';
import Select from 'react-select';
import SingleChoiceText from '@QPComponents/RenderAnswers/SingleChoiceText';
import SingleChoiceRich from '@QPComponents/RenderAnswers/SingleChoiceRich';
import ImageAnswer from '@QPComponents/RenderAnswers/ImageAnswer';
import ImageTextAnswer from '@QPComponents/RenderAnswers/ImageTextAnswer';
import OptIn from '@QPComponents/RenderAnswers/OptIn';
import TrueFalse from '@QPComponents/RenderAnswers/TrueFalse';
import ReadOnlyBlank from '@QPComponents/RenderAnswers/ReadOnlyBlank';

const FormInner = () => {
	const { values, setFieldValue } = useFormikContext();
	const [ showDescription, isShowDescription ] = useState( false );
	const [ showExplanation, isShowExplanation ] = useState( false );

	const questionType = values?.question_type;
	const answerType = values?.answer_type;

	const answerTypeOptions = [
		...( values?.question_type === 'fill_in_the_blanks'
			? [
					{
						label: __( 'ReadOnly', 'quizpress' ),
						value: 'read_only',
					},
					{
						label: __( 'Sequent (coming soon)', 'quizpress' ),
						value: 'sequent',
						isDisabled: true,
					},
			  ]
			: values?.question_type === 'drop_down'
			? [
					{
						label: __( 'Plain Text', 'quizpress' ),
						value: 'plain_text',
					},
			  ]
			: [
					{
						label: __( 'Plain Text', 'quizpress' ),
						value: 'plain_text',
					},
					{
						label: __( 'Rich Text', 'quizpress' ),
						value: 'rich_text',
					},
					{ label: __( 'Image', 'quizpress' ), value: 'image' },
					{
						label: __( 'Image & Text', 'quizpress' ),
						value: 'image_and_text',
					},
			  ] ),
	];

	const category = 'World cup quiz'; //todo: remove this after redux and category implementation

	const renderSwitchQuestionType = ( questionType ) => {
		switch ( questionType ) {
			case 'single_choice':
			case 'multiple_choice':
			case 'multiple_choice_horizontal':
			case 'single_choice_horizontal':
			case 'drop_down':
				if ( answerType === 'plain_text' ) {
					return (
						<SingleChoiceText
							ansName="answers"
							ansVal={ values?.answers ?? [] }
						/>
					);
				}
				if ( answerType === 'rich_text' ) {
					return (
						<SingleChoiceRich
							ansName="answers"
							ansVal={ values?.answers ?? [] }
						/>
					);
				}
				if ( answerType === 'image' ) {
					return (
						<ImageAnswer
							ansName="answers"
							ansVal={ values?.answers ?? [] }
						/>
					);
				}
				if ( answerType === 'image_and_text' ) {
					return (
						<ImageTextAnswer
							ansName="answers"
							ansVal={ values?.answers ?? [] }
						/>
					);
				}
				break;
			case 'opt_in':
			case 'date':
			case 'paragraph':
			case 'short_answer':
			case 'number':
			case 'file_upload':
				return (
					<OptIn
						type={ questionType }
						ansName="answers"
						ansVal={ values?.answers ?? [] }
					/>
				);
			case 'fill_in_the_blanks':
				if ( answerType === 'read_only' ) {
					return (
						<ReadOnlyBlank
							ansName="answers"
							ansVal={ values?.answers ?? [] }
						/>
					);
				}
				if ( answerType === 'sequent' ) {
					return (
						<SequentBlanks
							ansName="answers"
							ansVal={ values?.answers ?? [] }
						/>
					);
				}
				break;
			case 'true_false':
				return (
					<TrueFalse
						ansName="answers"
						ansVal={ values?.answers ?? [] }
						qstVal={ values }
					/>
				);
			// case 'checkbox_grid':
			// case 'radio_grid':
			// 	return <GridRowColumn ansName="answers" ansVal={values?.answers ?? []} />;
		}
	};

	const isShowAnsOptions =
		questionType === 'single_choice' ||
		questionType === 'multiple_choice' ||
		questionType === 'multiple_choice_horizontal' ||
		questionType === 'single_choice_horizontal' ||
		questionType === 'drop_down' ||
		questionType === 'file_upload' ||
		questionType === 'fill_in_the_blanks';

	const currentAnsType =
		answerTypeOptions.find( ( item ) => item.value === answerType ) ?? '';

	const currentQstType = questionTypeOptions
		.flatMap( ( item ) => item.options )
		.find( ( item ) => item.value === questionType );

	return (
		<Flex direction="column" w="full" className="quizpress-page-content">
			{ /* Question Details */ }
			<Flex
				flexDirection="column"
				bg="var(--quizpress-background)"
				boxShadow="var(--quizpress-shadow)"
				borderRadius="4px"
			>
				<Flex
					width="full"
					gap={ 6 }
					alignItems="stretch"
					padding="24px 12px 0 24px"
				>
					<Flex
						borderRight="1px solid var(--quizpress-border-color)"
						gap={ 3 }
						direction="column"
						width="65%"
						padding="0 24px 24px 0"
					>
						<Flex direction="column" gap="12px">
							<Flex direction="column" gap="8px">
								<Text
									as="label"
									className="quizpress-title quizpress-title-required"
								>
									{ __( 'QuestionTitle', 'quizpress' ) }
									<Span className="quizpress-title-required--sign">
										*
									</Span>
								</Text>
								<Field
									as={ Input }
									name={ `question_title` }
									placeholder={ __(
										'Enter title',
										'quizpress'
									) }
									defaultValue=""
									className="quizpress-input"
								/>
							</Flex>
							{ ! showDescription ? (
								<Text
									cursor="pointer"
									onClick={ () => isShowDescription( true ) }
									color="blue.500"
									p={ 0 }
									fontWeight="500"
									fontSize="0.875rem"
									justifyContent="left"
									margin={ 0 }
								>
									{ __( '+ Add Description', 'quizpress' ) }
								</Text>
							) : (
								<Flex direction="column" gap="8px">
									<Flex justifyContent="space-between">
										<Text className="quizpress-title">
											{ __( 'Description', 'quizpress' ) }
										</Text>
										<Button
											onClick={ () =>
												isShowDescription( false )
											}
											height="16px"
											variant="ghost"
										>
											{ __( 'close', 'quizpress' ) }
										</Button>
									</Flex>
									<Field
										as={ Textarea }
										className="quizpress-textarea"
										name={ `question_content` }
										placeholder={ __(
											'Enter description',
											'quizpress'
										) }
										rows={ 4 }
									/>
								</Flex>
							) }
							{ ! showExplanation ? (
								<Text
									cursor="pointer"
									onClick={ () => isShowExplanation( true ) }
									color="blue.500"
									p={ 0 }
									fontWeight="500"
									fontSize="0.875rem"
									justifyContent="left"
									margin={ 0 }
								>
									{ __( '+ Add Explanation', 'quizpress' ) }
								</Text>
							) : (
								<Flex direction="column" gap="8px">
									<Flex justifyContent="space-between">
										<Text className="quizpress-title">
											{ __( 'Explanation', 'quizpress' ) }
										</Text>
										<Button
											onClick={ () =>
												isShowExplanation( false )
											}
											height="16px"
											variant="ghost"
										>
											{ __( 'close', 'quizpress' ) }
										</Button>
									</Flex>
									<Field
										as={ Textarea }
										className="quizpress-textarea"
										name="question_explanation"
										placeholder={ __(
											'Enter explanation',
											'quizpress'
										) }
										rows={ 4 }
									/>
								</Flex>
							) }
						</Flex>

						{ /* it will change based on answer type */ }
						<Flex direction="column" gap={ 1 }>
							<Text className="quizpress-title">
								{ __( 'Answers', 'quizpress' ) }
							</Text>
							{ renderSwitchQuestionType( questionType ) }
						</Flex>
					</Flex>

					{ /* Answer type, question type and category */ }
					<Flex
						flexDirection="column"
						width={ '35%' }
						gap={ 4 }
						marginBottom="16px"
					>
						<Flex
							border="1px solid var(--quizpress-border-color)"
							direction="column"
							borderRadius="4px"
							p={ 6 }
						>
							<Flex direction="column" gap={ 3 } width="100%">
								{ /* Question type */ }
								<Flex
									as="label"
									direction="column"
									gap={ 2 }
									width="100%"
								>
									<Text className="quizpress-title">
										{ __( 'Question type', 'quizpress' ) }
									</Text>
									<Field
										as={ Select }
										name={ `question_type` }
										placeholder={ __(
											'Select question type',
											'quizpress'
										) }
										className="quizpress-select"
										classNamePrefix="quizpress-select"
										isSearchable={ true }
										isClearable={ true }
										options={ questionTypeOptions }
										value={ currentQstType }
										onChange={ ( option ) => {
											setFieldValue(
												`question_type`,
												option.value
											);
											if ( ! isShowAnsOptions ) {
												setFieldValue(
													`answer_type`,
													''
												);
											}
										} }
									/>
								</Flex>
								{ /* Answer type */ }
								{ isShowAnsOptions && (
									<Flex
										as="label"
										direction="column"
										gap={ 2 }
										width="100%"
									>
										<Text className="quizpress-title">
											{ __(
												'Answer Option',
												'quizpress'
											) }
										</Text>
										<Select
											name={ `answer_type` }
											placeholder={ __(
												'Select answer option',
												'quizpress'
											) }
											className="quizpress-select"
											classNamePrefix="quizpress-select"
											isSearchable={ true }
											isClearable={ true }
											options={ answerTypeOptions }
											value={ currentAnsType }
											onChange={ ( option ) => {
												setFieldValue(
													`answer_type`,
													option.value
												);
											} }
										/>
									</Flex>
								) }

								{ /* todo: Category here */ }
							</Flex>
						</Flex>
						{ /* Question Settings */ }
						<QuestionSettings />
					</Flex>
				</Flex>
			</Flex>
		</Flex>
	);
};

export default FormInner;
