import React, { useEffect, useState } from 'react';
import { __, sprintf } from '@wordpress/i18n';
import TopBar from '@QPComponents/TopBar';
import { useDispatch, useSelector } from 'react-redux';
import { formatDateTime, route_path, useQuery } from '@QPUtils/helper';
import { fetchAttemptById } from '@QPRedux/Slices/quizInsightsSlice/quizInsightsSlice';
import { Flex, Span } from '@chakra-ui/react';
import QPLabel from '@QPComponents/Labels/QPLabel';
import ListTable from '@QPComponents/ListTable';
import { Heading } from '@chakra-ui/react';
import AttemptAnswerDetails from './AttemptAnswerDetails';
import InstructorFeedback from './InstructorFeedback';

const AttemptDetails = () => {
	const attempts = useSelector( ( state ) => state.quizInsights );

	const [ dataFetchingStatus, setDataFetchingStatus ] = useState( false );

	const dispatch = useDispatch();

	const location = useQuery();
	const id = location.get( 'id' );
	const currentAttempt = attempts.data.find(
		( item ) => Number( item.attempt_id ) === Number( id )
	);

	useEffect( () => {
		( async () => {
			if (
				! attempts.data.length ||
				( attempts.data && attempts.data.length <= 1 )
			) {
				setDataFetchingStatus( true );
				dispatch( fetchAttemptById( id ) ).finally( () =>
					setDataFetchingStatus( false )
				);
			}
		} )();
	}, [] );

	const columns = [
		{
			name: __( 'Date', 'quizpress' ),
			cell: ( row ) => {
				return (
					<div>
						<Span>
							{ formatDateTime(
								row.attempt_started_at,
								'MMM DD, YYYY'
							) }
						</Span>
						<br />
						<Span>
							{ formatDateTime(
								row.attempt_started_at,
								'hh:mm A'
							) }
						</Span>
					</div>
				);
			},
		},
		{
			name: __( 'Questions', 'quizpress' ),
			cell: ( row ) => (
				<Flex direction="column" gap={ 2 } alignItems="flex-start">
					<Span>
						{ sprintf(
							// translate: %s total_questions
							__( `Total Questions: %s`, 'quizpress' ),
							row?.total_questions
						) }
					</Span>
					<Span>
						{
							// translate: %s total_answered_questions
							sprintf(
								__( `Attempted Questions: %s`, 'quizpress' ),
								row?.total_answered_questions
							)
						}
					</Span>
				</Flex>
			),
		},
		{
			name: __( 'Marks', 'quizpress' ),
			cell: ( row ) => (
				<Flex direction="column" gap={ 2 } alignItems="flex-start">
					<Span>
						{ sprintf(
							// translate: %s total_marks
							__( `Total Marks: %s`, 'quizpress' ),
							row?.total_marks
						) }
					</Span>
					<Span>
						{ sprintf(
							// translate: %s earned_marks
							__( `Earned marks: %s`, 'quizpress' ),
							row?.earned_marks
						) }
					</Span>
				</Flex>
			),
		},
		{
			name: __( 'Answers', 'quizpress' ),
			cell: ( row ) => (
				<Flex direction="column" gap={ 2 } alignItems="flex-start">
					<Span>
						{ sprintf(
							__(
								// translate: %s total_correct_answers
								`Correct: %s`,
								'quizpress'
							),
							row?.attempt_info?.total_correct_answers
						) }
					</Span>
				</Flex>
			),
		},
		{
			name: __( 'Result', 'quizpress' ),
			cell: ( row ) => (
				<Span
					m={ 0 }
					fontSize="md"
					className={ `quizpress-status quizpress-status--${ row?.attempt_status }` }
				>
					{ sprintf(
						// translate: %s attempt_status
						__( `%s`, 'quizpress' ),
						row?.attempt_status
					) }
				</Span>
			),
		},
	];

	return (
		<>
			<TopBar
				leftContent={ () => (
					<>
						<span className="quizpress-topbar-logo quizpress-icon quizpress-icon--quizpress" />
						<span className="quizpress-icon quizpress-icon--angle-right quizpress-muted" />
						<QPLabel
							as="h2"
							color="#4F46E5"
							type="subtitle"
							fontWeight="medium"
							label={ __( 'Quiz Attempts', 'quizpress' ) }
							href={ `${ route_path }admin.php?page=quizpress-quiz-insights` }
						/>
						<span className="quizpress-icon quizpress-icon--angle-right quizpress-muted" />
						<QPLabel
							as="h2"
							color="#4F46E5"
							type="subtitle"
							fontWeight="medium"
							label={ __( 'Attempt Details', 'quizpress' ) }
						/>
					</>
				) }
			/>
			<Flex
				direction="column"
				gap={ 6 }
				p={ 6 }
				borderRadius="4px"
				className="quizpress-page-content"
				bg="var(--quizpress-background)"
			>
				<Flex direction="column" gap={ 3 }>
					<Heading m={ 0 }>
						{ __( 'Attempt Result', 'quizpress' ) }
					</Heading>

					<ListTable
						key={ currentAttempt }
						columns={ columns }
						data={ [ currentAttempt ] ?? [] }
						showSubHeader={ false }
						subHeaderComponent={ false }
						isRowSelectable={ false }
						showColumnFilter={ false }
						noDataText={ __(
							'Please, complete Quizzes to see the available list here.',
							'quizpress'
						) }
						getSelectRowValue={ ( selectedRow ) => {} }
						showPagination={ false }
						totalItems={ [ currentAttempt ]?.length ?? 0 }
						dataFetchingStatus={ dataFetchingStatus }
						suffix="quiz-insight-details-table"
					/>
				</Flex>
				<AttemptAnswerDetails />
				<InstructorFeedback />
			</Flex>
		</>
	);
};

export default AttemptDetails;
