import React, { useEffect, useState } from 'react';
import { __, sprintf } from '@wordpress/i18n';
import TopBar from '@QPComponents/TopBar';
import { useDispatch, useSelector } from 'react-redux';
import { Flex, Span } from '@chakra-ui/react';
import QPLabel from '@QPComponents/Labels/QPLabel';
import ListTable from '@QPComponents/ListTable';
import { getQuizLeaderboard } from '@QPRedux/Slices/quizSlice/quizSlice';
import { capitalizeFirstLetter, route_path } from '@QPUtils/helper';

export default function QuizLeaderboard( { id } ) {
	const { leaderBoards } = useSelector( ( state ) => state.quizzes );
	const leaderboardData = leaderBoards.find(
		( item ) => Number( item.quiz_id ) === Number( id )
	);
	const [ dataFetchingStatus, setDataFetchingStatus ] = useState( false );

	const dispatch = useDispatch();

	useEffect( () => {
		( async () => {
			if ( ! leaderboardData?.leaderboard.length || ! leaderboardData ) {
				setDataFetchingStatus( true );
				dispatch( getQuizLeaderboard( id ) ).finally( () => {
					setDataFetchingStatus( false );
				} );
			}
		} )();
	}, [] );

	const tableData = leaderboardData?.leaderboard.map( ( item ) => ( {
		...item,
		type: leaderboardData?.quiz_type,
	} ) );

	const columns = [
		{
			name: __( 'Name', 'quizpress' ),
			cell: ( row ) => {
				return <Span>{ row?.display_name }</Span>;
			},
		},
		{
			name: __( 'ID', 'quizpress' ),
			cell: ( row ) => {
				return <Span>{ row?.user_id }</Span>;
			},
		},
		{
			name: __( 'Type', 'quizpress' ),
			cell: ( row ) => {
				return <Span>{ capitalizeFirstLetter( row?.type ) }</Span>;
			},
		},
		{
			name: __( 'Total Attempts', 'quizpress' ),
			cell: ( row ) => {
				return <Span>{ row?.attempts }</Span>;
			},
		},
		{
			name: __( 'Best Score', 'quizpress' ),
			cell: ( row ) => (
				<Span>
					{ sprintf(
						__(
							// translate: %s best_score
							`%s`,
							'quizpress'
						),
						row?.best_score
					) }
				</Span>
			),
		},
	];
	return (
		<>
			<TopBar
				leftContent={ () => (
					<>
						<span className="quizpress-topbar-logo quizpress-icon quizpress-icon--quizpress" />
						<span className="quizpress-icon quizpress-icon--angle-right quizpress-muted" />
						<QPLabel
							as="h2"
							color="#4F46E5"
							type="subtitle"
							fontWeight="medium"
							label={ __( 'Quizzes', 'quizpress' ) }
							href={ `${ route_path }admin.php?page=quizpress-quizzes` }
						/>

						<span className="quizpress-icon quizpress-icon--angle-right quizpress-muted" />
						<QPLabel
							as="h2"
							color="#4F46E5"
							type="subtitle"
							fontWeight="medium"
							label={ __( 'Attempt Details', 'quizpress' ) }
						/>
					</>
				) }
			/>
			<Flex
				direction="column"
				gap={ 6 }
				className="quizpress-page-content"
			>
				<ListTable
					columns={ columns }
					data={ tableData ?? [] }
					showSubHeader={ false }
					subHeaderComponent={ false }
					isRowSelectable={ false }
					showColumnFilter={ false }
					noDataText={ __(
						'No attempts found in this quiz',
						'quizpress'
					) }
					getSelectRowValue={ ( selectedRow ) => {
						// setSelectedItems([]);
						// setResetSelectedItems(false);
					} }
					showPagination={ false }
					totalItems={ tableData?.length ?? 0 }
					// totalRows={0}
					dataFetchingStatus={ dataFetchingStatus }
					suffix="quiz-insight-details-table"
				/>
			</Flex>
		</>
	);
}
