import React, { useCallback, useContext } from 'react';
import { __, sprintf } from '@wordpress/i18n';
import Stepper from '@QPComponents/Formik/Stepper';
import QPLabel from '@QPComponents/Labels/QPLabel';
import TopBar from '@QPComponents/TopBar';
import FormikWizardContext from './Context/FormikWizardContext';
import { route_path } from '@QPUtils/helper';
import { useFormikContext } from 'formik';
import { useNavigate } from 'react-router-dom';
import { Box, Button, Flex, Span } from '@chakra-ui/react';
import {
	outlineBtn,
	primaryBtn,
} from '../../../../../../assets/scss/chakra/recipe';

const QuizEditorTopBar = ( { id, steps } ) => {
	const { currentStepIndex, handlePrev, handleNext } =
		useContext( FormikWizardContext );
	const { values, isSubmitting, handleSubmit, dirty } = useFormikContext();

	const isSaveBtnDisabled =
		isSubmitting ||
		values?.title === '' ||
		( currentStepIndex === 2 && ! dirty );

	const navigate = useNavigate();

	const getNavigationPath = useCallback(
		( stepParam ) => {
			const action = id ? 'edit' : 'add';
			const idParam = id ? `&id=${ id }` : '';
			return `${ route_path }admin.php?page=quizpress-quizzes&action=${ action }${ idParam }&path=${ stepParam }`;
		},
		[ id ]
	);

	const handleNextNavigation = () => {
		let pathParam = '';
		if ( currentStepIndex === 0 ) {
			pathParam = 'questions';
		} else if ( currentStepIndex === 1 ) {
			pathParam = 'settings';
		}

		if ( pathParam ) {
			const newPath = getNavigationPath( pathParam );
			navigate( newPath );
		}

		if ( ( currentStepIndex === 0 && dirty ) || currentStepIndex === 2 ) {
			handleSubmit();
		}
		if ( currentStepIndex !== 2 ) {
			handleNext();
		}
	};

	const handlePrevNavigation = () => {
		let pathParam = '';
		if ( currentStepIndex === 1 ) {
			pathParam = 'name';
		} else if ( currentStepIndex === 2 ) {
			pathParam = 'questions';
		}

		if ( pathParam ) {
			const newPath = getNavigationPath( pathParam );
			navigate( newPath );
		}
		handlePrev();
	};

	const handleStepperClick = ( idx ) => {
		if ( ! id ) {
			return;
		}

		if ( idx < currentStepIndex ) {
			handlePrevNavigation();
		} else if ( idx > currentStepIndex ) {
			handleNextNavigation();
		}
	};

	const handlePublish = () => {
		values.status = 'publish';
		handleSubmit();
	};

	return (
		<TopBar
			topBarStyles={ { height: '73px' } }
			leftContent={ () => (
				<>
					<span className="quizpress-topbar-logo quizpress-icon quizpress-icon--quizpress" />
					<span className="quizpress-icon quizpress-icon--angle-right quizpress-muted" />

					<QPLabel
						as="h2"
						color="#4F46E5"
						type="subtitle"
						fontWeight="medium"
						href={ `${ route_path }admin.php?page=quizpress-quizzes` }
						label={ __( 'Quizzes', 'quizpress' ) }
					/>
					<span className="quizpress-icon quizpress-icon--angle-right quizpress-muted" />
					<QPLabel
						as="h2"
						color="#4F46E5"
						type="subtitle"
						fontWeight="medium"
						label={
							! values.title
								? __( 'New Quiz', 'quizpress' )
								: sprintf(
										// translators: %s: quiz title
										__( '%s', 'quizpress' ),
										values.title
								  )
						}
					/>
				</>
			) }
			middleContent={ () => (
				<Box
					position="absolute"
					left="50%"
					top="50%"
					transform="translateX(-50%) translateY(-50%)"
					transformOrigin="center"
				>
					<Stepper
						steps={ steps }
						currentStepIndex={ currentStepIndex }
						onStepClick={ handleStepperClick }
					/>
				</Box>
			) }
			rightContent={ () => (
				<Flex gap={ 2 } alignItems="center">
					{ currentStepIndex > 0 && (
						<Button
							{ ...outlineBtn }
							bg="var(--quizpress-background)"
							onClick={ handlePrevNavigation }
						>
							<Span className="quizpress-icon quizpress-icon--angle-left quizpress-primary" />
						</Button>
					) }

					<Button
						{ ...primaryBtn }
						onClick={ handleNextNavigation }
						disabled={ isSaveBtnDisabled }
						loading={ isSubmitting }
					>
						{ currentStepIndex === 0 &&
							__( 'Save & Continue', 'quizpress' ) }
						{ currentStepIndex === 1 && __( 'Next', 'quizpress' ) }
						{ currentStepIndex === 2 && __( 'Save', 'quizpress' ) }
						{ currentStepIndex !== 2 && (
							<Span className="quizpress-icon quizpress-icon--angle-right has-quizpress-blue-bg" />
						) }
					</Button>

					{ currentStepIndex === 2 && values.status !== 'publish' && (
						<Button
							{ ...primaryBtn }
							disabled={ values.status === 'publish' }
							onClick={ handlePublish }
						>
							{ __( 'Publish', 'quizpress' ) }
						</Button>
					) }
				</Flex>
			) }
		/>
	);
};

export default QuizEditorTopBar;
