import React, { useEffect, useState } from 'react';
import { __, sprintf } from '@wordpress/i18n';
import { useSelector } from 'react-redux';
import QuizDetailsModal from './QuizDetailsModal';
import { Button, Flex, Text } from '@chakra-ui/react';
import { InstructorFeedBack } from '../helper';
import { makeRequest } from '@QPUtils/helper';

export default function Pending( {
	attempts,
	submittedAttempt,
	type = 'quiz',
} ) {
	const quizzes = useSelector( ( state ) => state.quizzes );
	const [ openQuizDetails, setOpenDetailsModal ] = useState( false );
	const [ feedback, setFeedback ] = useState( '' );

	const currentQuiz = quizzes.data[ 0 ];
	const pendingSubTitle = () => {
		switch ( type ) {
			case 'quiz':
				return 'Your quiz is under manual review';
			case 'survey':
				return 'Your survey is under manual review';
			case 'exams':
				return 'Your exams are under manual review';
			default:
				return 'Your poll is under manual review';
		}
	};

	const isQuiz = type === 'quiz' || type === 'exams';

	const successfulAttempts = attempts.filter(
		( attempt ) => attempt.attempt_status === 'passed'
	);
	const pendingAttempts = attempts.filter(
		( attempt ) => attempt.attempt_status === 'pending'
	);

	useEffect( () => {
		makeRequest( `frontend/get_user_attempt_feedback`, {
			attempt_id: submittedAttempt?.attempt_id,
		} ).then( ( res ) => {
			setFeedback( res );
		} );
	}, [] );

	return (
		<React.Fragment>
			<Flex width="100%" maxW="420px" flexDirection="column" gap={ 6 }>
				<Flex
					flexDirection="column"
					background="#FFF7E9"
					boxShadow="0 0 1px 0 rgba(20, 26, 36, 0.30), 0 2px 4px 0 rgba(20, 26, 36, 0.10)"
					padding={ 6 }
					gap={ 6 }
					borderRadius="4px"
				>
					<Flex
						flexDirection="column"
						alignItems="center"
						paddingTop={ 3 }
					>
						<Text
							color="var(--quizpress-warning)"
							fontWeight="500"
							fontSize="1.5rem"
						>
							{ __( 'Pending', 'quizpress' ) }
						</Text>
						<Text color="#738496" fontSize="0.875rem">
							{ sprintf(
								__( '%s', 'quizpress' ),
								pendingSubTitle()
							) }
						</Text>
					</Flex>
					<Flex flexDirection="column" gap={ 4 } paddingTop={ 6 }>
						<Flex
							alignItems="center"
							justifyContent="space-between"
						>
							<Text fontSize="0.875rem">
								{ __( 'Total Question' ) }
							</Text>
							<Text fontSize="1rem" fontWeight="500">
								{ __(
									currentQuiz?.meta?.quizpress_quiz_questions
										.length,
									'quizpress'
								) }
							</Text>
						</Flex>
						<Flex
							alignItems="center"
							justifyContent="space-between"
						>
							<Text fontSize="0.875rem">
								{ __( 'Quiz Time' ) }
							</Text>
							<Text fontSize="1rem" fontWeight="500">
								{ __(
									currentQuiz?.meta?.quizpress_quiz_time,
									'quizpress'
								) }
							</Text>
						</Flex>
						<Flex
							alignItems="center"
							justifyContent="space-between"
						>
							<Text fontSize="0.875rem">
								{ __( 'Total Attempted' ) }
							</Text>
							<Text fontSize="1rem" fontWeight="500">
								{ __( attempts.length, 'quizpress' ) }
							</Text>
						</Flex>
						{ isQuiz && (
							<>
								<Flex
									alignItems="center"
									justifyContent="space-between"
								>
									<Text fontSize="0.875rem">
										{ __( 'Successful Attempted' ) }
									</Text>
									<Text fontSize="1rem" fontWeight="500">
										{ __(
											successfulAttempts.length,
											'quizpress'
										) }
									</Text>
								</Flex>
								<Flex
									alignItems="center"
									justifyContent="space-between"
								>
									<Text fontSize="0.875rem">
										{ __( 'Pending Attempted' ) }
									</Text>
									<Text fontSize="1rem" fontWeight="500">
										{ __(
											pendingAttempts.length,
											'quizpress'
										) }
									</Text>
								</Flex>
							</>
						) }
					</Flex>
					{ isQuiz && (
						<Button
							bg="var(--quizpress-warning)"
							color="#fff"
							onClick={ () => setOpenDetailsModal( true ) }
						>
							{ __( 'View Details', 'quizpress' ) }
						</Button>
					) }
				</Flex>

				<InstructorFeedBack feedback={ feedback } />
			</Flex>
			{ openQuizDetails && (
				<QuizDetailsModal
					isOpen={ openQuizDetails }
					onRequestClose={ () => setOpenDetailsModal( false ) }
					attempts={ attempts }
					submittedAttempt={ submittedAttempt }
				/>
			) }
		</React.Fragment>
	);
}
