import { useEffect, useState } from 'react';
import { __, sprintf } from '@wordpress/i18n';
import { Flex, Grid, Box, Text, Badge } from '@chakra-ui/react';
import ReactModal from '@QPComponents/Modal/ReactModal';
import { useSelector } from 'react-redux';
import { makeRequest } from '@QPUtils/helper';

import './styles.scss';

const QuizDetailsModal = ( {
	isOpen,
	onRequestClose,
	attempts,
	submittedAttempt,
} ) => {
	const quizzes = useSelector( ( state ) => state.quizzes );
	const currentQuiz = quizzes.data[ 0 ];

	const [ attemptAnswersDetails, setAttemptAnswersDetails ] = useState( [] );

	const renderTime = ( unit ) => {
		switch ( unit ) {
			case 'seconds':
				return 'sec';
			case 'minutes':
				return 'min';
			case 'hours':
				return 'hrs';
			default:
				return 'min';
		}
	};

	useEffect( () => {
		if ( submittedAttempt.attempt_id ) {
			const attempt = attempts.find(
				( attempt ) =>
					attempt.attempt_id === submittedAttempt.attempt_id
			);

			makeRequest( 'frontend/get_user_attempt_answer_details', {
				attempt_id: attempt.attempt_id,
				user_id: attempt.user_id,
			} ).then( ( response ) => {
				setAttemptAnswersDetails( response );
			} );
		}
	}, [ submittedAttempt.attempt_id ] );

	return (
		<ReactModal
			isOpen={ isOpen }
			onRequestClose={ onRequestClose }
			title={ __( 'Attempt Results', 'quizpress' ) }
			isEnabledResizer={ false }
			suffix="quiz-answer-details"
		>
			<Flex flexDirection="column" gap={ 6 }>
				<Flex
					width="100%"
					flexDirection="column"
					gap={ 4 }
					background="#F6F7F8"
					boxShadow=" 0 0 1px 0 rgba(20, 26, 36, 0.20), 0 1px 2px 0 rgba(20, 26, 36, 0.10)"
					padding={ 6 }
				>
					<Text fontSize="1rem" fontWeight="600" margin={ 0 }>
						{ __( 'Attempt Details', 'quizpress' ) }
					</Text>
					<Grid
						templateColumns={ {
							base: 'repeat(2, 1fr)',
							md: 'repeat(4, 1fr)',
						} }
						gap={ 4 }
					>
						<Flex alignItems="center" gap={ 3 }>
							<Text fontSize="0.875rem" fontWeight="400">
								{ __( 'Total Question', 'quizpress' ) }
							</Text>
							<Text fontSize="1rem" fontWeight="600">
								{ __(
									currentQuiz?.meta?.quizpress_quiz_questions
										.length,
									'quizpress'
								) }
							</Text>
						</Flex>

						<Flex
							alignItems="center"
							gap={ 3 }
							justifyContent={ {
								base: 'flex-end',
								lg: 'flex-start',
							} }
						>
							<Text fontSize="0.875rem" fontWeight="400">
								{ __( 'Total Attempted', 'quizpress' ) }
							</Text>
							<Text fontSize="1rem" fontWeight="600">
								{ __( attempts.length, 'quizpress' ) }
							</Text>
						</Flex>

						<Flex alignItems="center" gap={ 3 }>
							<Text fontSize="0.875rem" fontWeight="400">
								{ __( 'Quiz Time', 'quizpress' ) }
							</Text>
							<Text fontSize="1rem" fontWeight="600">
								{ Number(
									currentQuiz?.meta?.quizpress_quiz_time
								) === 0
									? __( 'Unlimited', 'quizpress' )
									: sprintf(
											__( '%s %s', 'quizpress' ),
											currentQuiz?.meta
												?.quizpress_quiz_time,
											renderTime(
												currentQuiz?.meta
													?.quizpress_quiz_time_unit
											)
									  ) }
							</Text>
						</Flex>

						<Flex
							alignItems="center"
							gap={ 3 }
							justifyContent={ {
								base: 'flex-end',
								lg: 'flex-start',
							} }
						>
							<Text fontSize="0.875rem" fontWeight="400">
								{ __( 'Passing Grade', 'quizpress' ) }
							</Text>
							<Text fontSize="1rem" fontWeight="600">
								{ __(
									`${ currentQuiz?.meta?.quizpress_quiz_passing_grade }%`,
									'quizpress'
								) }
							</Text>
						</Flex>
					</Grid>
				</Flex>

				<Box paddingTop={ 6 }>
					<Text
						margin={ 0 }
						fontWeight="400"
						fontSize="1rem"
						paddingBottom={ 3 }
					>
						{ __( 'Attempt Answers Details', 'quizpress' ) }
					</Text>
					<Grid
						templateColumns={ { base: '1fr', lg: '1fr 1fr' } }
						gap={ 6 }
					>
						{ attemptAnswersDetails.map( ( item ) => {
							return (
								<Flex
									boxShadow="0 0 1px 0 rgba(20, 26, 36, 0.30), 0 2px 4px 0 rgba(20, 26, 36, 0.10)"
									background={
										item.is_required_manual_review
											? '#fff7e9'
											: item.is_correct === 1
											? '#F2F7FF'
											: '#FDF1F1'
									}
									padding={ 6 }
									flexDirection="column"
									gap={ 3 }
									borderRadius="4px"
								>
									<Flex
										justifyContent="space-between"
										width="100%"
										alignItems="center"
									>
										<Flex alignItems="center" gap="2px">
											<Text
												fontSize="1rem"
												fontWeight="600"
											>
												{ __( 'Q:', 'quizpress' ) }
											</Text>
											<Text
												fontSize="0.875rem"
												fontWeight="400"
											>
												{ sprintf(
													__( '%s', 'quizpress' ),
													item.question_title
												) }
											</Text>
										</Flex>
										<Badge
											fontWeight="500"
											fontSize="0.75rem"
											padding="3px 12px"
											background={
												item.is_required_manual_review
													? 'var(--quizpress-warning)'
													: item.is_correct === 1
													? '#2ABB7F'
													: '#F15B50'
											}
											color="#FFF"
										>
											{ item.is_required_manual_review
												? 'Pending'
												: item.is_correct === 1
												? 'Correct'
												: 'Incorrect' }
										</Badge>
									</Flex>

									<Flex
										justifyContent="flex-start"
										gap={ 12 }
									>
										<Flex alignItems="center" gap={ 2 }>
											<Text
												fontSize="0.875rem"
												fontWeight="600"
												margin={ 0 }
											>
												{ __(
													'Correct Answer:',
													'quizpress'
												) }
											</Text>
											<Text
												fontSize="0.875rem"
												fontWeight="400"
												margin={ 0 }
											>
												{ item.correct_answer.length > 1
													? item.correct_answer
															.map(
																( a ) =>
																	a.answer_title
															)
															.join( ', ' )
													: item?.correct_answer[ 0 ]
															?.answer_title }
											</Text>
										</Flex>

										<Flex alignItems="center" gap={ 2 }>
											<Text
												fontSize="0.875rem"
												fontWeight="600"
											>
												{ __(
													'Given Answer:',
													'quizpress'
												) }
											</Text>
											<Text
												fontSize="0.875rem"
												fontWeight="400"
											>
												{ item.given_answer &&
												item.given_answer.length > 0
													? item.given_answer
															.length === 1
														? item.given_answer[ 0 ]
																.answer_title
														: item.given_answer
																.map(
																	( a ) =>
																		a.answer_title
																)
																.join( ', ' )
													: 'No answer' }
											</Text>
										</Flex>
									</Flex>

									{ currentQuiz?.meta
										?.quizpress_quiz_explanation_enabled &&
										! Boolean( item.is_correct ) && (
											<Flex alignItems="center" gap="2px">
												<Text
													fontSize="1rem"
													fontWeight="600"
												>
													{ __(
														'Explanation:',
														'quizpress'
													) }
												</Text>
												<Text
													fontSize="0.875rem"
													fontWeight="400"
												>
													{ sprintf(
														__( '%s', 'quizpress' ),
														item.question_explanation
													) }
												</Text>
											</Flex>
										) }
								</Flex>
							);
						} ) }
					</Grid>
				</Box>
			</Flex>
		</ReactModal>
	);
};

export default QuizDetailsModal;
