<?php
/**
 * Readabler
 * Web accessibility for Your WordPress site.
 * Exclusively on https://1.envato.market/readabler
 *
 * @encoding        UTF-8
 * @version         2.0.12
 * @copyright       (C) 2018 - 2024 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 * @license         Envato License https://1.envato.market/KYbje
 **/

namespace Merkulove\Readabler;

use Merkulove\Readabler\Unity\Plugin;
use Merkulove\Readabler\Unity\Settings;
use Merkulove\Readabler\Unity\TabAssignments;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * SINGLETON: Class adds admin styles.
 * @since 1.0.0
 **/
final class FrontStyles {

	/**
	 * The one true FrontStyles.
	 *
	 * @var FrontStyles
	 * @since 1.0.0
	 **/
	private static $instance;

	/**
	 * @var array
	 */
	private static $options;

	/**
	 * Sets up a new FrontStyles instance.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	private function __construct() {

		// Don't load scripts if Readabler AI is enabled.
		if (Tools::is_readabler_ai()) {return;}

		self::$options = Settings::get_instance()->options;

		/** Add plugin styles and keep priority low to prevent other plugins to override them. */
		if ( self::$options['late_load'] === 'on' ) {

			add_action( 'wp_footer', [ $this, 'enqueue_styles' ], 99 );

		} else {

			add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_styles' ], 99 );

		}

	}

	/**
	 * Add plugin styles.
	 *
	 * @return void
	 **@since 1.0.0
	 */
	public function enqueue_styles() {

		$options = Settings::get_instance()->options;

		/** Analyzer script */
		$this->analyzer_styles( $options );

		/** Checks if plugin should work on this page. */
		if ( ! TabAssignments::get_instance()->display() ) {
			return;
		}

		wp_enqueue_style( 'mdp-readabler', Plugin::get_url() . 'css/readabler' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );

		/** Add inline styles. */
		$css = FrontStyles::get_inline_css();

		/** Add custom CSS. */
		wp_add_inline_style( 'mdp-readabler', $css . Settings::get_instance()->options['custom_css'] );

	}

	/**
	 * Create a transparent color from a key color
	 *
	 * @param string $value
	 * @param string $alpha
	 *
	 * @return string
	 */
	public static function getTransparentColor( string $value, string $alpha = '.2' ): string {

		if ( false === strpos( $value, 'rgba' ) ) {
			return $value;
		}

		$color = str_replace( ' ', '', $value );
		sscanf( $color, 'rgba(%d,%d,%d,%f)', $red, $green, $blue, $a );

		return 'rgba(' . $red . ',' . $green . ',' . $blue . ',' . $alpha . ')';

	}

	/**
	 * Return CSS for start button.
	 * @return string
	 */
	public static function get_inline_css(): string {

		/** Shorthand for plugin settings. */
		$options = Settings::get_instance()->options;

		/** Prepare short variables. */
		$background           = $options['popup_background_color'];
		$background_dark      = $options['popup_background_color_dark'];
		$text                 = $options['popup_text_color'];
		$text_dark            = $options['popup_text_color_dark'];
		$key                  = $options['popup_key_color'];
		$key_dark             = $options['popup_key_color_dark'];
		$key_transparent      = self::getTransparentColor( $options['popup_key_color'] );
		$key_transparent_dark = self::getTransparentColor( $options['popup_key_color_dark'] );
		$popup_radius         = $options['popup_border_radius'];
		$duration             = $options['popup_animation_duration'] ?? '600';

		$btn_margin   = $options['button_margin'];
		$btn_padding  = $options['button_padding'];
		$btn_radius   = $options['button_border_radius'];
		$btn_color    = $options['button_color'];
		$btn_bg_color = $options['button_bgcolor'];

		$size               = $options['button_size'];
		$btn_color_hover    = $options['button_color_hover'];
		$btn_bg_color_hover = $options['button_bgcolor_hover'];
		$delay              = $options['button_entrance_timeout'];

		$tts_color = array_key_exists( 'text_to_speech_icon_color', $options ) ? $options['text_to_speech_icon_color'] : 'transparent';
		$tts_bg    = array_key_exists( 'text_to_speech_bg_color', $options ) ? $options['text_to_speech_bg_color'] : 'transparent';

		$root_css = self::root_css();

		// language=CSS
		/** @noinspection CssUnusedSymbol */
		$css = "
		:root{
			$root_css
		}
		.mdp-readabler-tts {
		
			--readabler-tts-bg: $tts_bg;
			--readabler-tts-color: $tts_color;
			
		}
        .mdp-readabler-trigger-button-box{
        
            --readabler-btn-margin: {$btn_margin}px;
            --readabler-btn-padding: {$btn_padding}px;
            --readabler-btn-radius: {$btn_radius}px;
            --readabler-btn-color: $btn_color;
            --readabler-btn-color-hover: $btn_color_hover;
            --readabler-btn-bg: $btn_bg_color;
            --readabler-btn-bg-hover: $btn_bg_color_hover;
            --readabler-btn-size: {$size}px;
            --readabler-btn-delay: {$delay}s;
        
        }
		#mdp-readabler-voice-navigation{
			--readabler-bg: $background;
            --readabler-bg-dark: $background_dark;
            --readabler-text: $text;
            --readabler-text-dark: $text_dark;
            --readabler-color: $key;
            --readabler-color-dark: $key_dark;
            --readabler-color-transparent: $key_transparent;
            --readabler-color-transparent-dark: $key_transparent_dark;
            --readabler-border-radius: {$popup_radius}px;
            --readabler-animate: {$duration}ms;
		}
		";

		/** If enabled Virtual Keyboard. */
		if (
			'on' === $options['virtual_keyboard'] ||
			'on' === $options['profile_blind_users']
		) {

			$kb_bg_color          = ($options['virtual_keyboard_bg_color'] ?? '');
			$kb_btn_bg_color      = ($options['virtual_keyboard_button_bg_color'] ?? '');
			$kb_btn_color         = ($options['virtual_keyboard_button_color'] ?? '');
			$kb_bg_color_dark     = ($options['virtual_keyboard_bg_color_dark'] ?? '');
			$kb_btn_bg_color_dark = ($options['virtual_keyboard_button_bg_color_dark'] ?? '');
			$kb_btn_color_dark    = ($options['virtual_keyboard_button_color_dark'] ?? '');

			// language=CSS
			/** @noinspection CssUnusedSymbol */
			$css .= "
	        #mdp-readabler-keyboard-box {
	        
	        	--readabler-keyboard-light-bg: $kb_bg_color;
	        	--readabler-keyboard-light-key-bg: $kb_btn_bg_color;
	        	--readabler-keyboard-light-key: $kb_btn_color;	        	
	        	--readabler-keyboard-dark-bg: $kb_bg_color_dark;
	        	--readabler-keyboard-dark-key-bg: $kb_btn_bg_color_dark;
	        	--readabler-keyboard-dark-key: $kb_btn_color_dark;
	        
	        }	        
	        ";

		}

		return $css;

	}

	/**
	 * Conditioning CSS for a root element
	 * @return string
	 */
	public static function root_css(): string {

		$options = Settings::get_instance()->options;

		$root_vars = [];

		$root_vars[] = '--readabler-overlay: ' . $options['popup_overlay_color'];
		$root_vars[] = '--readabler-bg: ' . $options['popup_background_color'];
		$root_vars[] = '--readabler-bg-dark: ' . $options['popup_background_color_dark'];
		$root_vars[] = '--readabler-text: ' . $options['popup_text_color'];
		$root_vars[] = '--readabler-text-dark: ' . $options['popup_text_color_dark'];
		$root_vars[] = '--readabler-color: ' . $options['popup_key_color'];
		$root_vars[] = '--readabler-color-dark: ' . $options['popup_key_color_dark'];
		$root_vars[] = '--readabler-color-transparent: ' . self::getTransparentColor( $options['popup_key_color'] );
		$root_vars[] = '--readabler-color-transparent-dark: ' . self::getTransparentColor( $options['popup_key_color_dark'] );

		$root_vars[] = '--readabler-color-transparent-15: ' . self::getTransparentColor( $options['popup_key_color'], '.15' );
		$root_vars[] = '--readabler-color-transparent-15-dark: ' . self::getTransparentColor( $options['popup_key_color_dark'], '.15' );

		$root_vars[] = '--readabler-color-transparent-25: ' . self::getTransparentColor( $options['popup_key_color'], '.25' );
		$root_vars[] = '--readabler-color-transparent-25-dark: ' . self::getTransparentColor( $options['popup_key_color_dark'], '.25' );

		$root_vars[] = '--readabler-color-transparent-50: ' . self::getTransparentColor( $options['popup_key_color'], '.5' );
		$root_vars[] = '--readabler-color-transparent-50-dark: ' . self::getTransparentColor( $options['popup_key_color_dark'], '.5' );

		$root_vars[] = '--readabler-border-radius: ' . $options['popup_border_radius'] . 'px';
		$root_vars[] = '--readabler-animate: ' . ( $options['popup_animation_duration'] ?? '600' ) . 'ms';

		if ( isset( $options['reading_mask'] ) && 'on' === $options['reading_mask'] ) {
			$root_vars[] = '--readabler-reading-mask-color: ' . ($options['reading_mask_color'] ?? 'rgba(0, 0, 0, 0.7)');
		}

		if ( isset( $options['text_magnifier'] ) && 'on' === $options['text_magnifier'] ) {
			$root_vars[] = '--readabler-text-magnifier-bg-color: ' . ($options['text_magnifier_bg_color'] ?? '');
			$root_vars[] = '--readabler-text-magnifier-color: ' . ($options['text_magnifier_color'] ?? '');
			$root_vars[] = '--readabler-text-magnifier-font-size: ' . ($options['text_magnifier_font_size'] ?? '') . 'px';
		}

		if (
			( isset( $options['highlight_titles'] ) && 'on' === $options['highlight_titles'] ) ||
			( isset( $options['profile_cognitive_disability'] ) && 'on' === $options['profile_cognitive_disability'] )
		) {
			$root_vars[] = '--readabler-highlight-titles-style: ' . ($options['highlight_titles_style'] ?? '');
			$root_vars[] = '--readabler-highlight-titles-color: ' . ($options['highlight_titles_color'] ?? '');
			$root_vars[] = '--readabler-highlight-titles-width: ' . ($options['highlight_titles_width'] ?? '') . 'px';
			$root_vars[] = '--readabler-highlight-titles-offset: ' . ($options['highlight_titles_offset'] ?? '') . 'px';
		}

		if (
			( isset( $options['highlight_links'] ) && 'on' === $options['highlight_links'] ) ||
			( isset( $options['profile_cognitive_disability'] ) && 'on' === $options['profile_cognitive_disability'] )
		) {
			$root_vars[] = '--readabler-highlight-links-style: ' . ($options['highlight_links_style'] ?? '');
			$root_vars[] = '--readabler-highlight-links-color: ' . ($options['highlight_links_color'] ?? '');
			$root_vars[] = '--readabler-highlight-links-width: ' . ($options['highlight_links_width'] ?? '') . 'px';
			$root_vars[] = '--readabler-highlight-links-offset: ' . ($options['highlight_links_offset'] ?? '') . 'px';
		}

		if ( isset( $options['reading_guide'] ) && 'on' === $options['reading_guide'] ) {
			$root_vars[] = '--readabler-reading-guide-width: ' . ($options['reading_guide_width'] ?? '') . 'px';
			$root_vars[] = '--readabler-reading-guide-height: ' . ($options['reading_guide_height'] ?? '') . 'px';
			$root_vars[] = '--readabler-reading-guide-bg-color: ' . ($options['reading_guide_background_color'] ?? '');
			$root_vars[] = '--readabler-reading-guide-border-color: ' . ($options['reading_guide_border_color'] ?? '');
			$root_vars[] = '--readabler-reading-guide-border-width: ' . ($options['reading_guide_border_width'] ?? '') . 'px';
			$root_vars[] = '--readabler-reading-guide-border-radius: ' . ($options['reading_guide_border_radius'] ?? '') . 'px';
			$root_vars[] = '--readabler-reading-guide-arrow: ' . ($options['reading_guide_arrow'] ?? '');
		}

		if ( isset( $options['highlight_hover'] ) && 'on' === $options['highlight_hover'] ) {
			$root_vars[] = '--readabler-highlight-hover-style: ' . ($options['highlight_hover_style'] ?? '');
			$root_vars[] = '--readabler-highlight-hover-color: ' . ($options['highlight_hover_color'] ?? '');
			$root_vars[] = '--readabler-highlight-hover-width: ' . ($options['highlight_hover_width'] ?? '') . 'px';
			$root_vars[] = '--readabler-highlight-hover-offset: ' . ($options['highlight_hover_offset'] ?? '') . 'px';
		}

		if ( isset( $options['highlight_focus'] ) && 'on' === $options['highlight_focus'] ) {
			$root_vars[] = '--readabler-highlight-focus-style: ' . ($options['highlight_focus_style'] ?? '');
			$root_vars[] = '--readabler-highlight-focus-color: ' . ($options['highlight_focus_color'] ?? '');
			$root_vars[] = '--readabler-highlight-focus-width: ' . ($options['highlight_focus_width'] ?? ''). 'px';
			$root_vars[] = '--readabler-highlight-focus-offset: ' . ($options['highlight_focus_offset'] ?? '') . 'px';
		}

		return implode( ';', $root_vars ) . ';';

	}

	/**
	 * Analyzer styles
	 *
	 * @param array $options
	 *
	 * @return void
	 */
	private function analyzer_styles( array $options = [] ) {

		if ( Tools::is_enqueue_analyzer() ) {

			wp_enqueue_style(
				'mdp-readabler-analyzer',
				Plugin::get_url() . 'css/readabler-analyzer' . Plugin::get_suffix() . '.css',
				[],
				Plugin::get_version()
			);

		} else {

			$this->admin_bar_styles( $options );

		}

	}

	/**
	 * Admin bar styles
	 *
	 * @param array $options
	 *
	 * @return void
	 */
	private function admin_bar_styles( array $options = [] ) {

		// Is admin bar enabled?
		$is_admin_bar = isset( $options['analyzer_admin_bar'] ) && $options['analyzer_admin_bar'] === 'on';

		// Is admin bar showing for current user
		$is_admin_bar_showing = is_admin_bar_showing();

		/** Admin bat styles */
		if ( $is_admin_bar && $is_admin_bar_showing ) {

			wp_enqueue_style(
				'mdp-readabler-analyzer',
				Plugin::get_url() . 'css/admin-bar' . Plugin::get_suffix() . '.css',
				[],
				Plugin::get_version()
			);

		}


	}

	/**
	 * Main FrontStyles Instance.
	 *
	 * Insures that only one instance of FrontStyles exists in memory at any one time.
	 *
	 * @static
	 * @return FrontStyles
	 * @since 1.0.0
	 **/
	public static function get_instance(): FrontStyles {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

} // End Class FrontStyles.
