<?php
defined( 'ABSPATH' ) or die( 'Something went wrong.' );


/**
 * File Monitoring class.
 *
 * @package SecuPress
 * @since 1.0
 */
class SecuPress_File_Monitoring extends SecuPress_Singleton {

	const VERSION = '1.0';

	/**
	 * Singleton The reference to *Singleton* instance of this class.
	 *
	 * @var (object)
	 */
	protected static $_instance;

	/**
	 * SecuPress_Background_Process_File_Monitoring instance.
	 *
	 * @var (object)
	 */
	protected $background_process;


	/** Public methods ========================================================================== */

	/**
	 * Add tasks to the queue and dispatch.
	 *
	 * @since 1.0
	 *
	 * @return (bool) True on success. False if it was already running.
	 */
	public function do_file_scan() {
		global $wp_version, $wp_local_package;

		if ( $this->is_monitoring_running() ) {
			return false;
		}

		// Cleanup previous batches.
		$this->stop_file_scan();

		$wp_core_files_hashes = get_site_option( SECUPRESS_WP_CORE_FILES_HASHES );

		if ( false === $wp_core_files_hashes || empty( $wp_core_files_hashes[ $wp_version ] ) ) {
			if ( ! $this->background_process->push_to_queue( 'get_wp_hashes' ) ) {
				return false;
			}
		}

		if ( isset( $wp_local_package, $wp_core_files_hashes['locale'] ) && $wp_core_files_hashes['locale'] !== $wp_local_package ) {
			$fix_dists = get_site_option( SECUPRESS_FIX_DISTS );

			if ( false === $fix_dists || ! isset( $fix_dists[ $wp_version ] ) ) {
				$this->background_process->push_to_queue( 'fix_dists' );
			}
		}

		$this->background_process->push_to_queue( 'scan_full_tree' )->save()->dispatch();
		return true;
	}


	/**
	 * Launch DB scan
	 *
	 * @since 2.0
	 */
	public function do_database_scan() {
		global $wpdb;
		$keywords = secupress_get_database_malware_keywords();

		if ( empty( $keywords ) ) {
			return [];
		}

		$reqs             = [];
		$sqls             = [];
		$sqls['posts']    = 'SELECT ID, post_content from ' . $wpdb->posts . ' WHERE 1=0';
		$sqls['opt_val']  = 'SELECT option_name, option_value from ' . $wpdb->options . ' WHERE 1=0';
		// $sqls['cpts']     = 'SELECT ID, post_content from ' . $wpdb->posts . ' WHERE 1=0';
		// $sqls['opts']     = 'SELECT option_name, option_value from ' . $wpdb->options . ' WHERE 1=0';
		// Build SQL queries first.
		foreach ( $keywords as $key => $all ) {
			switch( $key ) {
				case 'posts':
					foreach( $all as $items ) {
						$sqls['posts']   .= ' OR ( 1=1 ';
						$sqls['opt_val'] .= ' OR ( 1=1 ';
						foreach( $items as $item ) {
							$sqls['posts']   .= ' AND post_content LIKE "%' . ( $item ) . '%"';
							$sqls['opt_val'] .= ' AND option_value LIKE "%' . ( $item ) . '%"';
						}
						$sqls['posts']   .= ')';
						$sqls['opt_val'] .= ')';
					}
				break;
				// case 'cpts':
				// 	foreach( $all as $item ) {
				// 		$sqls[ $key ] .= ' OR post_type = "' . ( $item ) . '"';
				// 	}
				// break;
				// case 'opts':
				// 	foreach( $all as $item ) {
				// 		$sqls[ $key ] .= ' OR option_name = "' . ( $item ) . '"';
				// 	}
				// break;
			}
		}
		// Run them.		// Run them.
		foreach ( $sqls as $key => $sql ) {
			$reqs[ $key ] = $wpdb->get_results( $sql, ARRAY_A );
		}
		// Save them.
		update_site_option( SECUPRESS_DATABASE_MALWARES, str_rot13( json_encode( array_filter( $reqs ) ) ) );
		return true;
	}



	/**
	 * Launch content spam scan
	 *
	 * @since 2.2.6
	 */
	public function do_content_spam_scan() {
		if ( ! secupress_wp_version_is( '6.5' ) ) {
			// WP_HTML_Tag_Processor as we use it requires WP 6.5 or +.
			return true;
		}
		$contents          = [];
		$filename          = secupress_get_data_file_path( 'tag_attr' );
		$tags              = [];
		if ( file_exists( $filename ) ) {
			$tags          = explode( ',', file_get_contents( $filename ) );
		}
		/**
		 * Manage the tags to be filtered, format : "tag|attr", example "a|href"
		 * 
		 * @param (array) $tags
		 * @since 2.2.6
		 * @return (array) $tags
		 */
		$tags              = apply_filters( 'secupress.content-spam.scanner.tags', $tags );
		// So, nothing to scan?
		if ( empty( $tags ) ) {
			return false;
		}
		$tags              = array_fill_keys( $tags, [] );

		/**
		 * Tell if we have to also add 3 random post content from DB.
		 * 
		 * @param (bool|array) False to not do it, True to do it with default value, Array with your values.
		 * @since 2.2.6
		 * @return (bool|array)
		 */
		$query             = apply_filters( 'secupress.content-spam.scanner.query', true );
		if ( $query ) {
			if ( is_bool( $query ) ) {
				$query = [ 'orderby' => 'rand', 'fields' => 'ids', 'post_type' => 'any', 'posts_per_page' => 3 ];
			}
			if ( is_array( $query ) ) {
				$query         = new WP_Query( $query );
			}
		}
		if ( ! empty( $query->posts ) ) {
			foreach( $query->posts as $_pid ) {
				$contents[ $_pid ] = get_post( $_pid )->post_content;
			}
		}

		$urls              = [ home_url(), home_url( '/secupress-404-page-test-' . time() . '.html' ) ];
		/**
		 * Manage the urls to be tested online to find external resources loaded from the website.
		 * 
		 * @param (array) $urls
		 * @since 2.2.6
		 * @return (array) $urls
		 */
		$urls              = apply_filters( 'secupress.content-spam.scanner.test_urls', $urls );

		foreach( $urls as $url ) {
			$response = wp_remote_get( $url );
			if ( 200 !== wp_remote_retrieve_response_code( $response ) ) {
				continue;
			}
			$contents[ $url ] = wp_remote_retrieve_body( $response );
		}
		// So, nothing to scan?
		if ( empty( $contents ) ) {
			return false;
		}

		$allowed_domains   = [ trim( str_replace( [ 'https://', 'http://' ], '', home_url( '/' ) ), '/' ) ];
		/**
		 * Manage the allowed domains that does not need to be filtered.
		 * 
		 * @param (array) $allowed_domains
		 * @since 2.2.6
		 * @return (array) $allow_domains
		 */
		$allowed_domains   = apply_filters( 'secupress.content-spam.scanner.allowed_domains', $allowed_domains ); 

		foreach( $contents as $ID => $content ) {
			foreach( array_keys( $tags ) as $tag ) {
				$html = new WP_HTML_Tag_Processor( $content );
				list( $_tag, $_attr ) = explode( '|', $tag );
				while( $html->next_tag( $_tag ) ) {
					$attr    = $html->get_attribute( $_attr );
					if ( ! $attr ) {
						continue;
					}
					$attr    = explode( '#', $attr );
					$attr    = reset( $attr );
					if ( ! $_attr ) {
						continue;
					}
					$_attr    = strtolower( $_attr );
					if ( strpos( $attr, 'http://' ) === false && strpos( $attr, 'https://' ) === false ) {
						continue;
					}
					$href_dom = strtolower( parse_url( $attr, PHP_URL_HOST ) );
					$href_dom = parse_url( $attr, PHP_URL_HOST );

					// Shortcut the foreach, could be a waste.
					if ( secupress_cache_data( $href_dom ) ) {
						$tags[ $tag ][] = [ 'attr' => $attr, 'ID' => $ID ];
						// $tags[ $tag ]   = array_unique( $tags[ $tag ] );
						$tags[ $tag ]   = array_keys( array_column( $tags[ $tag ], null, 'attr') );
						continue;
					}
					$flag = false;
					if ( ! empty( $allowed_domains ) ) {
						$flag = 0;
						foreach( $allowed_domains as $allowed ) {
							if ( strpos( $href_dom, $allowed ) === false ) {
								++$flag;
							}
						}
					}
					if ( $flag > 0 || false === $flag ) {
						secupress_cache_data( $href_dom, 1 );
						$tags[ $tag ][] = [ 'attr' => $attr, 'ID' => $ID ];
						// $tags[ $tag ]   = array_unique( $tags[ $tag ] );
						$tags[ $tag ]   = array_keys( array_column( $tags[ $tag ], null, 'attr') );
					}
				}
			}
		}
		// Save them.
		update_site_option( SECUPRESS_CONTENT_SPAM_SCAN, str_rot13( json_encode( $tags ) ) );

		return true;
	}



	/**
	 * Remove everything from the queue.
	 *
	 * @since 1.0
	 */
	public function stop_file_scan() {
		$this->background_process->delete_queue();
	}


	/**
	 * Is process running.
	 * Check whether the current process is already running in a background process.
	 *
	 * @since 1.0
	 *
	 * @return (bool)
	 */
	public function is_monitoring_running() {
		return $this->background_process->is_monitoring_running();
	}

	public function get_batch_id() {
		return $this->background_process->get_batch();
	}
	/** Private methods ========================================================================= */

	/**
	 * Class init.
	 *
	 * @since 1.0
	 */
	protected function _init() {
		secupress_require_class_async();

		require_once( SECUPRESS_PRO_MODULES_PATH . 'file-system/plugins/inc/php/file-monitoring/class-secupress-background-process-file-monitoring.php' );

		$this->background_process = new SecuPress_Background_Process_File_Monitoring;
	}
}
