<?php
/**
 * Module Name: Detect Bad Themes
 * Description: Detect if a theme you're using is known as vulnerable, closed, or removed from w.org.
 * Main Module: plugins_themes
 * Author: SecuPress
 * Version: 2.2.6
 */

defined( 'SECUPRESS_VERSION' ) or die( 'Something went wrong.' );


add_action( 'secupress.plugins.loaded', 'secupress_bad_themes_async_init' );
/**
 * Instantiate bad plugins class.
 *
 * @author Julio Potier
 * @since 2.2.6
 */
function secupress_bad_themes_async_init() {
	add_filter( 'secupress_bad_themes_cron_interval', function(){return 1;} );
	add_filter( 'secupress_bad_themes_default_time_limit', function(){return 5;} );

	secupress_require_class_async();

	require_once( SECUPRESS_PRO_MODULES_PATH . 'plugins-themes/plugins/inc/php/class-secupress-background-process-bad-themes.php' );

	SecuPress_Background_Process_Bad_Themes::get_instance();
}

add_action( 'secupress_bad_themes_maybe_do_checks', 'secupress_bad_themes_maybe_do_checks' );
/**
 * Maybe process scheduled tests.
 *
 * @author Julio Potier
 * @since 2.2.6
 */
function secupress_bad_themes_maybe_do_checks() {
	$process = SecuPress_Background_Process_Bad_Themes::get_instance();
	if ( $process->is_processing() ) {
		return;
	}

	if ( ! function_exists ( 'wp_get_themes' ) ) {
		require_once( WPINC . '/theme.php' );
	}
	$retests  = array_keys( wp_get_themes() );

	array_map( [ $process, 'push_to_queue' ], $retests );

	$process->save()->dispatch();
}

add_action( 'admin_footer-themes.php', 'secupress_detect_bad_themes_after_theme_row' );
/**
 * Add a red banner on each "bad" theme on themes page
 *
 * @return void
 * @since 1.0
 */
function secupress_detect_bad_themes_after_theme_row() {
	if ( ( is_network_admin() || ! is_multisite() ) && ! current_user_can( 'update_themes' ) && ! current_user_can( 'delete_themes' ) && ! current_user_can( 'switch_themes' ) ) { // ie. Administrator.
		return;
	}

	$themes     = [ 'vulns'  => secupress_get_bad_themes( 'vulns' ),
					'old'    => secupress_get_bad_themes( 'old' ),
					'closed' => secupress_get_bad_themes( 'closed' ),
				];
	$all_themes = wp_get_themes();

	foreach ( $all_themes as $theme_name => $theme_data ) {
		$is_vuln    = isset( $themes['vulns'][ $theme_name ] );
		$is_old     = isset( $themes['old'][ $theme_name ] );
		$is_closed  = isset( $themes['closed'][ $theme_name ] );
		$theme_vuln = $is_vuln ? $themes['vulns'][ $theme_name ] : false;

		if ( ! $is_vuln && ! $is_old && ! $is_closed ) {
			continue;
		}
		if ( $is_vuln && version_compare( $theme_data['Version'], $theme_vuln['fixed_in'] ) === 1 && '' !== $theme_vuln['fixed_in'] ) {
			continue;
		}

		$current_theme = wp_get_theme();
		$current       = get_site_transient( 'update_themes' );
		$r             = isset( $current->response[ $theme_name ] ) ? (object) $current->response[ $theme_name ] : null;

		// HTML output.
		echo '<div class="update-message notice inline notice-error notice-alt secupress-bad-theme" data-theme="' . esc_attr( $theme_name ) . '"><p>';
		printf( '<em>' . sprintf( __( '%s Warning:', 'secupress' ), SECUPRESS_PLUGIN_NAME ) . '</em> ' );
		if ( $is_vuln ) {

			printf( _x( '<strong>%1$s %2$s</strong> is known to contain this vulnerability: %3$s.', 'unknow plugin', 'secupress' ),
				$theme_data['Name'],
				'' !== $theme_vuln['fixed_in'] ? sprintf( __( 'version %s (or lower)', 'secupress' ), $theme_vuln['fixed_in'] ) : __( 'all versions', 'secupress' ),
				'<strong><a href="' . esc_url( $theme_vuln['refs'] ) . '" target="_blank">' . esc_html( $theme_vuln['flaws'] ) . '</a></strong>'
			);


			if ( $theme_vuln['fixed_in'] && current_user_can( 'update_themes' ) ) {
				echo '<p>';

				if ( ! empty( $r->package ) ) {
					printf(
						'<span class="dashicons dashicons-update"></span> ' . __( 'We invite you to <a href="%1$s">Update</a> this theme in version %2$s.', 'secupress' ),
						wp_nonce_url( admin_url( 'update.php?action=upgrade-theme&theme=' ) . $theme_name, 'upgrade-theme_' . $theme_name ),
						'<strong>' . esc_html( isset( $r->new_version ) ? $r->new_version : $theme_vuln['fixed_in'] ) . '</strong>'
					);
				} else {
					'<span class="dashicons dashicons-update"></span> ' . __( 'We invite you to Update this theme <em>(Automatic update is unavailable for this theme.)</em>.', 'secupress' );
				}

				echo '</p>';

				if ( $theme_name === $current_theme->stylesheet || $theme_name === $current_theme->template ) {
					echo '<span class="dashicons dashicons-admin-appearance"></span> ' . __( 'We invite you to switch theme, then <strong>delete</strong> it.', 'secupress' );
				} else {
					$delete_url = wp_nonce_url( admin_url( 'themes.php?action=delete&stylesheet=' . $theme_name ), 'delete-theme_' . $theme_name );
					printf( '<span class="dashicons dashicons-admin-appearance"></span> ' . __( 'We invite you to <a href="%s">Delete</a> it.', 'secupress' ), $delete_url );
				}
			}

		} elseif( $is_closed ) {
			list( $date, $reason ) = explode( '|', $themes['closed'][ $theme_name ] );
			printf( _x( '<strong>%s</strong> has been removed from official repository since %s. Reason: %s', 'removed plugin', 'secupress' ), esc_html( $theme_data['Name'] ), esc_html( date_i18n( get_option( 'date_format' ), $date ) ), secupress_closed_reason_i18n( $reason, 'theme' ) );
		} else { // Old.
			$y = secupress_minmax_range( 2, 20, (int) (new DateTime( '@'.time() ))->diff( new DateTime( '@'.$themes['old'][ $theme_name ] ) )->y );
			printf( _x( '<strong>%s</strong> has not been updated on official repository for more than %d years now (since %s). It can be dangerous to continue to use it.', 'updated plugin', 'secupress' ), esc_html( $theme_data['Name'] ), $y, esc_html( date_i18n( get_option( 'date_format' ), $themes['old'][ $theme_name ] ) ) );
		}
		echo '</p></div>';
		?>
		</td>
	</tr>
	<?php
	}
}


add_action( 'admin_head', 'secupress_detect_bad_themes_add_notices' );
/**
 * Add a notice if a theme is considered as "bad".
 *
 * @return void
 * @since 1.0
 */
function secupress_detect_bad_themes_add_notices() {
	global $pagenow;

	if ( 'themes.php' === $pagenow || ( ( is_network_admin() || ! is_multisite() ) && ! current_user_can( secupress_get_capability() ) ) ) {
		return;
	}

	$themes     = [ 'vulns'  => secupress_get_bad_themes( 'vulns' ),
					'old'    => secupress_get_bad_themes( 'old' ),
					'closed' => secupress_get_bad_themes( 'closed' ),
				];
	if ( ! $themes['vulns'] && ! $themes['old'] && ! $themes['closed'] ) {
		return;
	}

	$counter = count( (array) $themes['vulns'] ) + count( $themes['old'] ) + count( $themes['closed'] );
	$url     = admin_url( 'themes.php' );
	$message = sprintf(
		_n(
			'Your installation contains %1$s theme considered as <em>bad</em>, check the details in <a href="%2$s">the themes page</a>.',
			'Your installation contains %1$s themes considered as <em>bad</em>, check the details in <a href="%2$s">the themes page</a>.',
			$counter,
			'secupress-pro'
		),
		'<strong>' . $counter . '</strong>',
		$url
	);

	secupress_add_notice( $message, 'error', 'bad-themes' );
}

add_action( 'secupress.pro.plugins.activation',                                     'secupress_bad_themes_activation' );
add_action( 'secupress.modules.activate_submodule_' . basename( __FILE__, '.php' ), 'secupress_bad_themes_activation' );
/**
 * Initiate the cron that will check for vulnerable themes twice-daily.
 *
 * @since 2.1
 * @author Julio Potier
 */
function secupress_bad_themes_activation() {
	if ( ! wp_next_scheduled( 'secupress_bad_themes' ) ) {
		wp_schedule_event( time(), 'daily', 'secupress_bad_themes' );
		wp_schedule_event( time(), 'daily', 'secupress_bad_themes_maybe_do_checks' );
	}
}

add_action( 'secupress_bad_themes', 'secupress_detect_bad_themes_async_get_and_store_infos' );
add_action( 'admin_post_secupress_bad_themes_update_data', 'secupress_detect_bad_themes_async_get_and_store_infos' );
/**
 * Save the data to refresh the vulnerable plugins. 
 * Moved from Pro to Free + renamed. Originally `secupress_detect_bad_themes_async_get_infos()` (deprecated).
 * 
 *
 * @since 2.2.6 Change the URL API call
 * @since 2.1 Moved from /core/admin/admin.php, old hook "admin_footer" via admin-post using AJAX
 * @since 1.1.3
 */
function secupress_detect_bad_themes_async_get_and_store_infos() {
	if ( ! wp_doing_cron() ) {
		check_admin_referer( 'secupress_bad_themes_update_data' );
	}
	if ( ! function_exists ( 'wp_get_theme' ) ) {
		require_once( ABSPATH . '/wp-includes/theme.php' );
	}
	$themes   = wp_get_themes();
	$themes  = wp_list_pluck( $themes, 'Version' );
	$nonce    = md5( serialize( $themes ) );
	$headers  = secupress_get_basic_auth_headers();
	$args     = [ 'body' => [ 'items' => $themes, 'type' => 'theme', '_wpnonce' => $nonce ], 'headers' => $headers ];

	$response = wp_remote_post( SECUPRESS_API_MAIN . 'vulns/v2/', $args );

	if ( ! is_wp_error( $response ) && 200 === wp_remote_retrieve_response_code( $response ) ) {
		$response = wp_remote_retrieve_body( $response );

		// Store the results
		$response = json_decode( $response, true );
		if ( count( $response ) >= 0 ) {
			$response = json_encode( $response );
			update_site_option( SECUPRESS_BAD_THEMES, $response );
			$dt = get_date_from_gmt( date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), time() ), get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) );
			secupress_set_option( 'bad_themes_last_update', date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $dt ) ) );
		}
	}
	if ( ! wp_doing_cron() ) {
		wp_safe_redirect( wp_get_referer() );
	}
}

add_action( 'secupress.pro.plugins.deactivation',                                     'secupress_bad_plugins_deactivation' );
add_action( 'secupress.modules.deactivate_submodule_' . basename( __FILE__, '.php' ), 'secupress_bad_plugins_deactivation' );
/**
 * Remove the crons.
 *
 * @since 2.1
 * @author Julio Potier
 */
function secupress_bad_themes_deactivation() {
	wp_clear_scheduled_hook( 'secupress_bad_themes' );
}