<?php
/**
 * Module Name: Bad URL Access
 * Description: Deny access to some sensitive files.
 * Main Module: sensitive_data
 * Author: SecuPress
 * Version: 2.3.13
 */

defined( 'SECUPRESS_VERSION' ) or die( 'Something went wrong.' );

/** --------------------------------------------------------------------------------------------- */
/** ACTIVATION / DEACTIVATION =================================================================== */
/** --------------------------------------------------------------------------------------------- */

add_action( 'secupress.modules.activation', 'secupress_pro_bad_url_access_activation' );
/**
 * On module activation, maybe write the rules.
 *
 * @since 1.0
 * @since 1.0.2 Return a boolean.
 * @author Grégory Viguier
 *
 * @return (bool) True if rules have been successfully written. False otherwise.
 */
function secupress_pro_bad_url_access_activation() {
	global $is_apache, $is_nginx, $is_iis7;

	switch( true ) {
		case $is_apache:
			$rules = secupress_pro_bad_url_access_apache_rules();
		break;
		case $is_nginx:
			$rules = secupress_pro_bad_url_access_nginx_rules();
		break;
		case $is_iis7:
			$rules = secupress_pro_bad_url_access_iis7_rules();
		break;
		default:
			$rules = '';
		break;
	}

	return secupress_add_module_rules_or_notice( array(
		'rules'  => $rules,
		'marker' => 'bad_url_access',
		'title'  => __( 'Bad URL Access', 'secupress' ),
	) );
}


add_action( 'secupress.modules.activate_submodule_' . basename( __FILE__, '.php' ), 'secupress_pro_bad_url_access_activate' );
/**
 * On module de/activation, rescan.
 *
 * @since 2.0
 * @author Julio Potier
 */
function secupress_pro_bad_url_access_activate() {
	secupress_pro_bad_url_access_activation();
	secupress_scanit( 'Bad_URL_Access', 3 );
}


add_action( 'secupress.modules.deactivate_submodule_' . basename( __FILE__, '.php' ), 'secupress_pro_bad_url_access_deactivate' );
/**
 * On module deactivation, maybe remove rewrite rules from the `.htaccess`/`web.config` file.
 *
 * @since 1.0
 * @author Grégory Viguier
 */
function secupress_pro_bad_url_access_deactivate() {
	secupress_remove_module_rules_or_notice( 'bad_url_access', __( 'Bad URL Access', 'secupress' ) );
	secupress_scanit( 'Bad_URL_Access', 3 );
}


add_filter( 'secupress.plugins.activation.write_rules', 'secupress_pro_bad_url_access_plugin_activate', 10, 2 );
/**
 * On SecuPress activation, add the rules to the list of the rules to write.
 *
 * @since 1.0
 * @author Grégory Viguier
 *
 * @param (array) $rules Other rules to write.
 * @return (array) Rules to write.
 */
function secupress_pro_bad_url_access_plugin_activate( $rules ) {
	global $is_apache, $is_nginx, $is_iis7;
	$marker = 'bad_url_access';

	switch( true ) {
		case $is_apache:
			$rules[ $marker ] = secupress_pro_bad_url_access_apache_rules();
		break;
		case $is_nginx:
			$rules[ $marker ] = secupress_pro_bad_url_access_nginx_rules();
		break;
		case $is_iis7:
			$rules[ $marker ] = [ 'nodes_string' => secupress_pro_bad_url_access_iis7_rules() ];
		break;
		default:
			$rules[ $marker ] = '';
	}

	return $rules;
}


/** --------------------------------------------------------------------------------------------- */
/** TOOLS ======================================================================================= */
/** --------------------------------------------------------------------------------------------- */

/**
 * Get a regex pattern matching the file extensions.
 *
 * @since 2.2.6 Usage of get_allowed_mime_types(), so we invert the way it worked, we let pass instead of blocking.
 * @author Julio Potier
 *
 * @return (string)
 */
function secupress_pro_bad_url_access_get_exts_pattern() {
	$exts = secupress_get_allowed_extensions();
	$exts = array_flip( array_flip( $exts ) );
	$exts = implode( '|', $exts );

	return $exts;
}

/**
 * Get rules for apache.
 *
 * @since 2.3.13 Keep the legacy code
 * @since 2.2.6 Revamp
 * @author Julio Potier
 * @since 1.0
 * @author Grégory Viguier
 *
 * @return (string)
 */
function secupress_pro_bad_url_access_apache_rules() {
	$WP_CONTENT_DIR     = basename( WP_CONTENT_DIR );
	$patterns   = secupress_bad_url_access_get_regex_pattern();
	$bases      = secupress_get_rewrite_bases();
	$base       = $bases['base'];
	$site_from  = '/' . $bases['site_from'];
	$path       = str_replace( ABSPATH, '', SECUPRESS_INC_PATH );
	$customurls = secupress_bad_url_access_sort_urls();

	$rules      = "<IfModule mod_rewrite.c>\n";
	$rules     .= "    RewriteEngine On\n";
	$rules     .= "    RewriteBase $base\n";
	$rules     .= "\n";
	$rules     .= "    # Define REDIRECT_PHP404 to 0 by default\n";
	$rules     .= "    RewriteRule ^ - [E=REDIRECT_PHP404:0]\n";
	$rules     .= "\n";
	$rules     .= "    # FOLDERS case\n";
	$rules     .= "    RewriteCond %{ENV:REDIRECT_PHP404} =0\n";
	$rules     .= "    RewriteCond %{REQUEST_URI} ^{$site_from}secupress-sandbox- [NC]\n";
	$rules     .= "    RewriteRule ^ - [L]\n";
	$rules     .= "    RewriteCond %{REQUEST_FILENAME} -d\n";
	$rules     .= "    RewriteCond %{REQUEST_URI} !^{$site_from}?$ [NC]\n";
	if ( ! empty( $customurls['folders'] ) ) {
		foreach ( $customurls['folders'] as $url ) {
			if ( ! $url ) {
				continue;
			}
			$url    = str_replace( home_url( '/' ), '/', $url );
			$rules .= "    RewriteCond %{REQUEST_URI} !^{$url}?$ [NC]\n";
		}
	}
	$rules     .= "    RewriteRule ^ - [E=REDIRECT_PHP404:folders]\n";
	$rules     .= "\n";
	$rules     .= "    # FILES case\n";
	$rules     .= "    RewriteCond %{ENV:REDIRECT_PHP404} =0\n";
	$rules     .= "    RewriteCond %{REQUEST_FILENAME} -f\n";
	$rules     .= "    RewriteCond %{REQUEST_URI} \.php(\?|$) [NC]\n";
	$rules     .= "    RewriteCond %{REQUEST_URI} ^/([^?]+)\n";
	$rules     .= "    RewriteCond %{REQUEST_URI} !^{$site_from}wp-admin/? [NC]\n";
	foreach ( $patterns as $pattern ) {
		$rules .= "    RewriteCond %{REQUEST_URI} !^{$site_from}{$pattern}$ [NC]\n";
	}
	if ( ! empty( $customurls['files'] ) ) {
		foreach ( $customurls['files'] as $url ) {
			if ( ! $url ) {
				continue;
			}
			$url    = str_replace( home_url( '/' ), '', $url );
			$rules .= "    RewriteCond %{REQUEST_URI} !^{$url}$ [NC]\n";
		}
	}
	$rules     .= "    RewriteRule ^ - [E=REDIRECT_PHP404:files]\n";
	$rules     .= "\n";
	$rules     .= "    # CONTENT case\n";
	$rules     .= "    RewriteCond %{ENV:REDIRECT_PHP404} =0\n";
	$rules     .= "    RewriteCond %{REQUEST_FILENAME} -f\n";
	$rules     .= "    RewriteCond %{REQUEST_URI} \.php(\?|$) [NC]\n";
	$rules     .= "    RewriteCond %{REQUEST_URI} ^{$site_from}{$WP_CONTENT_DIR} [NC]\n";
	if ( ! empty( $customurls['content'] ) ) {
		foreach ( $customurls['content'] as $url ) {
			if ( ! $url ) {
				continue;
			}
			$url    = str_replace( home_url( '/' ), '', $url );
			$rules .= "    RewriteCond %{REQUEST_URI} !^{$site_from}{$url}$ [NC]\n";
		}
	}
	$rules     .= "    RewriteRule ^ - [E=REDIRECT_PHP404:content]\n";
	$rules     .= "\n";
	$rules     .= "    # EXTS case\n";
	$rules     .= "    RewriteCond %{ENV:REDIRECT_PHP404} =0\n";
	$rules     .= "    RewriteCond %{REQUEST_FILENAME} -f\n";
	$rules     .= "    RewriteCond %{REQUEST_FILENAME} !\.(" . secupress_pro_bad_url_access_get_exts_pattern() . ")(\?|$) [NC]\n";
	$rules     .= "    RewriteRule ^ - [E=REDIRECT_PHP404:exts]\n";
	$rules     .= "\n";
	$rules     .= "    RewriteCond %{ENV:REDIRECT_PHP404} !=0\n";
	$rules     .= "    " . secupress_get_404_rule_for_rewrites() . "\n";
	$rules     .= "</IfModule>\n";

	return $rules;
}

/**
 * Get rules for nginx.
 *
 * @since 2.3.13 Keep the legacy code to prevent people from activating this feature without any configuration and them being blocked and then deactivate the plugin because well, "it does not work"
 * @since 2.2.6 Invert the behaviour
 * @author Julio Potier
 * @since 1.0
 * @author Grégory Viguier
 *
 * @return (string)
 */
function secupress_pro_bad_url_access_nginx_rules() {
	$marker     = 'bad_url_access';
	$bases      = secupress_get_rewrite_bases();
	$patterns   = secupress_bad_url_access_get_regex_pattern();
	$site_from  = '/' . $bases['site_from'];
	$path       = str_replace( ABSPATH, '', SECUPRESS_INC_PATH );
	$customurls = secupress_bad_url_access_sort_urls();
	$exts       = secupress_pro_bad_url_access_get_exts_pattern();
	$the_404    = secupress_get_404_rule_for_rewrites();
	$tuto       = sprintf( __( 'Then add the following in a file like %s because it should be loaded in a "http" context of NGINX.', 'secupress' ), secupress_code_me( '/etc/nginx/conf.d/secupress.conf' ) );

	$REDIRECT_PHP404  = '$REDIRECT_PHP404';
	$request_filename = '$request_filename';
	$request_uri      = '$request_uri';
	$REDIRECT_404_CHK = '$REDIRECT_404_CHK';
	$REDIRECT_PHPTYPE = '$REDIRECT_PHPTYPE';
	$WP_CONTENT_DIR   = basename( WP_CONTENT_DIR );

	// Folders
	$custom_folders   = '';
	if ( ! empty( $customurls['folders'] ) ) {
		foreach ( $customurls['folders'] as $url ) {
			if ( ! $url ) {
				continue;
			}
			$url             = str_replace( home_url( '/' ), '/', $url );
			$custom_folders .= "\t" . "\"~*D:{$url}?$ 0;\n";
		}
	}
	// Files
	$custom_files            = '';
	foreach ( $patterns as $pattern ) {
		$custom_files       .= "\t\t" . "\"~*F:{$site_from}{$pattern}(\?|$)\" O;\n";
	}
	if ( ! empty( $customurls['files'] ) ) {
		foreach ( $customurls['files'] as $url ) {
			if ( ! $url ) {
				continue;
			}
			$url             = str_replace( home_url( '/' ), '', $url );
			$custom_files   .= "\t\t" . "\"~*F:{$url}(\?|$)\") 0;\n";
		}
	}
	// Content
	$custom_content          = '';
	if ( ! empty( $customurls['content'] ) ) {
		foreach ( $customurls['content'] as $url ) {
			if ( ! $url ) {
				continue;
			}
			$url             = str_replace( home_url( '/' ), '/', $url );
			$custom_content .= "\t" . "\"~*F:/{$url}(\?|$) content;\n";
		}
	}

	$rules                   = "
server {
	# BEGIN SecuPress {$marker}

	if (-d $request_filename) {
		set $REDIRECT_PHPTYPE D;
	}
	if (-f $request_filename) {
		set $REDIRECT_PHPTYPE F;
	}
	if ($REDIRECT_PHP404 != 0) {
		$the_404
	}
	# END SecuPress
}	
# $tuto
http {
	# BEGIN SecuPress {$marker}

	# Define REDIRECT_PHP404 to 0 by default
	map '' $REDIRECT_PHPTYPE {
		default '';
	}

	# FOLDERS case and php FILES case
	map \"$REDIRECT_PHPTYPE:$request_uri\" $REDIRECT_404_CHK {
		default 0;
		'~*:/\.well-known/' 0;
		'~*(D|F):{$site_from}secupress-sandbox-' 0;
		'~D:{$site_from}?(\?|$)' 0;
		'~*D:{$site_from}wp-admin(/(?:\?.*)?)?$' 0;
		'~D:.*' folders;
{$custom_folders}
{$custom_content}
{$custom_files}
		'~*0:F:{$site_from}{$WP_CONTENT_DIR}/.*\.php(\?|$)' content;
		'~*F:.*\.php(\?|$)' files;
		'~*:.*\.php(\?|$)' files;
	}
	# EXTS case
	map \"$REDIRECT_404_CHK:$REDIRECT_PHPTYPE:$request_uri\" $REDIRECT_PHP404 {
		default 0;
		'~folders:.*:.*' folders;
		'~files:.*:.*' files;
		'~content:.*:.*' content;
		'~*0:F:.*\.($exts)(\?|$)' 0;
		'~*0:F:.*' exts;
	}
	# END SecuPress
}";

	return trim( $rules );
}

/**
 * Get rules for iis7.
 *
 * @since 2.3.13 Keep the legacy code to prevent people from activating this feature without any configuration and them being blocked and then deactivate the plugin because well, "it does not work"
 * @since 2.2.6
 * @author Julio Potier
 * @since 1.0
 * @author Grégory Viguier
 *
 * @return (string)
 */
function secupress_pro_bad_url_access_iis7_rules() {
	$marker         = 'bad_url_access';
	$patterns       = secupress_bad_url_access_get_regex_pattern();
	$bases          = secupress_get_rewrite_bases();
	$site_from      = '/' . $bases['site_from'];
	$WP_CONTENT_DIR = basename( WP_CONTENT_DIR );
	$path       = str_replace( ABSPATH, '', SECUPRESS_INC_PATH );
	$customurls = secupress_bad_url_access_sort_urls();
	$exts       = secupress_pro_bad_url_access_get_exts_pattern();
	$the_404    = secupress_get_404_rule_for_rewrites();

	// Folders
	$custom_folders   = '';
	if ( ! empty( $customurls['folders'] ) ) {
		foreach ( $customurls['folders'] as $url ) {
			if ( ! $url ) {
				continue;
			}
			$url    = str_replace( home_url( '/' ), '/', $url );
			$custom_folders   .= "			<add input='{REQUEST_URI}' pattern='^{$url}?$' negate='true' />\n";
		}
	}
	$custom_folders = rtrim( $custom_folders );

	// Files
	$custom_files        = '';
	foreach ( $patterns as $pattern ) {
		$custom_files   .= "			<add input='{REQUEST_URI}' pattern='^{$site_from}{$pattern}$' negate='true' ignoreCase='true' />\n";
	}
	if ( ! empty( $customurls['files'] ) ) {
		foreach ( $customurls['files'] as $url ) {
			if ( ! $url ) {
				continue;
			}
			$url    = str_replace( home_url( '/' ), '', $url );
			$custom_files   .= "			<add input='{REQUEST_URI}' pattern='^{$url}$' negate='true' ignoreCase='true' />\n";
		}
	}
	$custom_files = rtrim( $custom_files );

	// Content
	$custom_content       = '';
	if ( ! empty( $customurls['content'] ) ) {
		foreach ( $customurls['content'] as $url ) {
			if ( ! $url ) {
				continue;
			}
			$url    = str_replace( home_url( '/' ), '/', $url );
			$custom_content   .= "			<add input='{REQUEST_URI}' pattern='^{$url}$' negate='true' ignoreCase='true' />\n";
		}
	}
	$custom_content = rtrim( $custom_content );

	$rules      = "
<rules>
	<!-- Define REDIRECT_PHP404 to 0 by default -->
	<rule name='Set REDIRECT_PHP404 to 0' stopProcessing='true'>
		<match url='.*' />
		<conditions>
			<add input='{REDIRECT_PHP404}' pattern='^$' />
		</conditions>
		<action type='Rewrite' value='0' />
	</rule>

	<!-- FOLDERS case -->
	<rule name='Check FOLDERS case' stopProcessing='true'>
		<match url='.*' />
		<conditions logicalGrouping='MatchAll'>
			<add input='{REDIRECT_PHP404}' pattern='^0$' />
			<add input='{REQUEST_FILENAME}' matchType='IsDirectory' />
			<add input='{REQUEST_URI}' pattern='^/?$' negate='true' />
			<add input='{REQUEST_URI}' pattern='^{$site_from}wp-admin/?$' negate='true' />
			<add input='{REQUEST_URI}' pattern='^{$site_from}secupress-sandbox-' negate='true' />
{$custom_folders}
		</conditions>
		<action type='Rewrite' url='{R:0}' appendQueryString='false' />
		<serverVariables>
			<set name='REDIRECT_PHP404' value='folders' />
		</serverVariables>
	</rule>

	<!-- FILES case -->
	<rule name='Check FILES case' stopProcessing='true'>
		<match url='.*' />
		<conditions logicalGrouping='MatchAll'>
			<add input='{REDIRECT_PHP404}' pattern='^0$' />
			<add input='{REQUEST_FILENAME}' matchType='IsFile' />
			<add input='{REQUEST_FILENAME}' pattern='\.php(\?|$)' />
{$custom_files}
		</conditions>
		<action type='Rewrite' url='{R:0}' appendQueryString='false' />
		<serverVariables>
			<set name='REDIRECT_PHP404' value='files' />
		</serverVariables>
	</rule>

	<!-- CONTENT case -->
	<rule name='Check CONTENT case' stopProcessing='true'>
		<match url='.*' />
		<conditions logicalGrouping='MatchAll'>
			<add input='{REDIRECT_PHP404}' pattern='^0$' />
			<add input='{REQUEST_FILENAME}' matchType='IsFile' />
			<add input='{REQUEST_FILENAME}' pattern='\.php(\?|$)' />
			<add input='{REQUEST_URI}' pattern='^{$site_from}{$WP_CONTENT_DIR}/' />
{$custom_content}
		</conditions>
		<action type='Rewrite' url='{R:0}' appendQueryString='false' />
		<serverVariables>
			<set name='REDIRECT_PHP404' value='content' />
		</serverVariables>
	</rule>

	<!-- EXTS case -->
	<rule name='Check EXTS case' stopProcessing='true'>
		<match url='.*' />
		<conditions logicalGrouping='MatchAll'>
			<add input='{REDIRECT_PHP404}' pattern='^0$' />
			<add input='{REQUEST_FILENAME}' matchType='IsFile' />
			<add input='{REQUEST_FILENAME}' pattern='\.(jpg|jpeg|jpe|gif|png|bmp|tiff|tif|webp|avif|ico|heic|asf|asx|wmv|wmx|wm|avi|divx|flv|mov|qt|mpeg|mpg|mpe|mp4|m4v|ogv|webm|mkv|3gp|3gpp|3g2|3gp2|txt|asc|c|cc|h|srt|csv|tsv|ics|rtx)$' />
		</conditions>
		<action type='Rewrite' url='{R:0}' appendQueryString='false' />
		<serverVariables>
			<set name='REDIRECT_PHP404' value='exts' />
		</serverVariables>
	</rule>

	<!-- Handle REDIRECT_PHP404 -->
	<rule name='Handle 404 Error' stopProcessing='true'>
		<match url='.*' />
		<conditions>
			<add input='{REDIRECT_PHP404}' pattern='^(folders|files|content|exts)$' />
		</conditions>
		{$the_404}
	</rule>
</rules>";

	return trim( $rules );
}

add_action( 'admin_init', 'secupress_bad_url_set_expert' );
/**
 * Add our module to the global
 *
 * @since 2.3.17
 * @author Julio Potier
 **/
function secupress_bad_url_set_expert() {
	$GLOBALS['SECUPRESS_EXPERT_MODULES_ON']['bad_url_access'] = true;
}