<?php
/**
 * Module Name: Force Strong Encryption Pro
 * Description: Use a strong encryption system with a high cost value.
 * Main Module: users_login
 * Author: SecuPress
 * Version: 2.3.21
 */

defined( 'SECUPRESS_VERSION' ) or die( 'Something went wrong.' );

/** --------------------------------------------------------------------------------------------- */
/** Prevent Other Encryption System to Log In (prevent-low-encryption) ========================== */
/** --------------------------------------------------------------------------------------------- */

add_filter( 'check_password', 'secupress_force_strong_encryption_check_password', SECUPRESS_INT_MAX, 4 );
/**
 * Overwrite the wp_check_password() without declaring the function
 *
 * @since 2.3.21
 * @author Julio Potier
 * 
 * @param (bool) $check
 * @param (string) $password
 * @param (string) $hash
 * @param (int) $user_id
 * 
 * @return (bool) $check
 **/
function secupress_force_strong_encryption_check_password( 
	$check,
	#[\SensitiveParameter]
	$password,
	$hash,
	$user_id
) {
	if ( ! secupress_is_submodule_active( 'users-login', 'force-strong-encryption' ) ||
		! secupress_get_module_option( 'double-auth_prevent-low-encryption', 0, 'users-login' )
	) {
		return $check;
	}
	$algo = secupress_get_best_encryption_system();
	if ( wp_password_needs_rehash( $hash, $user_id ) ) {
		remove_all_actions( 'wp_set_password' ); // We do not want to trigger any hook/message/mail/alert.
		wp_set_password( $password, $user_id );
		$user = secupress_get_user_by( $user_id );
		if ( secupress_is_user( $user ) ) {
			$hash = $user->user_pass; // update the param now.
		}
	}

	if ( strlen( $hash ) <= 32 || // MD5 of worse.
		strlen( $password ) > 4096 || // Hack attempt.
		str_starts_with( $hash, '$P$' ) // Old WP pre 6.8 algo (PHPass/Blowfish).
	) {
		return false;
	}
	switch( $algo ) {
		case 'argon2id': // PASSWORD_ARGON2ID
		case 'argon2i':  // PASSWORD_ARGON2I
			$check  = password_verify( $password, $hash );
		break;
		case '2y': // PASSWORD_BCRYPT
			$verify = base64_encode( hash_hmac( 'sha384', $password, 'wp-sha384', true ) );
			$check  = password_verify( $verify, substr( $hash, 3 ) );
		break;
		default:
			$check  = apply_filters( 'secupress.plugins.force-strong-encryption.check_password.default', $check, $algo );
		break;
	}

	return $check;
}

/** --------------------------------------------------------------------------------------------- */
/** Prevent Password Hashes to be reused on other WP Sites (prevent-hash-reuse) ================= */
/** --------------------------------------------------------------------------------------------- */

add_action( 'wp_set_password', 'secupress_force_strong_encryption_blind_password_set_password', 9, 2 );
/**
 * Change the user's pass adding our suffix (when done manually)
 *
 * @since 2.3.21
 * @author Julio Potier
 * 
 * @see secupress_force_strong_encryption_remove_blind_password()
 * 
 * @param (string) $password
 * @param (int) $user_id
 **/
function secupress_force_strong_encryption_blind_password_set_password( 
	#[\SensitiveParameter]
	$password,
	$user_id
) {
	if ( ! secupress_is_submodule_active( 'users-login', 'force-strong-encryption' ) || 
		! secupress_get_module_option( 'double-auth_prevent-hash-reuse', 0, 'users-login' )
	) {
		return;
	}
	$key = secupress_generate_key_for_object( $user_id );
	update_user_meta( $user_id, 'secupress-blind-password', secupress_encrypt_secret_key( $key ) );
	remove_all_actions( 'wp_set_password' );
	wp_set_password( $password . $key, $user_id );
}

add_action( 'check_passwords', 'secupress_force_strong_encryption_blind_update_password', 10, 3 );
/**
 * Overwrite the given password (when updating the profile)
 *
 * @since 2.3.21
 * @author Julio Potier
 * 
 * @param (string) $username
 * @param (string) $pass1
 * @param (string) $pass2
 **/
function secupress_force_strong_encryption_blind_update_password( 
	$username, 
	#[\SensitiveParameter]
	&$pass1, 
	#[\SensitiveParameter]
	&$pass2
) {
	if ( ! secupress_is_submodule_active( 'users-login', 'force-strong-encryption' ) || 
		! secupress_get_module_option( 'double-auth_prevent-hash-reuse', 0, 'users-login' )
	) {
		return;
	}

	$_user  = secupress_get_user_by( $username );
	if ( ! secupress_is_user( $_user ) ) {
		return;
	}

	// @see edit_user()
	if ( empty( $pass1 ) || 
		str_contains( wp_unslash( $pass1 ), '\\' ) || 
		0 !== strcasecmp( $pass1, $pass2 )
	) {
		return;
	}

	$hash   = secupress_generate_key_for_object( $_user->ID );
	update_user_meta( $_user->ID, 'secupress-blind-password', $hash );
	$pass1 .= $hash;
	$pass2  = $pass1;
}

add_filter( 'authenticate', 'secupress_force_strong_encryption_manage_blind_password', 0, 3 );
/**
 * Handle our suffix on the user's password
 *
 * @since 2.3.21
 * @author Julio Potier
 * 
 * @param (WP_User|WP_Error) $user
 * @param (string) $username
 * @param (string) $password
 * 
 * @return 
 **/
function secupress_force_strong_encryption_manage_blind_password(
	$user,
	$username,
	#[\SensitiveParameter]
	$password
) {
	if ( ! secupress_is_submodule_active( 'users-login', 'force-strong-encryption' ) || 
		! secupress_get_module_option( 'double-auth_prevent-hash-reuse', 0, 'users-login' )
	) {
		return $user;
	}

	$_user   = secupress_get_user_by( $username );
	if ( ! secupress_is_user( $_user ) ) {
		return $user;
	}

	$suffix  = get_user_meta( $_user->ID, 'secupress-blind-password', true );
	$key     = secupress_generate_key_for_object( $_user->ID );
	$passwor = $password . $key;

	if ( ! $suffix ) {
		$needs_rehash = get_user_meta( $_user->ID, 'secupress-password-needs-rehash', true );
		if ( $needs_rehash ) {
			remove_filter( 'check_password', 'secupress_force_strong_encryption_check_password', SECUPRESS_INT_MAX );
		}
		if ( ! wp_check_password( $password, $_user->user_pass, $_user->ID ) ) {
			return $user;
		}
		if ( $needs_rehash ) {
			add_filter( 'check_password', 'secupress_force_strong_encryption_check_password', SECUPRESS_INT_MAX, 3 );
		}
		update_user_meta( $_user->ID, 'secupress-blind-password', secupress_encrypt_secret_key( $key ) );
		// Do not warn, this is not a hack, nobody has to know that now.
		remove_all_actions( 'wp_set_password' );
		wp_set_password( $passwor, $_user->ID );

		return $_user;
	} else {
		if ( wp_check_password( $passwor, $_user->user_pass, $_user->ID ) ) {

			return $_user;
		} else { // Wrong Blind Password Here
			secupress_log_attack( 'loginattempts' );
			return new WP_Error(
				'incorrect_password',
				sprintf(
					__( '<strong>Error:</strong> The password you entered for the username %s is incorrect.' ),
					secupress_tag_me( $username, 'strong' )
				) .
				' <a href="' . wp_lostpassword_url() . '">' . __( 'Lost your password?' ) . '</a>'
			);
		}
	}
}