<?php
/**
 * Module Name: Limit Password Spraying
 * Description: Limit passwords attempts and ban if too many tries have been done.
 * Main Module: users_login
 * Author: SecuPress
 * Version: 2.2.6
 */

defined( 'SECUPRESS_VERSION' ) or die( 'Something went wrong.' );

add_action( 'authenticate', 'secupress_passwordspraying', SECUPRESS_INT_MAX - 1, 3 );
/**
 * Check the number of attemps.
 *
 * @since 2.2.6
 * @author Julio Potier
 *
 * @param (null|object) $raw_user WP_User if the user is authenticated.
 *                      WP_Error or null otherwise.
 * @param (string)	    $username Username or email address.
 * @param (string)	    $password Password
 *
 * @return (null|object)
 */
function secupress_passwordspraying(
	$raw_user, 
	$username, 
	#[\SensitiveParameter]
	$password
) {
	static $done = false;

	if ( $done || ! $password || ! $username ) {
		return $raw_user;
	}
	$done = true;

	global $wpdb;

	if ( empty( $_POST ) || ! is_wp_error( $raw_user ) || false === ( $uid = username_exists( $username ) ) || secupress_ip_is_whitelisted() ) { // WPCS: CSRF ok.
		if ( ! empty( $raw_user->ID ) ) {
			delete_user_meta( $raw_user->ID, '_secupress_passwordspraying' );
		}

		return $raw_user;
	}

	$wpdb->query( 'START TRANSACTION' );

	$wpdb->query( $wpdb->prepare( "INSERT INTO $wpdb->usermeta (user_id, meta_key, meta_value) VALUES (%d, '_secupress_passwordspraying', %s)", $uid, $password ) );

	$last_tries   = secupress_get_user_metas( '_secupress_passwordspraying' );
	$last_tries   = is_array( $last_tries ) ? $last_tries : [];
	/**
	* Modify the levenshtein score to match your language maybe
	* Do not try 0 or 1, 0.65 (about one third max) recommended
	* 
	* @param (float) $lev_score
	* @since 2.2.6
	* @author Julio Potier
	* @return (float)
	*
	*/
	$lev_score    = (float) apply_filters( 'secupress.plugins.passwordspraying.levenshtein_score', 0.65 );
	/**
	* Modify the levenshtein matches threshold
	* 
	* @param (int) $matches
	* @since 2.2.6
	* @author Julio Potier
	* @return (int)
	*
	*/
	$max_matches  = (int) apply_filters( 'secupress.plugins.passwordspraying.max_matches', 3 );
	$matches      = array_filter( wp_list_pluck( $last_tries, 'meta_value' ),
					function( $last_try ) use ( $password, $lev_score ) {
						return secupress_levenshtein( $password, $last_try ) > $lev_score;
					} );

	if ( count( $matches ) > $max_matches ) {
		foreach ( $last_tries as $try ) {
			if ( array_search( $try['meta_value'], $matches ) !== false ) {
				$wpdb->query( $wpdb->prepare( "DELETE FROM $wpdb->usermeta WHERE user_id = %d AND meta_key = '_secupress_passwordspraying' AND meta_value = %s", $try['user_id'], $try['meta_value'] ) );
			}
		}
		secupress_ban_ip( (int) secupress_get_module_option( 'login-protection_time_ban', 5, 'users-login' ), null, [ 'die' => true, 'attack_type' => 'passwordspraying' ] );
	}

	$wpdb->query( 'COMMIT' );

	return $raw_user;
}
