<?php
defined( 'SECUPRESS_VERSION' ) or die( 'Something went wrong.' );

add_action( 'secupress.plugins.move-login.deny_login_access', 'secupress_pro_move_login_deny_login_access' );
/**
 * Add custom behaviour for move login
 *
 * @since 2.0
 * @author Julio Potier
 *
 **/
function secupress_pro_move_login_deny_login_access() {

	$setting       = secupress_get_module_option( 'move-login_whattodo', 'sperror', 'users-login' );
	if ( 'honeypot' === $setting && ! secupress_is_honeypotable() ) {
		$setting = 'sperror';
	}
	$error_message = '<p>' . __( 'This page does not exist, has moved or you are not allowed to access it.', 'secupress' ) . '</p>';
	$form          = true;
	
	switch ( $setting ) {
		case 'custom_page':
			$url  = secupress_get_module_option( 'move-login_custom_page_url', home_url(), 'users-login' );
			$page = wp_validate_redirect( $url, home_url() );
			wp_redirect( $page, 301 );
			die();
		break;

		case 'custom_error':
			$setting_message = wp_kses_post( secupress_get_module_option( 'move-login_custom_error_content', '', 'users-login' ) );
			$error_message   = ! empty( $setting_message ) ? $setting_message : $error_message;
		break;

		case 'honeypot':
			global $sp_action;
			$sp_action     = 'honeypot';
			$wp_login_form = wp_login_form( [ 'echo' => false ] );
			$banme_nonce   = wp_create_nonce( 'ban_me_please-' . date( 'ymdhi' ) );
			$wp_login_form = str_replace( 'action="' . wp_login_url(), 'action="' . home_url() . '/login?action=login', $wp_login_form );
			$wp_login_form = str_replace( '</form>', '<input type="hidden" name="token" value="' . $banme_nonce . '" /></form>', $wp_login_form );
			remove_action( 'login_init', 'secupress_move_login_maybe_deny_login_page', 0 );
			secupress_login_page( __( 'Log In', 'secupress' ), $wp_login_form );
		break;
	}
	echo '<style>';
	include( ABSPATH . WPINC . '/js/tinymce/skins/wordpress/wp-content.css' );
	echo '</style>';

	secupress_die( secupress_check_ban_ips_form( [ 'content'  => $error_message ] ), '', [ 'force_die' => true, 'attack_type' => 'move_login' ] );

}

add_action( 'admin_post_send_passwordless_validation_link', 'secupress_send_passwordless_validation_link' );
/**
 * Resend the validation link when activating passwordless
 *
 * @since 2.2.6
 * @author Julio Potier
**/
function secupress_send_passwordless_validation_link() {
	if ( ! isset( $_GET['_wpnonce'] ) || empty( $_GET['_wpnonce'] ) || ! check_admin_referer( 'send_passwordless_validation_link' ) ) {
		wp_die( 'Something went wrong.' );
	}
	secupress_passwordless_activation_validation_mail();
	wp_safe_redirect( wp_get_referer() );
	die();
}


/**
 * Get the user capability or role to use this plugin.
 *
 * @since 2.2.6 Deprecated
 * @since 1.0
 * @author Grégory Viguier
 *
 * @return (string) A user capability or role.
 */
function secupress_pro_sessions_control_get_capability() {
	_deprecated_function( __FUNCTION__, '2.2.6', "secupress_get_capability( false, 'sessions_control' )" );
	$capability = secupress_get_capability( false, 'sessions_control' );
	/**
	 * Filter the user capability or role.
	 *
	 * @since 2.2.6 Deprecated
	 * @since 1.0
	 *
	 * @param (string) $capability A user capability or role.
	 */
	if ( has_filter( 'secupress_pro.plugin.sessions_control.capability' ) ) {
		_deprecated_hook( 'secupress_pro.plugin.sessions_control.capability', '2.2.6', 'see secupress_get_capability()' );
	}
	return apply_filters( 'secupress_pro.plugin.sessions_control.capability', $capability );
}

/**
 * Get user IDs with active sessions.
 *
 * @since 2.2.6
 * @author Julio Potier
 *
 * @return (array) User IDs with active sessions.
 */
function secupress_pro_sessions_control_get_connected_user_ids() {
	static $user_ids = [];
	$user_ids = $user_ids ?? [];

	if ( ! empty( $user_ids ) ) {
		return $user_ids;
	}

    // Get all user IDs
    $users = get_users( [
	    'meta_key'     => 'session_tokens',
	    'meta_compare' => 'EXISTS'
	] );

    foreach ($users as $user) {
        $instance = WP_Session_Tokens::get_instance( $user->ID );
        $sessions = $instance->get_all();
        foreach ( $sessions as $session ) {
            if ( $session['expiration'] > time() ) {
                $user_ids[] = $user->ID;
                break; // No need to check further tokens for this user
            }
        }
    }

    return $user_ids;
}


/**
 * Check the verification code entered by the user.
 * @see https://github.com/emreakay/CodeIgniter-Aauth/blob/master/application/helpers/googleauthenticator_helper.php#L42
 */
function secupress_base32_verify( $secretkey, $thistry, $lasttimeslot ) {

	$shortcut = apply_filters( 'secupress.otp-auth.base32_verify', false, $thistry );
	if ( $shortcut ) {
		return true;
	}

	require_once( dirname( __FILE__ ) . '/plugins/inc/php/base32.php' );

	$tm        = floor( time() / 30 );
	$b32       = new Base32();
  	$secretkey = $b32->base32_decode( $secretkey );

	// Key from 30 seconds before is also valid.
	for ( $i = -1; $i <= 0; $i++ ) {
		// Pack time into binary string
		$time = chr( 0 ).chr( 0 ).chr( 0 ).chr( 0 ).pack( 'N*', $tm + $i );
		// Hash it with users secret key
		$hm = hash_hmac( 'SHA1', $time, $secretkey, true );
		// Use last nipple of result as index/offset
		$offset = ord( substr( $hm, -1 ) ) & 0x0F;
		// grab 4 bytes of the result
		$hashpart = substr( $hm, $offset, 4 );
		// Unpak binary value
		$value = unpack( "N", $hashpart );
		$value = $value[1];
		// Only 32 bits
		$value = $value & 0x7FFFFFFF;
		$value = $value % 1000000;
		if ( hash_equals( (string) $value, $thistry ) ) {
			// Check for replay (Man-in-the-middle) attack.
			if ( $lasttimeslot >= ( $tm + $i ) ) {
				do_action( 'secupress.otp-auth.mitm', $secretkey );
				return false;
			}
			// Return timeslot in which login happened.
			return $tm+$i;
		}
	}

	return false;
}


/**
 * Use this to send a passwordless link to anyone.
 *
 * @since 2.2.1
 * @author Julio Potier
 * 
 * @param (WP_User|int|string) $raw_user
 * @param (array) $args [ 'redirect_to' => (string), 'rememberme' => 0|1, 'sendmail' => (bool) ]
 * 
 * @return (array)
 **/
function secupress_passwordless_send_link( $raw_user, $args = [] ) {
	// Step 1 for everybody: if not a user yet, try to find them by ID, username or email.
	$raw_user  = secupress_get_user_by( $raw_user );
	if ( ! secupress_is_user( $raw_user ) ) {
		return [ 'success' => false, 'is_user' => false ];
	}

	$defaults  = [ 'redirect_to' => '', 'rememberme' => 0, 'sendmail' => true ];
	$args      = wp_parse_args( $args, $defaults );
	// // Step 1 succeeded: generate a token.
	remove_all_filters( 'random_password' );
	$key       = wp_generate_password( 32, false );
	update_user_meta( $raw_user->ID, 'secupress_passwordless_token', $key );

	$timing = apply_filters( 'secupress.plugin.passwordless.timing', 10 );
	if ( has_filter( 'secupress.plugin.passwordless.timing' ) ) {
		_deprecated_hook( 'secupress.plugin.passwordless.timing', '2.2.6', 'secupress.plugins.passwordless.timing' );
	}
	/**
	 * Filter the delay of validity of the email
	 *
	 * @since 2.0
	 *
	 * @param (int) $timing In minutes
	 */
	$timing    = apply_filters( 'secupress.plugins.passwordless.timing', $timing );
	update_user_meta( $raw_user->ID, 'secupress_passwordless_timeout', time() + ( $timing * MINUTE_IN_SECONDS ) );
	update_user_meta( $raw_user->ID, 'secupress_passwordless_rememberme', (int) $args['rememberme'] );

	$token_url = admin_url( 'admin-post.php?action=passwordless_autologin&token=' . $key . ( $args['redirect_to'] ? '&redirect_to=' . rawurlencode( $args['redirect_to'] ) : '' ) );

	if ( ! $args['sendmail'] ) {
		return [ 'success' => true, 'is_user' => true, 'WP_User' => $raw_user, 'mailsent' => false, 'token_url' => $token_url ];
	}
	
	$subject    = sprintf( __( '[%s] Your Magic Link for a Secure Login (2FA)', 'secupress' ), '###SITENAME###' );
	$subject    = apply_filters( 'secupress.plugin.passwordless_email_subject', $subject, $raw_user );
	if ( has_filter( 'secupress.plugin.passwordless_email_subject' ) ) {
		_deprecated_hook( 'secupress.plugin.passwordless_email_subject', '2.2.6', 'secupress.plugins.passwordless_email_subject' );
	}
	/**
	 * Filter the subject of the mail sent to the user.
	 *
	 * @since 2.0 $raw_user parameter
	 * @since 1.0
	 *
	 * @param (string) $subject The email subject.
	 * @param (WP_User) $raw_user The user
	 */
	$subject    = apply_filters( 'secupress.plugins.passwordless_email_subject', $subject, $raw_user );
	$body       = sprintf(
		/** Translators: 1 is a user name, 2 is a URL, 3 is an email address. */
		__( 'Hello %1$s,

You recently tried to log in on ###SITEURL###.

If this is correct, please click on the following Magic Link to really log in:
%2$s

You can safely ignore and delete this email if you do not want to take this action.

Regards,
All at ###SITENAME###
###SITEURL###', 'secupress' ),
		esc_html( $raw_user->display_name ),
		esc_url_raw( $token_url ),
	);

	$body = apply_filters( 'secupress.plugin.passwordless_email_message', $body, $raw_user, $token_url );
	if ( has_filter( 'secupress.plugin.passwordless_email_message' ) ) {
		_deprecated_hook( 'secupress.plugin.passwordless_email_message', '2.2.6', 'secupress.plugins.passwordless_email_message' );
	}
	/**
	 * Filter the body of the mail sent to the user.
	 * You can use ###ADMIN_EMAIL###, ###SITENAME###, ###SITEURL### as placeholder if needed.
	 *
	 * @since 2.0 $rawuser and $token_url parameters
	 * @since 1.0
	 *
	 * @param (string) $body The email body.
	 * @param (WP_User) $raw_user The user
	 * @param (string) $token_url The url to log-in
	 */
	$body = apply_filters( 'secupress.plugins.passwordless_email_message', $body, $raw_user, $token_url );
	
	$sent = secupress_send_mail( $raw_user->user_email, $subject, $body );

	// 'mailsent' => true, even if not really sent, this is to separate from the case where we do not want to send it.
	return [ 'success' => true, 'is_user' => true, 'WP_User' => $raw_user, 'mailsent' => true, 'token_url' => $token_url ];
}

add_action( 'login_form_passwordless_autologin', 'secupress_passwordless_autologin_validation' );
/**
 * Modify the login header page message for our action
 *
 * @since 1.0
 * @author Julio Potier
 */
function secupress_passwordless_autologin_validation() {
	login_header( __( 'Log In', 'secupress' ), '<p class="message">' . __( 'Please check your email for the Magic Link needed to log in.', 'secupress' ) . '</p>' );
	login_footer();
	die();
}



add_action( 'admin_post_passwordless_autologin',        'secupress_passwordless_autologin' );
add_action( 'admin_post_nopriv_passwordless_autologin', 'secupress_passwordless_autologin' );
/**
 * Automatically log-in a user with the correct token.
 *
 * @since 1.0
 * @author Julio Potier
 */
function secupress_passwordless_autologin() {
	$user_id            = 0;
	$fallback_error_msg = sprintf( __( 'This link is not valid for this user, please try to %slog-in again%s.', 'secupress' ), '<a href="' . esc_url( wp_login_url( '', true ) ) . '">', '</a>' );

	if ( empty( $_GET['token'] ) ) {
		$token = '';

		if ( has_action( 'secupress.plugin.passwordless.autologin_error' ) ) {
			_deprecated_hook( 'secupress.plugin.passwordless.autologin_error', '2.2.6', 'secupress.plugins.passwordless.autologin_error' );
		}
		do_action( 'secupress.plugin.passwordless.autologin_error', $user_id, $token, 'no token' );
		/**
		 * Triggers an action when an autologin action from the module has failed.
		 *
		 * @since 1.0
		 *
		 * @param (int)    $user_id    The user ID.
		 * @param (string) $token      The security token.
		 * @param (string) $error_code The error code.
		 */
		do_action( 'secupress.plugins.passwordless.autologin_error', $user_id, $token, 'no token' );

		secupress_die( $fallback_error_msg, '', array( 'force_die' => true ) );
	}

	// Get the user with the given token.
	$token   = $_GET['token'];
	// Prevent plugins to filter the users (like "Advanced Access Manager" does …)
	remove_all_actions( 'pre_get_users' );
	// Get the only user with that token
	$user_id = get_users( array(
		'meta_key'   => 'secupress_passwordless_token',
		'meta_value' => $token,
		'fields'     => 'ID',
		'number'     => 2, // 2, not 1!
	) );
	$user_id = count( $user_id ) === 1 ? (int) reset( $user_id ) : 0;
	$user    = $user_id ? get_user_by( 'id', $user_id ) : false;

	if ( ! secupress_is_user( $user ) ) {
		if ( has_action( 'secupress.plugin.passwordless.autologin_error' ) ) {
			_deprecated_hook( 'secupress.plugin.passwordless.autologin_error', '2.2.6', 'secupress.plugins.passwordless.autologin_error' );
		}
		do_action( 'secupress.plugin.passwordless.autologin_error', $user_id, $token, 'no user' );
		/** This action is documented in inc/modules/users-login/plugins/passwordless.php. */
		do_action( 'secupress.plugins.passwordless.autologin_error', $user_id, $token, 'no user' );

		secupress_die( $fallback_error_msg, '', array( 'force_die' => true ) );
	}

	// Test token validity period.
	$requested_redirect_to = ! empty( $_GET['redirect_to'] ) ? rawurldecode( $_GET['redirect_to'] ) : '';
	$time                  = get_user_meta( $user_id, 'secupress_passwordless_timeout', true );
	$rememberme            = get_user_meta( $user_id, 'secupress_passwordless_rememberme', true );

	delete_user_meta( $user_id, 'secupress_passwordless_token' );
	delete_user_meta( $user_id, 'secupress_passwordless_rememberme' );
	delete_user_meta( $user_id, 'secupress_passwordless_timeout' );

	if ( $time < time() ) {
		// The 10 minutes limit has passed.
		if ( has_action( 'secupress.plugin.passwordless.autologin_error' ) ) {
			_deprecated_hook( 'secupress.plugin.passwordless.autologin_error', '2.2.6', 'secupress.plugins.passwordless.autologin_error' );
		}
		do_action( 'secupress.plugin.passwordless.autologin_error', $user_id, $token, 'expired token' );
		/** This action is documented in inc/modules/users-login/plugins/passwordless.php. */
		do_action( 'secupress.plugins.passwordless.autologin_error', $user_id, $token, 'expired token' );

		$message = sprintf( __( 'This link is now expired, please try to <a href="%s">log-in again</a>.', 'secupress' ), esc_url( wp_login_url( $requested_redirect_to, true ) ) );
		secupress_die( $message, '', array( 'force_die' => true ) );
	}

	// Log in and redirect the user.
	$secure_cookie = secupress_server_is_ssl();
	$secure_args   = array(
		'user_login'    => $user->user_login,
		'user_password' => time(), // We don't have the real password, just pass something.
	);

	/** This filter is documented in wp-includes/user.php. */
	$secure_cookie = apply_filters( 'secure_signon_cookie', $secure_cookie, $secure_args );

	wp_set_auth_cookie( $user_id, (bool) $rememberme, $secure_cookie );

	if ( $requested_redirect_to ) {
		$redirect_to = $requested_redirect_to;
		// Redirect to https if user wants ssl.
		if ( $secure_cookie && false !== strpos( $redirect_to, 'wp-admin' ) ) {
			$redirect_to = preg_replace( '|^http://|', 'https://', $redirect_to );
		}
	} else {
		$redirect_to = admin_url( 'index.php' );
	}

	// Add 'index.php" to prevent infinite loop of login.
	if ( $redirect_to === admin_url() ) {
		$redirect_to = admin_url( 'index.php' );
	}

	/** This filter is documented in wp-login.php. */
	$redirect_to = apply_filters( 'login_redirect', $redirect_to, $requested_redirect_to, $user );

	if ( has_action( 'secupress.plugin.passwordless.autologin_success' ) ) {
		_deprecated_hook( 'secupress.plugin.passwordless.autologin_success', '2.2.6', 'secupress.plugins.passwordless.autologin_success' );
	}
	do_action( 'secupress.plugin.passwordless.autologin_success', $user, $token );
	/**
	 * Triggers an action when an autologin action from the module is a success.
	 *
	 * @since 1.0
	 *
	 * @param (int)    $user    The user
	 * @param (string) $token   The security token.
	 */
	do_action( 'secupress.plugins.passwordless.autologin_success', $user, $token );

	wp_safe_redirect( esc_url_raw( $redirect_to ) );
	die();
}

/**
 * Callback to destroy all sessions for all users (no JS).
 *
 * @since 2.2.6 $force
 * @author Julio Potier
 * @author Grégory Viguier
 * 
 * @param (bool) $keep_current_user_session true to keep the current user session
 */
function secupress_pro_sessions_control_destroy_all_sessions( $keep_current_user_session = false ) {
	// Get current user session.
	if ( $keep_current_user_session ) {
		$sessions_inst   = WP_Session_Tokens::get_instance( get_current_user_id() );
		$token_to_keep   = wp_get_session_token();
		$session_to_keep = $sessions_inst->get( $token_to_keep );
	}

	/** This filter is documented in /wp-includes/session.php */
	$manager = apply_filters( 'session_token_manager', 'WP_User_Meta_Session_Tokens' );
	call_user_func( array( $manager, 'destroy_all_for_all_users' ) );

	// Recreate current session.
	if ( $keep_current_user_session ) {
		$sessions_inst->update( $token_to_keep, $session_to_keep );
	}
}


/**
 * Handles sending password retrieval email to user.
 *
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @see retrieve_password()
 * 
 * @param (WP_User) $user_data
 * @param (bool) $with_email true will send the password reset email
 * 
 * @return (bool)
 */
function secupress_pro_reset_passwords( $user_data, $with_email ) { 
	$key      = get_password_reset_key( $user_data );

	wp_set_password( wp_generate_password( 64 ), $user_data->ID ); // Force password change

	if ( is_wp_error( $key ) ) {
		return false;
	}

	if ( ! $with_email ) {
		return true;
	}

	// with email
	// Redefining user_login ensures we return the right case in the email.
	$user_login = $user_data->user_login;
	$user_email = $user_data->user_email;
	if ( is_multisite() ) {
		$site_name = get_network()->site_name;
	} else {
		/*
		 * The blogname option is escaped with esc_html on the way into the database
		 * in sanitize_option we want to reverse this for the plain text arena of emails.
		 */
		$site_name = wp_specialchars_decode( get_option( 'blogname' ), ENT_QUOTES );
	}

	$message = __( 'A person from the website administration has requested a password reset for your account:', 'secupress' ) . "\r\n\r\n";
	/* translators: %s: Site name. */
	$message .= sprintf( __( 'Site Name: %s', 'secupress' ), $site_name ) . "\r\n\r\n";
	/* translators: %s: User login. */
	$message .= sprintf( __( 'Username: %s', 'secupress' ), $user_login ) . "\r\n\r\n";
	$message .= __( 'To reset your password, visit the following address:', 'secupress' ) . "\r\n\r\n";
	$message .= network_site_url( "wp-login.php?action=rp&key=$key&login=" . rawurlencode( $user_login ), 'login' ) . "\r\n";

	/* translators: Password reset notification email subject. %s: Site title. */
	$title = sprintf( __( '[%s] Mandatory Password Reset', 'secupress' ), $site_name );

	/**
	 * Filters the subject of the password reset email.
	 *
	 * @since 2.2.6
	 *
	 * @param string  $title      Default email title.
	 * @param string  $user_login The username for the user.
	 * @param WP_User $user_data  WP_User object.
	 */
	$title = apply_filters( 'secupress.retrieve_password_title', $title, $user_login, $user_data );

	/**
	 * Filters the message body of the password reset mail.
	 *
	 * If the filtered message is empty, the password reset email will not be sent.
	 *
	 * @since 2.2.6
	 *
	 * @param string  $message    Default mail message.
	 * @param string  $key        The activation key.
	 * @param string  $user_login The username for the user.
	 * @param WP_User $user_data  WP_User object.
	 */
	$message = apply_filters( 'secupress.retrieve_password_message', $message, $key, $user_login, $user_data );

	return wp_mail( $user_email, wp_specialchars_decode( $title ), $message );
}