<?php
namespace WPSecurityNinja\Plugin;

if (!defined('ABSPATH')) {
	exit;
}

define('WF_SN_WL_OPTIONS_KEY', 'wf_sn_wl');


/**
 * Class Wf_Sn_Wl
 *
 * Handles white labeling functionality for the Security Ninja plugin.
 *
 * @package WPSecurityNinja\Plugin
 */
class Wf_Sn_Wl
{

	public static $options = null;


	/**
	 * Initializes the white labeling functionality.
	 *
	 * @return void
	 */
	public static function init()
	{
		self::$options = self::get_options();

		if (is_admin()) {
			// add tab to Security Ninja tabs
			add_filter('sn_tabs', array(__NAMESPACE__ . '\Wf_Sn_Wl', 'sn_tabs'));
			add_action('admin_init', array(__NAMESPACE__ . '\Wf_Sn_Wl', 'admin_init'));
			// check and set default settings
			self::default_settings(false);
			
			// Enqueue media scripts for the whitelabel page
			add_action('admin_enqueue_scripts', array(__NAMESPACE__ . '\Wf_Sn_Wl', 'enqueue_media_scripts'));
		}

		if (self::is_active()) {

			add_action('admin_enqueue_scripts', array(__NAMESPACE__ . '\Wf_Sn_Wl', 'updates_core_page'));
			add_action('admin_enqueue_scripts', array(__NAMESPACE__ . '\Wf_Sn_Wl', 'do_action_admin_head_add_extra_css'));
			
			// Only apply display modifications on specific admin pages, not for update checks
			global $pagenow;
			if ($pagenow && !in_array($pagenow, array('update.php', 'update-core.php'))) {
				add_filter('all_plugins', array(__NAMESPACE__ . '\Wf_Sn_Wl', 'do_filter_all_plugins'), 9999);
			}

			add_action('pre_current_active_plugins', array(__NAMESPACE__ . '\Wf_Sn_Wl', 'action_pre_current_active_plugins'));

			// Add filter for plugin action links
			add_filter('plugin_action_links', array(__NAMESPACE__ . '\Wf_Sn_Wl', 'customize_plugin_action_links'), PHP_INT_MAX, 2);
		}
	}

	/**
	 * Removes plugin from list.
	 *
	 * @author   Lars Koudal
	 * @since    v0.0.1
	 * @version  v1.0.0  Sunday, January 3rd, 2021.
	 * @access   public static
	 * @param array $plugins List of all plugins.
	 * @return array Modified list of plugins.
	 */
	public static function do_filter_all_plugins($plugins)
	{
		$keys = array('security-ninja-premium/security-ninja.php', 'security-ninja/security-ninja.php');
		foreach ($keys as $key) {
			if (isset($plugins[$key])) {
				// Get whitelabel values
				$new_name = self::get_new_name();
				$new_url = self::get_new_url();
				$new_author_name = self::get_new_author_name();
				$new_desc = self::get_new_desc();
				$new_icon_url = self::get_new_icon_url();
				
				// Update plugin properties
				if ($new_name) {
					$plugins[$key]['Name'] = $new_name;
					$plugins[$key]['Title'] = $new_name;
				}
				
				if ($new_url) {
					$plugins[$key]['PluginURI'] = $new_url;
					$plugins[$key]['AuthorURI'] = $new_url;
				}
				
				if ($new_author_name) {
					$plugins[$key]['Author'] = $new_author_name;
					$plugins[$key]['AuthorName'] = $new_author_name;
				}
				
				if ($new_desc) {
					$plugins[$key]['Description'] = $new_desc;
				}
				
				// Add icon if available
				if ($new_icon_url) {
					$plugins[$key]['icons'] = array(
						'default' => $new_icon_url
					);
				}
			}
		}
		return $plugins;
	}





	/**
	 * Outputs simple CSS to hide the plugin icon.
	 *
	 * @author   Lars Koudal
	 * @since    v0.0.1
	 * @version  v1.0.0  Sunday, January 3rd, 2021.
	 * @access   public static
	 * @return void
	 */
	public static function do_action_admin_head_add_extra_css()
	{
		if (!self::is_active()) {
			return;
		}
		wp_register_style('admin-custom-style', false, array(), \WPSecurityNinja\Plugin\Utils::get_plugin_version());
		wp_enqueue_style('admin-custom-style');
		wp_add_inline_style('admin-custom-style', self::get_custom_css());
	}

	/**
	 * enqueue_custom_admin_style.
	 *
	 * @author	Lars Koudal
	 * @since	v0.0.1
	 * @version	v1.0.0	Tuesday, May 7th, 2024.
	 * @access	public static
	 * @return void
	 */
	public static function enqueue_custom_admin_style()
	{
		wp_register_style('admin-extra-sec-styling', false, array(), \WPSecurityNinja\Plugin\Utils::get_plugin_version());
		wp_enqueue_style('admin-extra-sec-styling');
		wp_add_inline_style('admin-extra-sec-styling', self::get_custom_css());
	}


	/**
	 * get_custom_css.
	 *
	 * @author  Lars Koudal
	 * @since   v0.0.1
	 * @version v1.0.0  Wednesday, April 24th, 2024.
	 * @access  private static
	 * @global
	 * @return string Custom CSS.
	 */
	private static function get_custom_css()
	{
		return '/* Set max width of menu icon */
.menu-top.toplevel_page_wf-sn img,
#toplevel_page_wf-sn .wp-menu-image img {max-width:20px;}
#security-ninja-update {display:none;}
tr[data-slug="security-ninja"] .open-plugin-details-modal {display:none;}';
	}

	/**
	 * Update strings on the update-core.php page.
	 *
	 * @author	Lars Koudal
	 * @since	v0.0.1
	 * @version	v1.0.0	Monday, May 20th, 2024.
	 * @access	public static
	 * @return void
	 */
	public static function updates_core_page()
	{
		global $pagenow;

		if ('update-core.php' === $pagenow) {
			$default_names = ['Security Ninja', 'Security Ninja (Premium)'];
			$newtitle = self::get_new_name();
			$newicon = self::get_new_icon_url();

			if (false !== $newtitle) {
				// Escaping for security
				$newtitle_js = esc_js($newtitle);
				$default_names_js = json_encode(array_map('esc_js', $default_names));

				// Conditionally set the new icon JS
				$newicon_js = $newicon !== false ? esc_js($newicon) : '';

				wp_add_inline_script(
					'updates',
					"
                var _secnin_default_names = $default_names_js;
                var _secnin_branded_name = '$newtitle_js';
                var _secnin_icon_url = '$newicon_js';
                
                // Replace or update image
                document.querySelectorAll( '#update-plugins-table .plugin-title img[src*=\'security-ninja\']' )
                .forEach(function(plugin) {
                    if (_secnin_icon_url) {
                        jQuery(plugin).attr('src', _secnin_icon_url).attr('width', 18);
                    }
                });
                
                // Renames plugin title
                document.querySelectorAll( '#update-plugins-table .plugin-title strong' )
                .forEach(function(plugin) {
                    _secnin_default_names.forEach(function(default_name) {
                        if (default_name === plugin.innerText) {
                            plugin.innerText = _secnin_branded_name;
                        }
                    });
                });
                "
				);
			}
		}
	}



	/**
	 * Hides the plugin from list of active plugins
	 *
	 * @author   Lars Koudal
	 * @since    v0.0.1
	 * @version  v1.0.0  Sunday, January 3rd, 2021.
	 * @access   public static
	 * @return void
	 */
	public static function action_pre_current_active_plugins()
	{
		global $wp_list_table;
		$hidearr   = array('security-ninja/security-ninja.php', 'security-ninja-premium/security-ninja.php');
		$myplugins = $wp_list_table->items;
		foreach ($myplugins as $key => $val) {
			if (in_array($key, $hidearr, true)) {

				$new_name        = self::get_new_name();
				$new_url         = self::get_new_url();
				$new_author_name = self::get_new_author_name();
				$new_desc        = self::get_new_desc();
				$wl_newiconurl   = self::get_new_icon_url();

				if ($wl_newiconurl) {
					$wp_list_table->items[$key]['icons']['default'] = $wl_newiconurl;
				}

				if ($new_name) {
					$wp_list_table->items[$key]['Name'] = $new_name;
				}
				if ($new_name) {
					$wp_list_table->items[$key]['Title'] = $new_name;
				}

				if ($new_author_name) {
					$wp_list_table->items[$key]['Author'] = $new_author_name;
				}
				if ($new_author_name) {
					$wp_list_table->items[$key]['AuthorName'] = $new_author_name;
				}

				if ($new_url) {
					$wp_list_table->items[$key]['PluginURI'] = $new_url;
				}
				if ($new_url) {
					$wp_list_table->items[$key]['AuthorURI'] = $new_url;
				}

				if ($new_desc) {
					$wp_list_table->items[$key]['Description'] = $new_desc;
				}
			}
		}
	}


	/**
	 * get_options.
	 *
	 * @author   Lars Koudal
	 * @since    v0.0.1
	 * @version  v1.0.0  Sunday, January 3rd, 2021.
	 * @access   public static
	 * @return array White label options.
	 */
	public static function get_options()
	{

		if (!is_null(self::$options)) {
			return self::$options;
		}

		$defaults = array(
			'wl_active'         => '0',
			'wl_newname'        => 'Security Ninja',
			'wl_newdesc'        => '',
			'wl_newauthor'      => '',
			'wl_newurl'         => 'https://wpsecurityninja.com/',
			'wl_newiconurl'     => '',
			'wl_newmenuiconurl' => '',
		);
		$options  = get_option('wf_sn_wl', array());
		$options  = array_merge($defaults, $options);
		return $options;
	}



	/**
	 * add new tab
	 *
	 * @author   Lars Koudal
	 * @since    v0.0.1
	 * @version  v1.0.0  Sunday, January 3rd, 2021.
	 * @access   public static
	 * @param array $tabs Existing tabs.
	 * @return array Modified tabs.
	 */
	public static function sn_tabs($tabs)
	{

		$whitelabel_tab = array(
			'id'       => 'sn_whitelabel',
			'class'    => '',
			'label'    => esc_html__('White label', 'security-ninja'),
			'callback' => array(__NAMESPACE__ . '\\Wf_Sn_Wl', 'do_page'),
		);

		// Check if active and then remove the tab
		if (self::is_active()) {

			$whitelabel_tab = array(
				'id'       => 'sn_whitelabel',
				'class'    => 'hide',
				'label'    => esc_html__('White label', 'security-ninja'),
				'callback' => array(__NAMESPACE__ . '\\Wf_Sn_Wl', 'do_page'),
			);

			$tabs[] = $whitelabel_tab;
			return $tabs;
		}

		$done     = 0;
		$tabcount = count($tabs);
		for ($i = 0; $i < $tabcount; $i++) {
			if ('sn_whitelabel' === $tabs[$i]['id']) {
				$tabs[$i] = $whitelabel_tab;
				$done       = 1;
				break;
			}
		} // for
		if (!$done) {
			$tabs[] = $whitelabel_tab;
		}
		return $tabs;
	}



	/**
	 * Display admin page
	 *
	 * @author   Lars Koudal
	 * @since    v0.0.1
	 * @version  v1.0.0  Sunday, January 3rd, 2021.
	 * @access   public static
	 * @return void
	 */
	public static function do_page()
	{
		global $wpdb, $secnin_fs;

?>
		<div class="submit-test-container sncard settings-card">
			<h2><span class="dashicons dashicons-id"></span> <?php esc_html_e('White label', 'security-ninja'); ?></h2>
		<p>Customize the plugin branding to match your own.</p>
		<?php
			echo '<form action="options.php" method="post">';
			settings_fields('wf_sn_wl');
			echo '<table class="form-table"><tbody>';
			echo '<tr valign="top">
				<th scope="row"><label for="wf_sn_wl_wl_active"><h3>' . esc_html__('Enable whitelabel', 'security-ninja').'</h3>';
				echo '<p class="description">' . esc_html__('This option allows you to white label the plugin, customizing it with your own branding.', 'security-ninja') . '</p></br>';
				echo '</label></th>
				<td class="sn-cf-options">';

			\WPSecurityNinja\Plugin\Utils::create_toggle_switch(
				'wf_sn_wl_wl_active',
				array(
					'saved_value' => self::$options['wl_active'],
					'option_key'  => 'wf_sn_wl[wl_active]',
				)
			);
			echo '</td></tr>';

			echo '<tr><td>';
echo '<div class="sncard warning">';
			echo '<h3>' . esc_html__('Warning - Enabling white labeling will make this tab disappear.', 'security-ninja') . '</h3>';
			
			echo '<p>'. esc_html__('To disable white labeling or change the branding, you will need to manually navigate to a specific URL in your browser and reload the page.', 'security-ninja') . '</p>';
			echo '<p><strong>' . esc_html__('Bookmark the following URL to easily access the white label settings again:', 'security-ninja') . '</strong></p>';

			?>
			<input type="text" value="<?php echo esc_url(admin_url('admin.php?page=wf-sn')); ?>#sn_whitelabel" disabled>
		</div>	
		</td>
			</tr>
			<tr>
				<th scope="row"><label for="input_id"><h3><?php echo esc_html__('Plugin Name', 'security-ninja'); ?></h3>
			
			</label></th><td></td></tr>
			<tr>
				<td class="fullwidth" colspan="2"><input name="<?php echo esc_attr('wf_sn_wl'); ?>[wl_newname]" type="text" id="input_id" value="<?php echo esc_attr(self::$options['wl_newname']); ?>" class="regular-text" placeholder="<?php esc_attr_e('Security Ninja', 'security-ninja'); ?>"></td>
			</tr>

			<tr>
				<th scope="row"><label for="input_id"><h3><?php echo esc_html__('Plugin Description', 'security-ninja'); ?></h3></label></th></tr>
				<tr><td class="fullwidth" colspan="2"><input name="<?php echo esc_attr('wf_sn_wl'); ?>[wl_newdesc]" type="text" id="input_id" value="<?php echo esc_attr(self::$options['wl_newdesc']); ?>" class="regular-text" placeholder="<?php esc_attr_e('Since 2011 Security Ninja has helped thousands of site owners like you to feel safe!', 'security-ninja'); ?>"></td>
			</tr>

			<tr>
				<th scope="row"><label for="input_id"><h3><?php echo esc_html__('Author Name', 'security-ninja'); ?></h3></label></th></tr>
				<tr><td class="fullwidth" colspan="2"><input name="<?php echo esc_attr('wf_sn_wl'); ?>[wl_newauthor]" type="text" id="input_id" value="<?php echo esc_attr(self::$options['wl_newauthor']); ?>" class="regular-text" placeholder="<?php esc_attr_e('WP Security Ninja', 'security-ninja'); ?>"></td>
			</tr>

			<tr>
				<th scope="row"><label for="input_id"><h3><?php echo esc_html__('Author URL', 'security-ninja'); ?></h3>
				<p class="description"><?php esc_html_e('Enter the new URL for both the author and the plugin.', 'security-ninja'); ?></p>
			</label></th></tr>
				<tr><td class="fullwidth" colspan="2"><input name="<?php echo esc_attr('wf_sn_wl'); ?>[wl_newurl]" type="text" id="input_id" value="<?php echo esc_attr(self::$options['wl_newurl']); ?>" class="regular-text" placeholder="https://wpsecurityninja.com/">
				
			</td>
			
			</tr>

			<tr>
				<th scope="row"><label for="plugin_icon_url"><h3><?php echo esc_html__('Plugin Icon URL', 'security-ninja'); ?></h3><p class="description"><?php esc_html_e('The little square image used to represent the plugin, eg on the update-core page.', 'security-ninja'); ?></p>
			</label></th></tr>
				<tr><td class="fullwidth" colspan="2">
					<div class="media-upload-field">
						<input name="<?php echo esc_attr('wf_sn_wl'); ?>[wl_newiconurl]" type="text" id="plugin_icon_url" value="<?php echo esc_attr(self::$options['wl_newiconurl']); ?>" class="regular-text">
						<button type="button" class="button button-secondary" id="upload_plugin_icon_button"><?php esc_html_e('Select Image', 'security-ninja'); ?></button>
						<button type="button" class="button button-secondary" id="clear_plugin_icon_button"><?php esc_html_e('Clear', 'security-ninja'); ?></button>
					</div>
					<div id="plugin_icon_preview" class="image-preview">
						<?php if (!empty(self::$options['wl_newiconurl'])) : ?>
							<img src="<?php echo esc_url(self::$options['wl_newiconurl']); ?>" alt="<?php esc_attr_e('Plugin Icon', 'security-ninja'); ?>" style="max-width: 32px; max-height: 32px;">
						<?php endif; ?>
					</div>
					
				</td>
			</tr>

			<tr>
				<th scope="row"><label for="menu_icon_url"><h3><?php echo esc_html__('Plugin Menu Icon URL', 'security-ninja'); ?></h3><p class="description"><?php esc_html_e('This is the little menu icon in the sidebar', 'security-ninja'); ?></p></label></th></tr>
				<tr><td colspan="2" class="fullwidth">
					<div class="media-upload-field">
						<input name="<?php echo esc_attr('wf_sn_wl'); ?>[wl_newmenuiconurl]" type="text" id="menu_icon_url" value="<?php echo esc_attr(self::$options['wl_newmenuiconurl']); ?>" class="regular-text">
						<button type="button" class="button button-secondary" id="upload_menu_icon_button"><?php esc_html_e('Select Image', 'security-ninja'); ?></button>
						<button type="button" class="button button-secondary" id="clear_menu_icon_button"><?php esc_html_e('Clear', 'security-ninja'); ?></button>
					</div>
					<div id="menu_icon_preview" class="image-preview">
						<?php if (!empty(self::$options['wl_newmenuiconurl'])) : ?>
							<img src="<?php echo esc_url(self::$options['wl_newmenuiconurl']); ?>" alt="<?php esc_attr_e('Menu Icon', 'security-ninja'); ?>" style="max-width: 32px; max-height: 32px;">
						<?php endif; ?>
					</div>
					
				</td>
			</tr>

			<tr>
				<td colspan="2" class="fullwidth">
					<p class="submit"><input type="submit" value="<?php esc_attr_e('Save Changes', 'security-ninja'); ?>" class="input-button button-primary" name="Submit" />
				</td>
			</tr>
			</tbody>
			</table>

			</form>

			<style>
				.media-upload-field {
					display: flex;
					align-items: center;
					margin-bottom: 10px;
				}
				.media-upload-field input {
					margin-right: 10px;
				}
				.image-preview {
					margin-top: 10px;
					min-height: 32px;
				}
			</style>

			<script type="text/javascript">
				jQuery(document).ready(function($) {
					// Plugin Icon Media Uploader
					$('#upload_plugin_icon_button').click(function(e) {
						e.preventDefault();
						var custom_uploader = wp.media({
							title: '<?php esc_html_e('Select Plugin Icon', 'security-ninja'); ?>',
							button: {
								text: '<?php esc_html_e('Use this image', 'security-ninja'); ?>'
							},
							multiple: false
						});
						custom_uploader.on('select', function() {
							var attachment = custom_uploader.state().get('selection').first().toJSON();
							$('#plugin_icon_url').val(attachment.url);
							$('#plugin_icon_preview').html('<img src="' + attachment.url + '" alt="<?php esc_attr_e('Plugin Icon', 'security-ninja'); ?>" style="max-width: 32px; max-height: 32px;">');
						});
						custom_uploader.open();
					});

					// Clear Plugin Icon
					$('#clear_plugin_icon_button').click(function(e) {
						e.preventDefault();
						$('#plugin_icon_url').val('');
						$('#plugin_icon_preview').empty();
					});

					// Menu Icon Media Uploader
					$('#upload_menu_icon_button').click(function(e) {
						e.preventDefault();
						var custom_uploader = wp.media({
							title: '<?php esc_html_e('Select Menu Icon', 'security-ninja'); ?>',
							button: {
								text: '<?php esc_html_e('Use this image', 'security-ninja'); ?>'
							},
							multiple: false
						});
						custom_uploader.on('select', function() {
							var attachment = custom_uploader.state().get('selection').first().toJSON();
							$('#menu_icon_url').val(attachment.url);
							$('#menu_icon_preview').html('<img src="' + attachment.url + '" alt="<?php esc_attr_e('Menu Icon', 'security-ninja'); ?>" style="max-width: 32px; max-height: 32px;">');
						});
						custom_uploader.open();
					});

					// Clear Menu Icon
					$('#clear_menu_icon_button').click(function(e) {
						e.preventDefault();
						$('#menu_icon_url').val('');
						$('#menu_icon_preview').empty();
					});
				});
			</script>
		</div>
<?php
	}





	/**
	 * Sets default white label settings.
	 *
	 * @author   Lars Koudal
	 * @since    v0.0.1
	 * @version  v1.0.0  Sunday, January 3rd, 2021.
	 * @access   public static
	 * @param bool $force Whether to force update default settings.
	 * @return void
	 */
	public static function default_settings($force = false)
	{
		$defaults = array(
			'wl_active'         => '0',
			'wl_newname'        => '',
			'wl_newdesc'        => '',
			'wl_newauthor'      => '',
			'wl_newurl'         => '',
			'wl_newiconurl'     => '',
			'wl_newmenuiconurl' => '',

		);

		$options = get_option('wf_sn_wl');

		if ($force || !$options || !$options['wl_active']) {
			update_option('wf_sn_wl', $defaults, false);
		}
	} // default_settings


	/**
	 * Performs cleanup when the plugin is deactivated.
	 *
	 * @author   Lars Koudal
	 * @since    v0.0.1
	 * @version  v1.0.0  Sunday, January 3rd, 2021.
	 * @access   public static
	 * @return void
	 */
	public static function deactivate()
	{
		$centraloptions = Wf_Sn::get_options();
		if (!isset($centraloptions['remove_settings_deactivate'])) {
			return;
		}
		if ($centraloptions['remove_settings_deactivate']) {
			wp_clear_scheduled_hook('secnin_run_core_scanner');
			delete_option('wf_sn_wl');
		}
	}




	/**
	 * Retrieves the white labeled URL.
	 *
	 * @author   Lars Koudal
	 * @since    v0.0.1
	 * @version  v1.0.0  Sunday, January 3rd, 2021.
	 * @access   public static
	 * @return string White labeled URL or empty string.
	 */
	public static function get_new_url()
	{
		$newurl = '';
		if ((isset(self::$options['wl_newurl']))
			&& ('' !== self::$options['wl_newurl'])
		) {
			$newurl = self::$options['wl_newurl'];
		}
		return $newurl;
	}


	/**
	 * Retrieves the white labeled icon URL.
	 *
	 * @author	Lars Koudal
	 * @since	v0.0.1
	 * @access	public static
	 * @return	mixed
	 */
	public static function get_new_icon_url()
	{
		$newurl = '';
		if ((isset(self::$options['wl_newiconurl']))
			&& ('' !== self::$options['wl_newiconurl'])
		) {
			$newurl = self::$options['wl_newiconurl'];
		}
		return $newurl;
	}




	/**
	 * Retrieves the white labeled menu icon URL.
	 *
	 * @author   Lars Koudal
	 * @since    v0.0.1
	 * @version  v1.0.0  Sunday, January 3rd, 2021.
	 * @access   public static
	 * @return string White labeled menu icon URL.
	 */
	public static function get_new_menu_icon_url()
	{
		$newmenuiconurl = '';
		if ((isset(self::$options['wl_newmenuiconurl']))
			&& ('' !== self::$options['wl_newmenuiconurl'])
		) {
			$newmenuiconurl = self::$options['wl_newmenuiconurl'];
		}
		return $newmenuiconurl;
	}





	/**
	 * Returns the whitelabel name of plugin, if any - else returns "Security";
	 *
	 * @author   Lars Koudal
	 * @since    v0.0.1
	 * @version  v1.0.0  Sunday, January 3rd, 2021.
	 * @access   public static
	 * @return   mixed
	 */
	public static function get_new_name()
	{
		$newname = 'Security';
		if ((isset(self::$options['wl_newname']))
			&& ('' !== self::$options['wl_newname'])
		) {
			$newname = self::$options['wl_newname'];
		}
		return $newname;
	}





	/**
	 * Retrieves the white labeled plugin description.
	 *
	 * @author   Lars Koudal
	 * @since    v0.0.1
	 * @version  v1.0.0  Sunday, January 3rd, 2021.
	 * @access   public static
	 * @return string|false White labeled description or false if not set.
	 */
	public static function get_new_desc()
	{

		if ((isset(self::$options['wl_newdesc']))
			&& ('' !== self::$options['wl_newdesc'])
		) {
			$newdesc = self::$options['wl_newdesc'];
			return $newdesc;
		}
		return false;
	}


	/**
	 * Retrieves the white labeled author name.
	 *
	 * @author	Unknown
	 * @since	v0.0.1
	 * @version	v1.0.0	Friday, June 14th, 2024.
	 * @access	public static
	 * @return string White labeled author name or 'WP Security Ninja'.
	 */
	public static function get_new_author_name()
	{
		$newauthorname = 'WP Security Ninja';
		if ((isset(self::$options['wl_newauthor']))
			&& ('' !== self::$options['wl_newauthor'])
		) {
			$newauthorname = self::$options['wl_newauthor'];
		}
		return $newauthorname;
	}





	/**
	 * Checks if white labeling is active.
	 *
	 * @author	Unknown
	 * @since	v0.0.1
	 * @version	v1.0.0	Friday, June 14th, 2024.
	 * @access	public static
	 * @return bool True if white labeling is active, false otherwise.
	 */
	public static function is_active()
	{
		if (!isset(self::$options['wl_active'])) {
			return false;
		}
		return (bool) self::$options['wl_active'];
	}


	/**
	 * Initializes admin-specific functionality.
	 *
	 * @author  Lars Koudal
	 * @since   v0.0.1
	 * @version v1.0.0  Sunday, January 3rd, 2021.
	 * @version v1.0.1  Tuesday, November 14th, 2023.
	 * @access  public static
	 * @global
	 * @return void
	 */
	public static function admin_init()
	{
		register_setting('wf_sn_wl', 'wf_sn_wl', array(__NAMESPACE__ . '\\Wf_Sn_Wl', 'sanitize_settings'));

		if (self::is_active()) {
			// Filter if whitelabel is not turned on
			global $submenu;
			// Filter out submenu items we do not want shown.
			if (isset($submenu['wf-sn'])) {
				$newwfsn = array();
				foreach ($submenu['wf-sn'] as $sfs) {
					if (!in_array($sfs[2], array('wf-sn-affiliation', 'wf-sn-account', 'wf-sn-contact', 'wf-sn-pricing', 'wf-sn-addons'), true)) {
						$newwfsn[] = $sfs;
					}
				}
				$submenu['wf-sn'] = $newwfsn; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
			}
		}
	}



	/**
	 * Sanitizes white label settings before saving.
	 *
	 * @author	Unknown
	 * @since	v0.0.1
	 * @version	v1.0.0	Friday, June 14th, 2024.
	 * @access	public static
	 * @param array $values Input values to sanitize.
	 * @return array Sanitized values.
	 */
	public static function sanitize_settings($values)
	{
		$old_options = get_option('wf_sn_wl');
		if (!is_array($values)) {
			$values = array();
		}
		$old_options['wl_active']         = 0;
		$old_options['wl_newname']        = '';
		$old_options['wl_newdesc']        = '';
		$old_options['wl_newauthor']      = '';
		$old_options['wl_newurl']         = '';
		$old_options['wl_newiconurl']     = '';
		$old_options['wl_newmenuiconurl'] = '';

		foreach ($values as $key => $value) {
			switch ($key) {
				case 'wl_active':
					$values[$key] = intval($value);
					break;
				case 'wl_newname':
				case 'wl_newdesc':
				case 'wl_newauthor':
				case 'wl_newurl':
				case 'wl_newiconurl':
				case 'wl_newmenuiconurl':
					$values[$key] = sanitize_text_field($value);
					break;
			}
		}

		return array_merge($old_options, $values);
	}

	/**
	 * Customizes the plugin action links in the plugins page.
	 * When whitelabel is enabled, this removes all Freemius actions except "Deactivate" and "Update".
	 *
	 * @author   Lars Koudal
	 * @since    v0.0.1
	 * @version  v1.0.1  Tuesday, January 7th, 2025.
	 * @access   public static
	 * @param array  $actions     Array of plugin action links.
	 * @param string $plugin_file Path to the plugin file relative to the plugins directory.
	 * @return array Modified array of plugin action links.
	 */
	public static function customize_plugin_action_links($actions, $plugin_file)
	{
		// Check if this is our plugin
		$our_plugins = array('security-ninja-premium/security-ninja.php', 'security-ninja/security-ninja.php');
		
		if (in_array($plugin_file, $our_plugins, true)) {
			// Keep only the deactivate and update links
			$deactivate_link = isset($actions['deactivate']) ? $actions['deactivate'] : '';
			$update_link = isset($actions['update']) ? $actions['update'] : '';
			
			// Clear all actions
			$actions = array();
			
			// Add back only the deactivate and update links
			if (!empty($deactivate_link)) {
				$actions['deactivate'] = $deactivate_link;
			}
			
			if (!empty($update_link)) {
				$actions['update'] = $update_link;
			}
		}
		
		return $actions;
	}

	/**
	 * Enqueues the WordPress media scripts for the whitelabel page.
	 *
	 * @author   Lars Koudal
	 * @since    v0.0.1
	 * @version  v1.0.0  Monday, May 20th, 2024.
	 * @access   public static
	 * @param string $hook The current admin page.
	 * @return void
	 */
	public static function enqueue_media_scripts($hook)
	{
		// Only enqueue on the Security Ninja page with the whitelabel tab
		if ('toplevel_page_wf-sn' !== $hook) {
			return;
		}
		
		// Enqueue the media scripts
		wp_enqueue_media();
	}
}
add_action('plugins_loaded', array(__NAMESPACE__ . '\Wf_Sn_Wl', 'init'));
register_deactivation_hook(WF_SN_BASE_FILE, array(__NAMESPACE__ . '\Wf_Sn_Wl', 'deactivate'));
