<?php
/**
 * Cloud Firewall WooCommerce Protection
 * 
 * This file contains the WooCommerce protection form content for the Cloud Firewall module.
 */

namespace WPSecurityNinja\Plugin;

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Ensure we have access to the main class
if (!class_exists('Wf_sn_cf')) {
    return;
}

/**
 * Render the WooCommerce protection form content
 * 
 * @param array $options Current options
 * @param array $ips IP data
 */
function wf_sn_cf_render_woocommerce_content($options, $ips) {
    ?>

<table class="form-table">
					<tbody>
						<tr>
							<td colspan="2">
								<h2><?php esc_html_e('WooCommerce Protection Settings', 'security-ninja'); ?></h2>
								<p class="description"><?php esc_html_e('Enhanced security protection specifically for WooCommerce stores. These features help prevent fraud, abuse, and automated attacks on your e-commerce site.', 'security-ninja'); ?></p>
							</td>
						</tr>

						<!-- Current Status Section - Moved to top -->
						<tr>
							<td colspan="2">
								<div class="sncard settings-card">
									<h3><?php esc_html_e('Current Status', 'security-ninja'); ?></h3>
									<?php
									if (class_exists('WooCommerce')) {
										echo '<p class="description" style="color: green;"><strong>' . esc_html__('✓ WooCommerce detected and protection available', 'security-ninja') . '</strong></p>';
									} else {
										echo '<p class="description" style="color: orange;"><strong>' . esc_html__('⚠ WooCommerce not detected - protection will be disabled', 'security-ninja') . '</strong></p>';
									}
									?>
									
									<h4><?php esc_html_e('Protection Status:', 'security-ninja'); ?></h4>
									<ul style="margin: 10px 0; padding-left: 20px;">
										<li><strong><?php esc_html_e('Rate Limiting:', 'security-ninja'); ?></strong> 
											<?php echo $options['woo_rate_limiting_enabled'] ? '<span style="color: green;">✓ Enabled</span>' : '<span style="color: red;">✗ Disabled</span>'; ?>
										</li>
										<li><strong><?php esc_html_e('Coupon Protection:', 'security-ninja'); ?></strong> 
											<?php echo $options['woo_coupon_protection_enabled'] ? '<span style="color: green;">✓ Enabled</span>' : '<span style="color: red;">✗ Disabled</span>'; ?>
										</li>
									</ul>
								</div>
							</td>
						</tr>

						<tr>
							<td colspan="2">
								<div class="sncard settings-card">
									<h3><?php esc_html_e('How WooCommerce Protection Works', 'security-ninja'); ?></h3>
									<ul class="security-test-list">
										<li><?php esc_html_e('Rate limiting for checkout, add to cart, and order placement', 'security-ninja'); ?></li>
										<li><?php esc_html_e('Coupon code brute force protection with temporary bans', 'security-ninja'); ?></li>
										<li><?php esc_html_e('Automatic blocking of suspicious IPs with temporary bans', 'security-ninja'); ?></li>
										<li><?php esc_html_e('Comprehensive logging of all security events', 'security-ninja'); ?></li>
										<li><?php esc_html_e('Safe for legitimate customers with conservative default settings', 'security-ninja'); ?></li>
									</ul>
									
									<h3><?php esc_html_e('Recommended Settings for Different Store Types', 'security-ninja'); ?></h3>
									<div>
										<table><tr>
										<td><strong><?php esc_html_e('Small Store (< 100 orders/month):', 'security-ninja'); ?></strong>
										<ul style="margin: 5px 0 15px 20px;">
											<li><?php esc_html_e('Checkout Rate Limit: 3', 'security-ninja'); ?></li>
											<li><?php esc_html_e('Add to Cart Limit: 10', 'security-ninja'); ?></li>
											<li><?php esc_html_e('Order Rate Limit: 2', 'security-ninja'); ?></li>
											<li><?php esc_html_e('Coupon Failed Attempts: 3', 'security-ninja'); ?></li>
										</ul>
								</td>
								<td>
										<p><strong><?php esc_html_e('Medium Store (100-1000 orders/month):', 'security-ninja'); ?></strong></p>
										<ul style="margin: 5px 0 15px 20px;">
											<li><?php esc_html_e('Checkout Rate Limit: 5', 'security-ninja'); ?></li>
											<li><?php esc_html_e('Add to Cart Limit: 15', 'security-ninja'); ?></li>
											<li><?php esc_html_e('Order Rate Limit: 3', 'security-ninja'); ?></li>
											<li><?php esc_html_e('Coupon Failed Attempts: 4', 'security-ninja'); ?></li>
										</ul>
								</td>
								<td>
										<p><strong><?php esc_html_e('Large Store (> 1000 orders/month):', 'security-ninja'); ?></strong></p>
										<ul style="margin: 5px 0 15px 20px;">
											<li><?php esc_html_e('Checkout Rate Limit: 8', 'security-ninja'); ?></li>
											<li><?php esc_html_e('Add to Cart Limit: 20', 'security-ninja'); ?></li>
											<li><?php esc_html_e('Order Rate Limit: 5', 'security-ninja'); ?></li>
											<li><?php esc_html_e('Coupon Failed Attempts: 5', 'security-ninja'); ?></li>
										</ul>
								</td>
								</tr>
							</table>

						</div>
							</td>
						</tr>

						<tr>
							<td colspan="2">
								<div class="sncard settings-card">
									<h3><?php esc_html_e('Important Safety Notes', 'security-ninja'); ?></h3>
									<div style="background: #fff3cd; border: 1px solid #ffeaa7; padding: 10px; border-radius: 4px; margin: 10px 0;">
										<ul style="margin: 0; padding-left: 20px;">
											<li><?php esc_html_e('These protections are designed to be safe for legitimate customers', 'security-ninja'); ?></li>
											<li><?php esc_html_e('Start with conservative settings and adjust based on your traffic', 'security-ninja'); ?></li>
											<li><?php esc_html_e('All blocked attempts are logged for review', 'security-ninja'); ?></li>
											<li><?php esc_html_e('Rate limits are temporary and automatically reset', 'security-ninja'); ?></li>
										</ul>
									</div>
								</div>
							</td>
						</tr>



						<!-- Rate Limiting Section -->
						<tr>
							<td colspan="2">
								<h3><?php esc_html_e('Rate Limiting Protection', 'security-ninja'); ?></h3>
								<p class="description"><?php esc_html_e('Prevent abuse of WooCommerce functionality through rate limiting.', 'security-ninja'); ?></p>
							</td>
						</tr>

						<tr valign="top">
							<th scope="row">
								<label for="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_rate_limiting_enabled'; ?>">
									<h3><?php esc_html_e('Enable Rate Limiting', 'security-ninja'); ?></h3>
									<p class="description"><?php esc_html_e('Enable rate limiting for WooCommerce actions (checkout, add to cart, order placement).', 'security-ninja'); ?></p>
								</label>
							</th>
							<td class="sn-cf-options">
								<?php
								\WPSecurityNinja\Plugin\Utils::create_toggle_switch(
									WF_SN_CF_OPTIONS_KEY . '_woo_rate_limiting_enabled',
									array(
										'saved_value' => $options['woo_rate_limiting_enabled'],
										'option_key'  => WF_SN_CF_OPTIONS_KEY . '[woo_rate_limiting_enabled]',
									)
								);
								?>
							</td>
						</tr>

						<tr valign="top">
							<th scope="row">
								<label for="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_checkout_rate_limit'; ?>">
									<h3><?php esc_html_e('Checkout Rate Limit', 'security-ninja'); ?></h3>
									<p class="description"><?php esc_html_e('Maximum number of checkout attempts per time window.', 'security-ninja'); ?></p>
								</label>
							</th>
							<td class="alignright">
								<input type="number" 
									id="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_checkout_rate_limit'; ?>" 
									name="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY); ?>[woo_checkout_rate_limit]" 
									value="<?php echo esc_attr($options['woo_checkout_rate_limit']); ?>" 
									min="2" 
									max="10" 
									class="small-text" />
								<p class="description"><?php esc_html_e('Range: 2-10 attempts. Recommended: 3-5 attempts', 'security-ninja'); ?></p>
							</td>
						</tr>

						<tr valign="top">
							<th scope="row">
								<label for="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_checkout_window'; ?>">
									<h3><?php esc_html_e('Checkout Window (seconds)', 'security-ninja'); ?></h3>
									<p class="description"><?php esc_html_e('Time window for checkout rate limiting.', 'security-ninja'); ?></p>
								</label>
							</th>
							<td class="alignright">
								<input type="number" 
									id="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_checkout_window'; ?>" 
									name="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY); ?>[woo_checkout_window]" 
									value="<?php echo esc_attr($options['woo_checkout_window']); ?>" 
									min="180" 
									max="1800" 
									class="small-text" />
								<p class="description"><?php esc_html_e('Range: 180-1800 seconds. Recommended: 180 seconds (3 minutes)', 'security-ninja'); ?></p>
							</td>
						</tr>

						<tr valign="top">
							<th scope="row">
								<label for="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_add_to_cart_limit'; ?>">
									<h3><?php esc_html_e('Add to Cart Rate Limit', 'security-ninja'); ?></h3>
									<p class="description"><?php esc_html_e('Maximum number of add to cart actions per time window.', 'security-ninja'); ?></p>
								</label>
							</th>
							<td class="alignright">
								<input type="number" 
									id="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_add_to_cart_limit'; ?>" 
									name="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY); ?>[woo_add_to_cart_limit]" 
									value="<?php echo esc_attr($options['woo_add_to_cart_limit']); ?>" 
									min="5" 
									max="20" 
									class="small-text" />
								<p class="description"><?php esc_html_e('Range: 5-20 actions. Recommended: 10-15 actions', 'security-ninja'); ?></p>
							</td>
						</tr>

						<tr valign="top">
							<th scope="row">
								<label for="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_add_to_cart_window'; ?>">
									<h3><?php esc_html_e('Add to Cart Window (seconds)', 'security-ninja'); ?></h3>
									<p class="description"><?php esc_html_e('Time window for add to cart rate limiting.', 'security-ninja'); ?></p>
								</label>
							</th>
							<td class="alignright">
								<input type="number" 
									id="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_add_to_cart_window'; ?>" 
									name="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY); ?>[woo_add_to_cart_window]" 
									value="<?php echo esc_attr($options['woo_add_to_cart_window']); ?>" 
									min="30" 
									max="300" 
									class="small-text" />
								<p class="description"><?php esc_html_e('Range: 30-300 seconds. Recommended: 60 seconds (1 minute)', 'security-ninja'); ?></p>
							</td>
						</tr>

						<tr valign="top">
							<th scope="row">
								<label for="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_order_rate_limit'; ?>">
									<h3><?php esc_html_e('Order Rate Limit', 'security-ninja'); ?></h3>
									<p class="description"><?php esc_html_e('Maximum number of order placement attempts per time window.', 'security-ninja'); ?></p>
								</label>
							</th>
							<td class="alignright">
								<input type="number" 
									id="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_order_rate_limit'; ?>" 
									name="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY); ?>[woo_order_rate_limit]" 
									value="<?php echo esc_attr($options['woo_order_rate_limit']); ?>" 
									min="1" 
									max="5" 
									class="small-text" />
								<p class="description"><?php esc_html_e('Range: 1-5 orders. Recommended: 2-3 orders', 'security-ninja'); ?></p>
							</td>
						</tr>

						<tr valign="top">
							<th scope="row">
								<label for="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_order_window'; ?>">
									<h3><?php esc_html_e('Order Window (seconds)', 'security-ninja'); ?></h3>
									<p class="description"><?php esc_html_e('Time window for order rate limiting.', 'security-ninja'); ?></p>
								</label>
							</th>
							<td class="alignright">
								<input type="number" 
									id="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_order_window'; ?>" 
									name="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY); ?>[woo_order_window]" 
									value="<?php echo esc_attr($options['woo_order_window']); ?>" 
									min="300" 
									max="3600" 
									class="small-text" />
								<p class="description"><?php esc_html_e('Range: 300-3600 seconds. Recommended: 600 seconds (10 minutes)', 'security-ninja'); ?></p>
							</td>
						</tr>

						<!-- Coupon Protection Section -->
						<tr>
							<td colspan="2">
								<h3><?php esc_html_e('Coupon Code Protection', 'security-ninja'); ?></h3>
								<p class="description"><?php esc_html_e('Prevent brute force attacks on your coupon codes. This feature temporarily blocks IPs that attempt too many invalid coupon codes.', 'security-ninja'); ?></p>
							</td>
						</tr>

						<tr valign="top">
							<th scope="row">
								<label for="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_coupon_protection_enabled'; ?>">
									<h3><?php esc_html_e('Enable Coupon Protection', 'security-ninja'); ?></h3>
									<p class="description"><?php esc_html_e('Protect against coupon code brute force attacks.', 'security-ninja'); ?></p>
								</label>
							</th>
							<td class="sn-cf-options">
								<?php
								\WPSecurityNinja\Plugin\Utils::create_toggle_switch(
									WF_SN_CF_OPTIONS_KEY . '_woo_coupon_protection_enabled',
									array(
										'saved_value' => $options['woo_coupon_protection_enabled'],
										'option_key'  => WF_SN_CF_OPTIONS_KEY . '[woo_coupon_protection_enabled]',
									)
								);
								?>
							</td>
						</tr>

						<tr valign="top">
							<th scope="row">
								<label for="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_coupon_failed_attempts'; ?>">
									<h3><?php esc_html_e('Failed Attempts Before Ban', 'security-ninja'); ?></h3>
									<p class="description"><?php esc_html_e('Number of failed coupon attempts before temporary ban.', 'security-ninja'); ?></p>
								</label>
							</th>
							<td class="alignright">
								<input type="number" 
									id="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_coupon_failed_attempts'; ?>" 
									name="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY); ?>[woo_coupon_failed_attempts]" 
									value="<?php echo esc_attr($options['woo_coupon_failed_attempts']); ?>" 
									min="3" 
									max="15" 
									class="small-text" />
								<p class="description"><?php esc_html_e('Range: 3-15 attempts. Recommended: 3-5 attempts (stricter to catch bots)', 'security-ninja'); ?></p>
							</td>
						</tr>

						<tr valign="top">
							<th scope="row">
								<label for="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_coupon_window'; ?>">
									<h3><?php esc_html_e('Coupon Window (seconds)', 'security-ninja'); ?></h3>
									<p class="description"><?php esc_html_e('Time window to count failed coupon attempts.', 'security-ninja'); ?></p>
								</label>
							</th>
							<td class="alignright">
								<input type="number" 
									id="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_coupon_window'; ?>" 
									name="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY); ?>[woo_coupon_window]" 
									value="<?php echo esc_attr($options['woo_coupon_window']); ?>" 
									min="180" 
									max="1800" 
									class="small-text" />
								<p class="description"><?php esc_html_e('Range: 180-1800 seconds. Recommended: 180 seconds (3 minutes)', 'security-ninja'); ?></p>
							</td>
						</tr>

						<tr valign="top">
							<th scope="row">
								<label for="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_coupon_ban_time'; ?>">
									<h3><?php esc_html_e('Coupon Ban Duration (seconds)', 'security-ninja'); ?></h3>
									<p class="description"><?php esc_html_e('How long to ban IP from coupon usage.', 'security-ninja'); ?></p>
								</label>
							</th>
							<td class="alignright">
								<input type="number" 
									id="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY) . '_woo_coupon_ban_time'; ?>" 
									name="<?php echo esc_attr(WF_SN_CF_OPTIONS_KEY); ?>[woo_coupon_ban_time]" 
									value="<?php echo esc_attr($options['woo_coupon_ban_time']); ?>" 
									min="300" 
									max="3600" 
									class="small-text" />
								<p class="description"><?php esc_html_e('Range: 300-3600 seconds. Recommended: 900 seconds (15 minutes)', 'security-ninja'); ?></p>
							</td>
						</tr>

						</tr>
					</tbody>
				</table>

<?php
}