<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Alexander Khmelnitskiy (info@alexander.khmelnitskiy.ua), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Plugin as Plugin;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

final class AdminScripts {

	/**
	 * The one true AdminScripts.
	 *
	 * @var AdminScripts
	 * @since 3.0.0
	 **/
	private static $instance;

	/**
	 * Sets up a new AdminScripts instance.
	 *
	 * @since 3.0.0
	 * @access public
	 **/
	private function __construct() {

		/** Add admin styles. */
		add_action( 'admin_enqueue_scripts', [ $this, 'admin_scripts' ] );

	}

	/**
	 * Add JavaScrips for admin area.
	 *
	 * @since   3.0.0
	 * @return void
	 **/
	public function admin_scripts() {

		/** Plugin Settings Page. */
		$this->settings_scripts();

	}

	/**
	 * Scripts for plugin setting page.
	 *
	 * @since   3.0.0
	 * @return void
	 **/
	private function settings_scripts() {

		/** Add scripts only on plugin settings page. */
		$screen = get_current_screen();
		if ( null === $screen || ! in_array( $screen->base, Plugin::get_menu_bases() ) ) { return; }

		wp_enqueue_script( 'data-tables', Plugin::get_url() . 'js/datatables' . Plugin::get_suffix() . '.js', [ 'jquery' ], Plugin::get_version(), true );

		wp_enqueue_script( 'mdp-speaker-admin', Plugin::get_url() . 'js/admin' . Plugin::get_suffix() . '.js', [ 'jquery', 'dataTables' ], Plugin::get_version(), true );
		wp_localize_script('mdp-speaker-admin', 'mdpSpeakerAdmin', [
			'ajaxURL'   => admin_url('admin-ajax.php'),
			'nonce'     => wp_create_nonce( 'speaker-nonce' ), // Nonce for security.
			'voicesList'    => $this->rendered_voices(),
			'languagesList' => Language::$languages,
			'pluginUrl' => Plugin::get_url(),
			'translations' => [

				'play_rate' => __( 'The ratio of clicks on the play button to the total number of visitors', 'speaker' ),
				'time_average' => __( 'The average number of seconds played', 'speaker' ),
				'init' => __( 'The number of times the player was initialized', 'speaker' ),
				'play' => __( 'The number of times the play button was clicked', 'speaker' ),
				'pause' => __( 'The number of times the pause button was clicked', 'speaker' ),
				'time' => __( 'The total number of seconds played', 'speaker' ),
				'bounce_rate' => __( 'The ratio of visitors who did not listen to the end of the audio', 'speaker' ),

                'selectLocale' => __( 'Select locale', 'speaker' ),
                'localeCodeOfCountry' => __( 'Locale code of country', 'speaker' ),

                'selectVoice' => __( 'Select voice', 'speaker' ),
                'voiceName' => __( 'Voice name', 'speaker' ),

				'noVoices' => __( 'No one found', 'speaker' ),
				'placeholderValue' => __( 'Choose an option', 'speaker' ),
				'searchPlaceholderValue' => __( 'Language or locale code', 'speaker' ),
				'loadingText' => __( 'Loading...', 'speaker' ),
				'noResultsText' => __( 'No results found', 'speaker' ),
				'noChoicesText' => __( 'No choices to choose from', 'speaker' ),
				'addItemText' => __( 'Press Enter', 'speaker' ),
				'voiceSelectLabel' => __( 'Post voice', 'speaker' ),

				'error' => __( 'Error', 'speaker' ),
				'done' => __( 'Done!', 'speaker' ),
				'requestError' => __( 'Request error', 'speaker' ),
				'requestAborted' => __( 'Request aborted', 'speaker' ),

				'youCannotDuplicate' => __( 'You cannot duplicate this template', 'speaker' ),

			],
		] );

	}

    /**
     * Get rendered voices.
     * @return array
     */
    private function rendered_voices(): array {

        $rendered_voices = array();

        /** Skip if API key is empty. */
        if ( ! SpeakerHelper::is_key_exists() ) {
            return $rendered_voices;
        }

        /** Get voices list. */
        $voices = get_transient( 'speaker_list_voices' );
        if ( $voices ) {

            $unique_voices = array();
            foreach ( $voices as $voice ) :

                /** Skip already rendered voices. */
                if ( in_array( $voice->getName(), $unique_voices ) ) {
                    continue;
                } else {
                    $rendered_voices[] = [
                        'name' => $voice->getName(),
                        'lang' => $voice->getLanguageCodes()[0],
                        'gender' => $voice->getSsmlGender(),
                    ];
                    $unique_voices[] = $voice->getName();
                }

            endforeach;

        }

        return $rendered_voices;

    }

	/**
	 * Main AdminScripts Instance.
	 *
	 * Insures that only one instance of AdminScripts exists in memory at any one time.
	 *
	 * @static
	 * @return AdminScripts
	 * @since 3.0.0
	 **/
	public static function get_instance(): AdminScripts {

        /** @noinspection SelfClassReferencingInspection */
        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof AdminScripts ) ) {

			self::$instance = new AdminScripts;

		}

		return self::$instance;

	}

} // End Class AdminScripts.
