<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Settings;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

final class AudioFile {

	/**
	 * @access private
	 * @var AudioFile
	 **/
	private static $instance;

	/**
	 * Run actions and filters related to audio file.
	 *
	 * @param int $post_id
	 * @param string $prefix
	 * @param bool $legacy
	 * @param int $page_index
	 *
	 * @return string
	 */
	public static function file_name( int $post_id = 0, string $prefix = '', bool $legacy = false, int $page_index = 0 ): string {

		if ( ! $post_id ) { return ''; }

		$options = Settings::get_instance()->options;

		// Get file format
		$is_advanced_voice = isset( $options[ 'advanced_voice' ] ) && $options[ 'advanced_voice' ] === 'on';
		$file_format = $is_advanced_voice ? $options['audio-format'] : 'mp3';

		// Return file name for version lower than 4.0.0
		if ( $legacy ) {

			return apply_filters(
				'speaker_file_name',
				esc_attr( $prefix ) . 'post-' . $post_id . '.' . esc_attr( $file_format )
			);

		}

		// Get post
		$post = get_post( $post_id );
		$file_name_base = $options[ 'file_name' ] ?? 'post-[id]';

		// [id]
		if ( strpos( $file_name_base, '[id]' ) !== false ) {
			$file_name_base = str_replace( '[id]', $post_id, $file_name_base );
		}

		// [name]
		if ( strpos( $file_name_base, '[name]' ) !== false ) {
			$post_name = $post->post_name;
			$file_name_base = str_replace( '[name]', esc_attr( $post_name ), $file_name_base );
		}

		// [title]
		if ( strpos( $file_name_base, '[title]' ) !== false ) {
			$post_title = $post->post_title;
			$file_name_base = str_replace( '[title]', esc_attr( $post_title ), $file_name_base );
		}

		// [author]
		if ( strpos( $file_name_base, '[author]' ) !== false ) {
			$post_author = get_the_author_meta( 'display_name', $post->post_author );
			$file_name_base = str_replace( '[author]', esc_attr( $post_author ), $file_name_base );
		}

		// [type]
		if ( strpos( $file_name_base, '[type]' ) !== false ) {
			$post_type = $post->post_type;
			$file_name_base = str_replace( '[type]', esc_attr( $post_type ), $file_name_base );
		}

		// Add page suffix
		if ( $page_index > 0 ) {
			$file_name_base .= '-p' . $page_index;
		}

		return apply_filters(
			'speaker_file_name',
			esc_attr( $prefix ) . esc_attr( $file_name_base ) . '.' . esc_attr( $file_format )
		);

	}

	/**
	 * Get URL to audio file.
	 *
	 * @param int $id
	 * @param int $page_index
	 * @param string $prefix
	 *
	 * @return string
	 */
	public static function url( int $id = 0, int $page_index = 0, string $prefix = '' ) : string {

		/** Get path to upload folder. */
		$upload_dir     = wp_get_upload_dir();
		$upload_baseurl = $upload_dir[ 'baseurl' ];

		/** URL to post audio file. */
		return $upload_baseurl . '/speaker/' . self::file_name( $id, $prefix, false, $page_index );

	}

	/**
	 * Get path to audio file.
	 *
	 * @param int $id
	 * @param int $page_index
	 * @param string $prefix
	 * @param bool $legacy
	 *
	 * @return string
	 */
	public static function path( int $id = 0, int $page_index = 0, string $prefix = '', bool $legacy = false ): string {

		/** Get path to upload folder. */
		$upload_dir     = wp_get_upload_dir();
		$upload_basedir = $upload_dir[ 'basedir' ];

		/** Path to post audio file. */
		return $upload_basedir . '/speaker/' . self::file_name( $id, $prefix, $legacy, $page_index );

	}

	/**
	 * Checks if there is audio for the current post.
	 *
	 * @param int $id
	 * @param int $page_index
	 *
	 * @return false|int
	 */
	public static function audio_exists( int $id = 0, int $page_index = 0 ) {

		/** Current post ID. */
		if ( ! $id ) {
			$id = get_the_ID();
		}
		if ( ! $id ) { return false; }

		/** Get post meta. */
		$audio_from_meta = get_post_meta( $id, 'mdp_speaker_post_audio', true );
		if ( $audio_from_meta && $audio_from_meta !== '' ) {
			return true;
		}

		/** Path to post audio file. */
		$audio_file = AudioFile::path( $id, $page_index );

		/** True if we have audio. */
		if ( file_exists( $audio_file ) ) {

			// Return file modified time
			return filemtime( $audio_file );

		}

		// Return file modified time for versions lower than 4.0.0
		$options = Settings::get_instance()->options;
		if ( $options[ 'legacy' ] === 'on' ) {

			$audio_file = AudioFile::path( $id, $page_index, '', true );
			if ( file_exists( $audio_file ) ) {
				return filemtime( $audio_file );
			}

		}

		return false;

	}

	/**
	 * Remove audio file by ID.
	 *
	 * @param int $id
	 * @param int $page_index
	 *
	 * @return void
	 */
	public static function remove_by_id( int $id, int $page_index = 0 ) {

		/** Path to post audio file. */
		$audio_file = self::path( $id, $page_index );

		/** Remove audio file. */
		wp_delete_file( $audio_file );

	}

	/**
	 * Main AudioFile Instance
	 * @return AudioFile
	 */
	public static function get_instance(): AudioFile {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
