<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Google\ApiCore\ApiException;
use Merkulove\Speaker\Unity\Plugin;
use Merkulove\Speaker\Unity\Settings;
use Merkulove\Speaker\Unity\Tab;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove\Speaker
 */
final class Config {

	/**
	 * The one true Settings.
	 *
     * @since 1.0.0
     * @access private
	 * @var Config
	 **/
	private static $instance;

	/**
	 * Add more settings tab
	 * Must be run first in prepare settings
	 * @return void
	 */
	private function add_tabs() {

		// Tabs in reverse order

		Tab::add_settings_tab(
			'design',
			0,
			'display_settings',
			esc_html__( 'Advanced', 'speaker' ),
			esc_html__( 'Advanced Settings', 'speaker' ),
			esc_html__( 'Advanced settings allow you to configure the plugin to work with your theme filters and hooks, as well as manage user rights. Please make sure you understand what you are doing before changing anything on this tab.', 'speaker' )
		);

		Tab::add_settings_tab(
			'analytics',
			0,
			'stacked_line_chart',
			esc_html__( 'Analytics', 'speaker' ),
			esc_html__( 'Analytics Settings', 'speaker' ),
			esc_html__( 'Use the plugin analytics system to collect insensitive, anonymous listening data on generated audio files. The data will be displayed only in the admin panel.', 'speaker' )
		);

		Tab::add_settings_tab(
			'multilingual',
			0,
			'language',
			esc_html__( 'Multilingual', 'speaker' ),
			esc_html__( 'Multilingual support', 'speaker' ),
			esc_html__( 'This tab controls multilingual audio generation. You have to install and configure multilingualism on your site using WPML, Polylang or other similar plugins before enabling the Multilingual option. Select a locale and specify the language you want to use for it.', 'speaker' )
		);

		$google_link = '<a href="https://support.google.com/podcast-publishers/answer/9889544" target="_blank">' .
			esc_html__( 'RSS feed guidelines for Google Podcasts', 'speaker' ) .
			'</a>';
		$apple_link = '<a href="https://help.apple.com/itc/podcasts_connect/" target="_blank">' .
			esc_html__( 'Apple Podcast’s Guide to RSS', 'speaker' ) .
			'</a>';
		$spotify_link = '<a href="https://content-ops.atspotify.com/hc/en-us/articles/360052921671-Podcast-Delivery-Specification-1-9" target="_blank">' .
			esc_html__( 'Spotify Podcast Delivery Specification', 'speaker' ) .
			'</a>';
		Tab::add_settings_tab(
			'rss',
			0,
			'rss_feed',
			esc_html__( 'Podcasts', 'speaker' ),
			esc_html__( 'RSS podcasts feed settings', 'speaker' ),
			wp_sprintf(
				/* translators: %1$s: Google Podcasts link, %2$s: Apple Podcasts link, %3$s: Spotify Podcasts link */
				esc_html__( 'Create RSS feed podcasts following the %1$s, %2$s, %3$s or other podcasts services.', 'speaker' ),
				$google_link,
				$apple_link,
				$spotify_link
			)
		);

		Tab::add_settings_tab(
			'schema',
			0,
			'data_object',
			esc_html__( 'Speakable Markup', 'speaker' ),
			esc_html__( 'Speakable Markup Settings', 'speaker' ),
			esc_html__( 'This tab allows you to configure the markup of the speakable schema. This markup is used by Google to read the content of the site aloud.', 'speaker' )
		);

		Tab::add_settings_tab(
			'storage',
			0,
			'storage',
			esc_html__( 'Storage', 'speaker' ),
			esc_html__( 'Storage Settings', 'speaker' ),
			esc_html__( 'This tab allows you to configure the storage of audio files. You can use the local server storage or cloud storage. If you use cloud storage, you need to configure the connection to it.', 'speaker' ) . '<br><span style="color: orangered">' . esc_html__( 'If you change the Audio File Name, the audio files with the old name will no longer be displayed as audio for the post.', 'speaker' ) . '</span>'
		);

		Tab::add_settings_tab(
			'player',
			0,
			'play_circle',
			esc_html__( 'Audio Player', 'speaker' ),
			esc_html__( 'Audio Player Settings', 'speaker' ),
			esc_html__( 'This tab allows you to configure the player. You can customize the appearance of the player, as well as the behavior of the player when playing audio.', 'speaker' )
		);

		Tab::add_settings_tab(
			'content',
			0,
			'difference',
			esc_html__( 'Audio Content', 'speaker' ),
			esc_html__( 'Audio Content Settings', 'speaker' ),
			esc_html__( 'Customize the content elements that will be added to the audio version of the post during audio generation.', 'speaker' )
		);

		Tab::add_settings_tab(
			'post_types',
			0,
			'article',
			esc_html__( 'Post Types', 'speaker' ),
			esc_html__( 'Post Types Settings', 'speaker' ),
			esc_html__( 'Customize for which post formats the player and voice interface will be available, and you can also create and customize special voice templates for each type of post.', 'speaker' )
		);

        // Reset notice markup
        if ( isset( $_GET['speaker_error'] ) ) {
            if ( isset( $_GET['message'] ) && $_GET['message'] ) {
                Notice::reset_notice_markup( $_GET['message'] );
            }
        }

		$tab_options = get_option( 'mdp_speaker_general_settings', array() );
		$no_key_notice = ( isset( $tab_options[ 'dnd-api-key' ] ) && $tab_options[ 'dnd-api-key' ] !== '' ) ? '' : Notice::start_notice_markup();

        Tab::add_settings_tab(
			'general',
			0,
			'graphic_eq',
			esc_html__( 'Speech Synthesis', 'speaker' ),
			esc_html__( 'Speech Synthesis Settings', 'speaker' ),
            self::msg_php_7_to_8() . esc_html__( 'Set the Google API Key File configured to Text-To-Speech API and configure the global language and voice settings, as well as finer voice settings if necessary.', 'speaker' ) . '<br>' . $no_key_notice
		);

	}

	/**
	 * Display server requirements on status tab
	 */
	private function add_server_requirements() {

		$tabs = Plugin::get_tabs();

		$tabs[ 'status' ][ 'reports' ][ 'server' ][ 'mbstring_installed' ] = true;
//		$tabs[ 'status' ][ 'reports' ][ 'server' ][ 'server_time' ] = true;
		$tabs[ 'status' ][ 'reports' ][ 'server' ][ 'allow_url_fopen' ] = false;

		$tabs[ 'custom_css' ][ 'description' ] = esc_html__( 'Use custom CSS for more flexibility in customizing the player view for your theme. The parent class of the block with the player:', 'speaker' ) . ' <code>.mdp-speaker-wrapper</code>';
		$tabs[ 'assignments' ][ 'description' ] = esc_html__( 'Set the conditions under which the player will be shown/hidden on the page. Find out more about this from', 'speaker' ) . ' <a href="https://docs.merkulov.design/assignments-in-wordpress-plugins/" target="_blank">' . esc_html__( 'Assignments in WordPress plugin', 'speaker' ) . '</a>.';
		$tabs[ 'activation' ][ 'description' ] = esc_html__( 'Activate the plugin with a purchase key to get access to all plugin features including hidden features. Note that an unauthenticated plugin only has basic functionality for audio generation.', 'speaker' );
		$tabs[ 'migration' ][ 'description' ] = esc_html__( 'Use this tab to export (import) plugin settings to another site and(or) permanently remove all settings from the current site.', 'speaker' );
		$tabs[ 'status' ][ 'description' ] = esc_html__( 'This tab displays the status of the plugin and the server requirements for its operation. Please note that all the necessary items are marked with a green checkmark, otherwise the plugin will not be able to work on your server.', 'speaker' );
		$tabs[ 'updates' ][ 'description' ] = esc_html__( 'This page contains all information about plugin updates and also allows you to check for an update for this plugin only.', 'speaker' );

		Plugin::set_tabs( $tabs );

	}

	/**
	 * Prepare plugin settings by modifying the default one.
	 *
	 * @return void
	 * @throws ApiException
	 * @since 1.0.0
	 * @access public
	 *
	 */
    public function prepare_settings() {

        /** Reset API Key on fatal error. */
        if ( isset( $_GET['reset-api-key'] ) && $_GET['reset-api-key'] ) {
            Caster::get_instance()->reset_api_key();
        }

        if (!function_exists('is_plugin_active')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        /** PHP version check. */
        if (version_compare(PHP_VERSION, '8.1', '<')) {
            if ( !is_plugin_active( 'speaker-php7/speaker-php7.php' ) ) {
                add_action('admin_notices', function () {
                    echo self::msg_php_7_to_8();
                });
            }
        } else {
            if ( is_plugin_active( 'speaker-php7/speaker-php7.php' ) ) {
                add_action('admin_notices', function () {
                    echo wp_sprintf(
                        '<div class="notice notice-warning is-dismissible"><p><strong>Speaker</strong>: %s %s. %s</p></div>', //But you have installed the PHP 7 compatibility plugin.
                        esc_html__('You have installed the Speaker PHP 7 compatibility plugin. Your current PHP version is', 'speaker'),
                        esc_attr( PHP_VERSION),
                        esc_html__('Speaker plugin compatible with PHP 8.1 or higher by default, you can deactivate and remove the Speaker PHP 7 compatibility plugin.', 'speaker')
                    );
                });
            }
        }

	    /** Set plugin settings */
	    $tabs = Plugin::get_tabs();
		unset( $tabs[ 'general' ] );
		Plugin::set_tabs( $tabs );

		/** Add server requirements. */
		$this->add_server_requirements();

		/** Add more settings tab. */
        $this->add_tabs();

        /** Shorthand access to plugin settings. */
        $options = Settings::get_instance()->options;

		TabVoice::controls( $options );

		TabDesign::controls( $options );

		TabPlayer::controls();

		TabSchema::controls();

		TabPostTypes::controls();

		TabContent::controls( $options );

		TabStorage::controls();

		TabRSS::controls();

		TabAnalytics::controls();

		TabMultilingual::controls();

	    Settings::get_instance()->get_options();

    }

    /**
     * @return string
     */
    private static function msg_php_7_to_8(): string {

        if (version_compare(PHP_VERSION, '8.1', '<')) {
            if ( !is_plugin_active( 'speaker-php7/speaker-php7.php' ) ) {
                return wp_sprintf(
                    '<div class="notice notice-error is-dismissible">
                                    <p>
                                        <strong>Speaker</strong>: %s %s. %s <strong>%s</strong> %s <code>%s</code> %s <code>%s</code> %s <a href="https://1.envato.market/cc-downloads" target="_blank">Codecanyon</a>.
                                    </p>
                                </div>',
                    esc_html__('Plugin requires PHP 8.1 or higher. Your current PHP version is', 'speaker'),
                    esc_attr( PHP_VERSION),
                    esc_html__('Update your PHP version or install', 'speaker'),
                    'Speaker - PHP7',
                    esc_html__('extension to continue using the plugin with the current PHP version. Find the', 'speaker'),
                    'speaker-php7.zip',
                    esc_html__('in', 'speaker'),
                    'plugin/speaker-extensions',
                    esc_html__( 'folder of the Speaker package downloaded from', 'speaker' )
                );
            }
        }
        return '';

    }

	/**
	 * Main Settings Instance.
	 *
	 * @return Config
	 **/
	public static function get_instance(): Config {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
