<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Settings;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

final class DashboardWidget {

	/**
	 * The one true DashboardWidget.
	 * @var DashboardWidget
	 */
	private static $instance;

    /**
     * Google cloud free limits
     * @var array|int[]
     */
    private static $quota = array(
        'wavenet' => 1000000,
        'neural' => 1000000,
        'standard' => 4000000
    );

    /** Empty stats array */
    private static $empty_stats = array(
        'standard' => 0,
        'wavenet' => 0,
        'neural' => 0,
    );

    /**
     * Option key for store stats
     * @var string
     */
    private static $option_key = 'mdp_speaker_stats';

    /**
     * Constructor
     */
	private function __construct() {

        $options = Settings::get_instance()->options;
        if ( $options[ 'analytics' ] === 'on' && $options[ 'analytics_dashboard_widget' ] !== 'on' ) { return; }

        /** Add status widget to dashboard. */
        add_action( 'wp_dashboard_setup', [ $this, 'add_dashboard_widgets' ] );

	}

    /**
     * Add dashboards
     * @return void
     */
    public function add_dashboard_widgets() {

        wp_add_dashboard_widget(
            'speaker-stats',
            esc_html__( 'Speaker statistics', 'speaker' ),
            [ $this, 'widget_speaker_stats' ]
        );

    }

    /**
     * Add statistics widget
     * @return void
     */
    public function widget_speaker_stats() {

        $statistics = get_option( self::$option_key, self::$empty_stats );
        if ( ! $statistics ) { return; }

        // Show empty stats if in current month no one characters used
        if ( isset( $statistics['month'] ) && $statistics['month'] !== date( 'n' ) ) {
            $statistics = self::$empty_stats;
        }

        ?>
        <div>
            <h3><b><?php echo esc_html__( 'Used quota in', 'speaker' ) . ' ' . date( 'F' ); ?></b></h3>
            <?php
                foreach ( $statistics as $key => $value ) {

                    if ( $key === 'month' ) { continue; }

                    $percent = $value / self::$quota[ $key ];
                    $min_percent = $percent > 0 && $percent < 1 ? 1 . '%' : $percent . '%';
                    if ( $percent === 0 ) { continue; }

                    ?><div class="speaker-stats--box-<?php esc_attr_e( $key ); ?>"><?php

                    echo wp_sprintf(
                        '<h4>%s %s</h4>',
                        ucwords( $key ),
                        esc_html__( 'voices', 'speaker' )
                    );
                    echo wp_sprintf(
                        '<p><b>%s</b> %s (%s)</p>',
                        $value,
                        esc_html__( 'characters', 'speaker' ),
                        $percent . '%'
                    );
                    echo wp_sprintf(
                        '<div class="speaker-stats--%s" data-value="%s"><span style="width: %s"></span></div>',
                        $key,
                        $value,
                        $min_percent
                    );

                    ?></div><?php

                }
            ?>
            <p class="speaker-stats--disclaimer"><?php esc_html_e( 'Please note that this is an estimate of the characters submitted for voiceover for the current site. You can find out the exact data on the use of your quota for the entire project from the', 'speaker' ); ?> <a href="https://console.cloud.google.com/home/dashboard" target="_blank"><?php esc_attr_e( 'Google Cloud', 'speaker' ); ?></a> <?php esc_html_e( 'dashboard', 'speaker' ); ?>.</p>
        </div>
        <?php

    }

    /**
     * Count characters
     */
    public function countChars( $text, $lang_name ) {

        $count = strlen( $text );
        $month_id = date( 'n' );
        $type = $this->get_lang_type( $lang_name );

        $new_stats = self::$empty_stats;
        $new_stats[ 'month' ] = $month_id;

        $statistics = get_option( self::$option_key, $new_stats );

        // Clear stats for new month
        if ( $statistics[ 'month' ] !== $month_id ) {

            $statistics = $new_stats;

        }

        // Save new stats
        $statistics[ $type ] += $count;

        update_option( self::$option_key, $statistics );

    }

    /**
     * Get language type
     * @param $lang_name
     * @return string
     */
    private function get_lang_type( $lang_name ): string
    {

        if ( strpos( $lang_name, 'Wavenet') ) {

            $type = 'wavenet';

        } else if ( strpos( $lang_name, 'Neural') ) {

            $type = 'neural';

        } else {

            $type = 'standard';

        }

        return $type;

    }

	/**
	 * Main DashboardWidget Instance.
	 *
	 * @return DashboardWidget
	 **/
	public static function get_instance(): DashboardWidget {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
