<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Alexander Khmelnitskiy (info@alexander.khmelnitskiy.ua), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\UI;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove\Speaker
 */
final class ErrorHandler {

	/**
	 * Custom error handler function.
	 *
	 * @param  int          $err_no   Error number. (can be a PHP Error level constant)
	 * @param  string       $err_str  Error description.
	 * @param  string|false $err_file File in which the error occurs.
	 * @param  int|false    $err_line Line number where the error is situated.
	 *
	 * @since 3.0.0
	 * @access public
	 *
	 * @example:
	 * <code>
	 * // Set custom error handler.
	 * set_error_handler( [ErrorHandler::class, 'error_handler'] );
	 *
	 * ...
	 *
	 * // Trigger custom error.
	 * trigger_error("A custom error has been triggered" );
	 *
	 * ...
	 *
	 * // Restores previous error handler.
	 * restore_error_handler();
	 * </code>
	 **/
	public static function error_handler( $err_no, $err_str, $err_file = false, $err_line = false ) {

		/** Render "Error" message. */
		UI::get_instance()->render_snackbar(
			esc_html__( 'Error number: ', 'speaker' ) . $err_no . '. ' .
			$err_str . esc_html__( ' in ', 'speaker' ) . $err_file .
			esc_html__( ' on line ', 'speaker' ) . $err_line,
			'error',
			-1
		);

	}

	/**
	 * Custom error handler function.
	 *
	 * @param $exception
	 *
	 * @since 3.0.0
	 * @access public
	 *
	 * @example:
	 * <code>
	 * // Set custom exception handler.
	 * set_exception_handler( [ErrorHandler::class, 'exception_handler'] );
	 *
	 * ...
	 *
	 * // Restores previous exception handler.
	 * restore_exception_handler();
	 * </code>
	 **/
	public static function exception_handler( $exception ) {

		$msg = $exception->getMessage();
		$file = $exception->getFile();
		$line = $exception->getLine();
		$isGuzzleError = strpos( $msg, 'GuzzleHttp' );

		?>
        <div class="mdp-alert-error" style="font-family: sans-serif">
            <strong><?php
            esc_html_e( 'Error: ', 'speaker' );
            ?></strong><?php
            esc_html_e( $msg );
            ?><br><?php
            esc_html_e( 'In: ', 'speaker' );
            esc_html_e( $file );
            esc_html_e( ' on line ', 'speaker' );
            esc_html_e( $line );
            if ( $isGuzzleError ) {
                ?><br><a href="https://merkulove.zendesk.com/hc/en-us/articles/360014533637-Conflict-of-Guzzle-library-versions" target="_blank" rel="noopener"><?php
                esc_html_e( 'Find out how to fix this problem', 'speaker' );
                ?></a><?php
            }
		    ?>
        </div>
        <?php

        // Show any Speaker errors only for website admins
        if ( ! is_admin() ) {
            return;
        }

        /**
         * Show reset Google Text-to-Speech and Google Storage keys links if the error is error file is the one of the speaker files
         * /wp-content/plugins/speaker/
         */
        if( strpos( $file, '/wp-content/plugins/speaker/' ) !== false ) {

            /**
             * Reset API Key on fatal error if the error is caused by an invalid Google Text-to-Speech key file
             */
            if( strpos( $file, '/wp-content/plugins/speaker/vendor/' ) !== false && strpos( $file, '/google/auth/src/Credentials/' ) !== false ) {
                Caster::get_instance()->reset_api_key( $msg );
                return;
            }
            if( strpos( $file, '/wp-content/plugins/speaker-php7/vendor/' ) !== false && strpos( $file, '/google/auth/src/Credentials/' ) !== false ) {
                Caster::get_instance()->reset_api_key( $msg );
                return;
            }

            /**
             * Reset API Key on fail to push limit error if the error is caused by an invalid Google Storage key file
             */
            if( strpos( $file, '/wp-content/plugins/speaker/vendor/' ) !== false && strpos( $file, 'CodedInputStream.php' ) !== false ) {
                Caster::get_instance()->reset_api_key( $msg );
                return;
            }
            if( strpos( $file, '/wp-content/plugins/speaker-php7/vendor/' ) !== false && strpos( $file, 'CodedInputStream.php' ) !== false ) {
                Caster::get_instance()->reset_api_key( $msg );
                return;
            }

            /**
             * Bad request GuzzleHttp error
             */
            if( $isGuzzleError && strpos( $file, '/wp-content/plugins/speaker/vendor/' ) !== false && strpos( $file, '/RequestException.php' ) !== false ) {
//                Caster::get_instance()->reset_api_key($msg);
//                return;
            }
            if( $isGuzzleError && strpos( $file, '/wp-content/plugins/speaker-php7/vendor/' ) !== false && strpos( $file, '/RequestException.php' ) !== false ) {
//                Caster::get_instance()->reset_api_key($msg);
//                return;
            }

            /**
             * Show reset Google Text-to-Speech and Google Storage keys links
             */
            if( strpos( $file, '/wp-content/plugins/speaker/vendor/' ) !== false || strpos( $file, '/wp-content/plugins/speaker-php7/vendor/' ) !== false ) {
                if ( strpos( $msg, 'syntax error') !== false ) {
                    Caster::get_instance()->reset_api_key( 'Incompatible PHP version' );
                    return;
                }
            }

            echo wp_sprintf( '<p style="font-family: sans-serif">%1$s <a href="%2$s" title="%3$s">%3$s</a> %6$s <a href="%4$s" title="%5$s">%5$s</a></p>',
                esc_html__( 'If you believe that the error is caused by invalid Google API keys, please reset the settings: ', 'speaker' ),
                admin_url( 'admin.php?page=mdp_speaker_settings&tab=voice&reset-api-key=1' ),
                esc_html__( 'Text-to-Speech', 'speaker' ),
                admin_url( 'admin.php?page=mdp_speaker_settings&tab=storage&reset-drive-key=1' ),
                esc_html__( 'Google Storage', 'speaker' ),
                esc_html__( 'or', 'speaker' )
            );
        }

	}


}
