<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Settings;
use Merkulove\Speaker\Unity\TabAssignments;
use Merkulove\Speaker\Unity\Plugin as Plugin;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove\Speaker
 */
final class FrontScripts {

	/**
	 * The one true FrontScripts.
	 *
	 * @var FrontScripts
	 * @since 3.0.0
	 **/
	private static $instance;

	/**
	 * Sets up a new FrontScripts instance.
	 *
	 * @since 3.0.0
	 * @access public
	 **/
	private function __construct() {

		/** Add plugin scripts. */
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_scripts' ] );

	}

	/**
	 * Add plugin scripts.
	 *
	 * @return void
	 * @since   1.0.0
	 **/
	public function enqueue_scripts() {

        if ( ! TabAssignments::get_instance()->display() ) { return; }

        /** Options shorthand */
        $options = Settings::get_instance()->options;

        wp_enqueue_script(
            'mdp-speaker',
            Plugin::get_url() . 'js/speaker' . Plugin::get_suffix() . '.js',
            array(),
            Plugin::get_version(),
            array(
                'strategy' => 'async',
                'in_footer' => true
            )
        );

		wp_localize_script(
			'mdp-speaker',
			'mdpSpeaker',
			array(
				'share' => [
					'type'          => esc_attr( $options[ 'share_type' ] ),
					'shareTitle'    => esc_attr( $options[ 'share_title' ] ),
					'iconsUrl'      => esc_url( Plugin::get_url() . 'images/mdc-icons/' ),
				],
				'settings' => [
					'loop'          => 'on' === $options[ 'player_loop' ] ? 'true' : 'false',
					'autoplay'      => 'on' === $options[ 'player_autoplay' ] ? 'true' : 'false',
					'speed'         => 'on' === $options[ 'player_speed_controls' ] ? 'true' : 'false',
                    'iOSUnmute'     => isset( $options[ 'ios_unmute' ] ) && 'on' === $options[ 'ios_unmute' ] ? 'true' : 'false',
				],
				'translations' => [
					'play'          => __( 'Play', 'speaker' ),
					'pause'         => __( 'Pause', 'speaker' ),
					'repeatOn'      => __( 'Repeat: On', 'speaker' ),
					'repeatOff'     => __( 'Repeat: Off', 'speaker' ),
					'muteOn'        => __( 'Mute: On', 'speaker' ),
					'muteOff'       => __( 'Mute: Off', 'speaker' ),
					'volume'        => __( 'Volume', 'speaker' ),
					'download'      => __( 'Download', 'speaker' ),
					'speed'         => __( 'Speed', 'speaker' ),
					'speedChrome'   => __( 'Playback speed', 'speaker' ),
					'share'         => __( 'Share', 'speaker' ),
					'options'       => __( 'Options', 'speaker' ),
					'normal'        => __( 'Normal', 'speaker' ),
				],
				'controls' => [
					'download'      => $options[ 'player_download' ] === 'on' && $options[ 'player_download_link' ] === 'inline' ? 'true' : 'false',
					'mute'          => $options[ 'player_mute' ] === 'on' ? 'true' : 'false',
					'repeat'        => $options[ 'player_repeat' ] === 'on' ? 'true' : 'false',
					'speed'         => $options[ 'player_speed' ] === 'on' && $options[ 'player_speed_controls' ] === 'off' ? 'true' : 'false',
					'share'         => $options[ 'player_share' ] === 'on' ? 'true' : 'false',
					'volume'        => $options[ 'player_volume' ] === 'on' ? 'true' : 'false',
					'elapsed'       => $options[ 'player_elapsed' ] === 'on' ? 'true' : 'false',
					'duration'      => $options[ 'player_duration' ] === 'on' ? 'true' : 'false',
				],
				'icons' => [
					'play'      => self::get_svg( 'player_icon_play', $options ),
					'pause'     => self::get_svg( 'player_icon_pause', $options ),
					'repeatOn'  => self::get_svg( 'player_icon_repeat_on', $options ),
					'repeatOff' => self::get_svg( 'player_icon_repeat_off', $options ),
					'muteOn'    => self::get_svg( 'player_icon_mute', $options ),
					'muteOff'   => self::get_svg( 'player_icon_volume', $options ),
					'download'  => self::get_svg( 'player_icon_download', $options ),
					'speed'     => self::get_svg( 'player_icon_speed', $options ),
					'share'     => self::get_svg( 'player_icon_share', $options ),
				],
				'analytics' => [
					'nonce' => wp_create_nonce( 'mdp_speaker_analytics' ),
					'ajaxUrl' => admin_url( 'admin-ajax.php' ),
					'enable' => $options[ 'analytics' ] === 'on' ? 'true' : 'false',
					'precision' => $options[ 'analytics_precision' ],
				],
			)
		);

		// Wave player script
		if ( 'speaker-wavesurfer' === $options[ 'player_style' ] ) {

			wp_enqueue_script(
                'mdp-speaker-player-wavesurfer',
                Plugin::get_url() . 'js/speaker-player-wavesurfer' . Plugin::get_suffix() . '.js',
                array(),
                Plugin::get_version(),
                array(
                    'strategy' => 'async',
                    'in_footer' => true
                )
            );

			wp_localize_script(
				'mdp-speaker-player-wavesurfer',
				'mdpSpeakerPlayerWave',
				array(
					'playerWaveColor'       => $options[ 'player_color_wave' ],
					'playerProgressColor'   => $options[ 'player_color_progress' ],
					'playerCursorColor'     => $options[ 'wavesurfer_color_cursor' ],
					'playerWaveHeight'      => $options[ 'wavesurfer_wave_height' ],
					'playerBarRadius'       => $options[ 'wavesurfer_bar_radius' ],
					'playerBarWidth'        => $options[ 'wavesurfer_bar_width' ],
					'playerBarMinHeight'    => $options[ 'wavesurfer_bar_min_height' ],
					'playerBarGap'          => $options[ 'wavesurfer_bar_gap' ],
					'largePlayPause'        => $options[ 'wavesurfer_large_play_pause' ] === 'on' ? 'true' : 'false',
					'waveform'              => $options[ 'wavesurfer_waveform' ] === 'on' ? 'true' : 'false',
					'thumbnail'             => esc_attr( $options[ 'player_thumbnail' ] )
				)
			);

		}

		// MediaElement player script
		if ( in_array( $options[ 'player_style' ], [ 'speaker-mediaelement', 'speaker-round', 'speaker-rounded', 'speaker-squared' ] ) ) {

			wp_enqueue_script(
                'mdp-speaker-player-mediaelement',
                Plugin::get_url() . 'js/speaker-player-mediaelement' . Plugin::get_suffix() . '.js',
                array(
                    'mediaelement'
                ),
                Plugin::get_version(),
                array(
                    'strategy' => 'async',
                    'in_footer' => true
                )
            );

		}

		// Chrome player script
		if ( 'speaker-chrome' === $options[ 'player_style' ] ) {

			wp_enqueue_script(
                'mdp-speaker-player-chrome',
                Plugin::get_url() . 'js/speaker-player-chrome' . Plugin::get_suffix() . '.js',
                array(
                    'mediaelement'
                ),
                Plugin::get_version(),
                array(
                    'strategy' => 'async',
                    'in_footer' => true
                )
            );

		}

        /** Remove WP mediaElement if set Default Browser Player. */
        if ( 'speaker-browser-default' === $options['player_style'] ) { return; }

        wp_enqueue_script( 'jquery' );

	}

    /**
     * Get svg code of the svg file
     *
     * @param $key
     * @param array $options
     *
     * @return string
     */
    private static function get_svg( $key, array $options = [] ): string {
        $file_path = Plugin::get_path() . 'images/mdc-icons/' . $options[ $key ];

        if ( !file_exists( $file_path ) ) { return ''; }

        return file_get_contents( $file_path );
    }

	/**
	 * Main FrontScripts Instance.
	 *
	 * @return FrontScripts
	 **/
	public static function get_instance(): FrontScripts {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof FrontScripts ) ) {

			self::$instance = new FrontScripts;

		}

		return self::$instance;

	}

}
