<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Settings;
use Merkulove\Speaker\Unity\Plugin as Plugin;
use Merkulove\Speaker\Unity\TabAssignments;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove\Speaker
 */
final class FrontStyles {

	/**
	 * The one true FrontStyles.
	 *
	 * @var FrontStyles
	 * @since 3.0.0
	 **/
	private static $instance;

	/**
	 * Sets up a new FrontStyles instance.
	 *
	 * @since 3.0.0
	 * @access public
	 **/
	private function __construct() {

		// Manage WP mediaElement depends on player style.
		$player_style = Settings::get_instance()->options['player_style'] ?? 'speaker-mediaelement';
		if ( in_array( $player_style, [ 'speaker-browser-default', 'speaker-wavesurfer' ] ) ) {

			// Remove mediaelement styles and scripts.
			add_filter( 'wp_audio_shortcode_library', '__return_empty_string', 11 );

		} else {

			// Add mediaelement script for any theme.
			add_action( 'wp_enqueue_scripts', function () {
				wp_enqueue_script('wp-mediaelement' );
			} );

		}

		// Add plugin styles.
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_styles' ] );

	}

	/**
	 * Add plugin styles.
	 *
	 * @since 3.0.0
	 * @return void
	 **/
	public function enqueue_styles() {

		/** Checks if plugin should work on this page. */
		if ( ! TabAssignments::get_instance()->display() ) { return; }

		$options = Settings::get_instance()->options;

		// Speaker core styles for any player style.
		wp_enqueue_style( 'mdp-speaker', Plugin::get_url() . 'css/speaker' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );

		// Wavesurfer player styles.
		if ( $options[ 'player_style' ] === 'speaker-wavesurfer' ) {

			wp_enqueue_style( 'mdp-speaker-player-wavesurfer', Plugin::get_url() . 'css/speaker-player-wavesurfer' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );

			$css = $this->wavesurfer_inline_css( $options );
			wp_add_inline_style( 'mdp-speaker-player-wavesurfer', $css );

		}

		// Mediaelement player styles.
		if ( in_array( $options[ 'player_style' ], [ 'speaker-mediaelement', 'speaker-round', 'speaker-rounded', 'speaker-squared' ] ) ) {

			wp_enqueue_style( 'mdp-speaker-player-mediaelement', Plugin::get_url() . 'css/speaker-player-mediaelement' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );

		}

		// Chrome player styles.
		if ( $options[ 'player_style' ] === 'speaker-chrome' ) {

			wp_enqueue_style( 'mdp-speaker-player-chrome', Plugin::get_url() . 'css/speaker-player-chrome' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );

		}

		// Inline styles for any player style.
		wp_add_inline_style( 'mdp-speaker', $this->inline_css( $options ) . $options[ 'custom_css' ] );

	}

	/**
	 * Inline CSS.
	 *
	 * @param $options
	 *
	 * @return string
	 */
	public function inline_css( $options ): string {

		$slug = 'player';

		$player_margin = Settings::get_sides_css( $slug . '_margin', $options );
		$player_padding = Settings::get_sides_css( $slug . '_padding', $options );
		$player_radius = Settings::get_sides_css( $slug. '_border_radius', $options );
		$player_color = $options[ $slug . '_color' ];

		return "
            .mdp-speaker-wrapper {
	            --mdp-speaker-player-padding: $player_padding;
	            --mdp-speaker-player-margin: $player_margin;
	            --mdp-speaker-player-radius: $player_radius;
	            --mdp-speaker-player-color: $player_color;
	            --mdp-speaker-player-thumbnail-height: {$options['player_thumbnail_height']}px;
	            --mdp-speaker-player-color-controls: {$options['player_color_controls']};
	            --mdp-speaker-player-color-controls-hover: {$options['player_color_controls_hover']};
	            --mdp-speaker-player-color-text: {$options['player_color_text']};
	            --mdp-speaker-player-icon-size: {$options['player_icon_size']}px;
	            --mdp-speaker-player-font-size: {$options['player_font_size']}px;
	            --mdp-speaker-player-wave-color: {$options['player_color_wave']};
	            --mdp-speaker-player-progress-color: {$options['player_color_progress']};
	            --mdp-speaker-row-gap: {$options[ 'row_gap' ]}px;
	        }
        ";

	}

	/**
	 * Wavesurfer inline CSS.
	 * @param $options
	 *
	 * @return string
	 */
	public function wavesurfer_inline_css( $options ): string {

		return "
            .mdp-speaker-wrapper {
	            --mdp-speaker-wavesurfer-player-color-hover: {$options['wavesurfer_color_hover']};
	            --mdp-speaker-wavesurfer-player-thumbnail-width: {$options['wavesurfer_thumbnail_width']}px;
	            --mdp-speaker-wavesurfer-player-thumbnail-overlay: {$options['wavesurfer_thumbnail_overlay']};
	        }
        ";

	}

	/**
	 * Main FrontStyles Instance.
	 *
	 * @return FrontStyles
	 **/
	public static function get_instance(): FrontStyles {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
