<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Alexander Khmelnitskiy (info@alexander.khmelnitskiy.ua), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Google\ApiCore\ApiException;
use Merkulove\Speaker\Unity\Settings;
use Merkulove\Speaker\Unity\UI;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

final class MetaBox {

	/**
	 * The one true MetaBox.
	 *
	 * @var MetaBox
	 * @since 3.0.0
	 **/
	private static $instance;

	/**
	 * Sets up a new MetaBox instance.
	 *
	 * @since 3.0.0
	 * @access public
	 **/
	private function __construct() {

        $capability = Settings::get_instance()->options[ 'capability_audio' ] ?? 'manage_options';
        if ( ! current_user_can( $capability ) ) { return; }

		MetaBoxOptions::get_instance();

		add_action( 'add_meta_boxes', [ $this, 'meta_box' ] );

	}

	/**
	 * Add Meta Box for Post/Page.
	 *
	 * @since 3.0.0
	 * @access public
	 **/
	public function meta_box() {

		/** Get selected post types. */
		$screens = Settings::get_instance()->options['cpt_support'];

		foreach ( $screens as $screen ) {

			/** Add Speaker Metabox */
			add_meta_box(
				'mdp_speaker_box_id',
				'Speaker',
				[ $this, 'meta_box_html' ],
				$screen,
				'side',
				'core'
			);

		}

	}

	/**
     * Render Meta Box.
     *
	 * @return void
	 */
	public function meta_box_html() {

        $post_id = get_the_ID();

        // Check if multipage post and audio exists.
        if ( SpeakerHelper::is_multipage( $post_id ) ) {
	        $f_time = AudioFile::audio_exists( $post_id, 1 );
        } else {
	        $f_time = AudioFile::audio_exists();
        }

        // Show analytics and player if audio exists
		if ( $f_time ) {

            // Show date of creation.
            $this->metabox_date_of_creation( $f_time );

			// Show analytics if audio exists.
			Analytics::get_instance()->analytics_metabox();

			// Show audio player if audio exist.
			$this->metabox_player();

        }

		// Show "Generate Audio" button if Post already saved and published.
		$status = get_post_status();
		if ( 'publish' !== $status ) :

            // Show warning for unpublished posts.
            $this->meta_box_html_status();

		elseif ( post_password_required() ) :

            // Show warning for password protected posts.
            $this->meta_box_html_password();

		elseif ( Settings::get_instance()->options['dnd-api-key'] ) :

            // Show generate button.
            $this->meta_box_toolbar();
			$this->generate_buttons();

        endif;

	}

	/**
     * Metabox toolbar
	 * @return void
	 */
    private function meta_box_toolbar() {

        // Tabs order
        $tabs = [
	        'content' => esc_html__( 'Content', 'speaker' ),
            'voice' => esc_html__( 'Voice', 'speaker' ),
            'audio' => esc_html__( 'File', 'speaker' )
        ];

        // Add buttons.
	    $buttons = [];
	    foreach ( $tabs as $tab => $title ) {
		    $buttons[] = wp_sprintf(
			    '<button class="mdp-speaker-metabox-toolbar--button%s" data-tab="%s">%s</button>',
			    $tab === 'content' ? ' is-active' : '',
			    esc_attr( $tab ),
			    esc_html__( $title, 'speaker' )
		    );
	    }

        echo wp_sprintf(
            '<div class="mdp-speaker-metabox-toolbar">
                <div class="mdp-speaker-metabox-toolbar--buttons">%s</div>
                <div class="mdp-speaker-metabox-toolbar--content is-active" data-tab="content">%s</div>
                <div class="mdp-speaker-metabox-toolbar--content" data-tab="voice">%s</div>
                <div class="mdp-speaker-metabox-toolbar--content" data-tab="audio">%s</div>
            </div>',
            implode( '', $buttons ),
            $this->metabox_template_select(),
	        $this->metabox_post_voice(),
            $this->metabox_post_audio()
        );

    }

	/**
     * Generate buttons.
	 * @return void
	 */
    private function generate_buttons() {

        $post_id = get_the_ID();

	    // Check if multipage post and audio exists.
	    if ( SpeakerHelper::is_multipage( $post_id ) ) {
		    $audio_exists = AudioFile::audio_exists( $post_id, 1 );
	    } else {
		    $audio_exists = AudioFile::audio_exists();
	    }

        $generate = wp_sprintf(
            '<button id="mdp_speaker_generate" type="button" class="button-large components-button is-button is-primary is-large">%s</button>',
            $audio_exists ? esc_html__( 'Re-create audio', 'speaker' ) : esc_html__( 'Create audio', 'speaker' )
        );

        $remove = $audio_exists ? wp_sprintf(
            '<button id="mdp_speaker_remove" type="button" class="button-large components-button button-link-delete is-button is-default is-large">%s</button>',
            esc_html__( 'Remove', 'speaker' )
        ) : '';

	    echo wp_sprintf(
		    '<div class="mdp-speaker-meta-box-controls mdp-speaker-meta-box-generate-button">
                %s
                %s
            </div>',
		    $generate,
		    $remove
	    );

    }

	/**
     * Post voice selector
	 * @return string
	 */
    private function metabox_post_voice(): string {

        $options = Settings::get_instance()->options;

        $post_id = get_the_ID();
        $locale_props = $options[ 'multilang' ] === 'on' ? SpeakerHelper::multilang_properties( $options, SpeakerHelper::get_post_locale() ) : array();

        return wp_sprintf(
            '<div class="mdp-speaker-post-voice-controls">
                %s
            </div>
            <div id="mdp-speaker-post-voice" class="choices-wrapper" data-id="%s" data-global-voice="%s" data-locale-voice="%s" data-post-voice="%s"></div>
            <p>%s</p>',
	        $this->post_voice_controls(),
            $post_id,
            esc_attr( $options[ 'language' ] ),
            esc_attr( $locale_props[ 1 ] ?? '' ),
            get_post_meta( $post_id, 'mdp_speaker_post_voice', true ) ?: '',
            esc_html__( 'Select custom post voice for current post only.', 'speaker' )
        );

    }

	/**
     * Metabox post audio
	 * @return string
	 */
    private function metabox_post_audio(): string {

        $post_id = get_the_ID();
        return wp_sprintf(
            '<div class="mdp-speaker-audio-file-controls">
                %s
            </div>
            <div class="mdp-speaker-post-voice" data-id="%s">
                <label for="mdp-speaker-post-voice">%s</label>
                <input type="text" id="mdp-speaker-post-audio" name="mdp_speaker_post_audio" value="%s">
                <button type="button" id="mdp-speaker-post-audio-ml" class="mdp-speaker-audio-file-button" title="%s">
                    <span class="dashicons dashicons-admin-media"></span>
                </button>
            </div>
            <p>%s</p>',
            $this->post_audio_controls(),
	        $post_id,
            esc_html__( 'Audio file', 'speaker' ),
	        get_post_meta( $post_id, 'mdp_speaker_post_audio', true ) ?: '',
	        esc_html__( 'Medial Library', 'speaker' ),
	        esc_html__( 'Set a custom audio file for this post by URL or Media Library.', 'speaker' )
        );

    }

	/**
	 * Render Player code for MetaBox in the admin area.
	 *
	 * @since 3.0.0
	 * @access public
	 **/
	private function metabox_player() {

		$options = Settings::get_instance()->options;
		$download = ! in_array( $options[ 'player_download_link' ], [ 'none', 'backend' ] );
        $post_id = get_the_ID();

		$audio_from_meta = get_post_meta( $post_id, 'mdp_speaker_post_audio', true );
        if ( SpeakerHelper::is_multipage( $post_id ) && empty( $audio_from_meta ) ) {

            $pages_counter = SpeakerHelper::pages_counter( $post_id );

            echo '<div class="mdp-speaker-metabox-paginated-player">';

            echo '<select id="mdp-speaker-paginated-audio">';
            for( $page_index = 1; $page_index <= $pages_counter; $page_index++ ) {

                echo wp_sprintf(
                    '<option value="page-%1$d">%2$s %1$d</option>',
	                $page_index,
                    esc_html__( 'Page', 'speaker' )
                );

            }
	        echo '</select>';

	        for( $page_index = 1; $page_index <= $pages_counter; $page_index++ ) {

		        $audio_url = SpeakerCaster::get_instance()->get_audio_url( $post_id, $page_index );
                $page_visible = $page_index === 1 ? ' is-visible' : '';

		        ?>
                <div class="mdp-speaker-paginated-audio--page<?php echo esc_attr( $page_visible ); ?>" data-page="page-<?php echo esc_attr( $page_index ); ?>" data-url="<?php echo esc_url( $audio_url ) ?>">
                    <div class="mdp-speaker-box">
                        <div data-download="<?php echo esc_attr( $download ); ?>">
                            <?php echo do_shortcode( '[audio src="' . $audio_url . '" preload="metadata"]' ); ?>
                        </div>
                    </div>
                </div>
		        <?php

            }

            echo '</div>';

        } else {

	        $audio_url = $audio_from_meta && $audio_from_meta !== '' ? $audio_from_meta : SpeakerCaster::get_instance()->get_audio_url();

	        ?>
            <div class="mdp-speaker-box">
                <div data-download="<?php echo esc_attr( $download ); ?>">
			        <?php echo do_shortcode( '[audio src="' . $audio_url . '" preload="metadata"]' ); ?>
                </div>
            </div>
	        <?php

        }

	}

	/**
     * Date of creation for meta-box
     *
	 * @param $f_time
	 *
	 * @return void
	 */
    private function metabox_date_of_creation( $f_time ) {

	    $options = Settings::get_instance()->options;

        $analytics_button = '<button class="speaker-analytics-accordion-button material-icons">equalizer</button>';
        if ( $options[ 'analytics' ] !== 'on' || $options[ 'analytics_metabox' ] !== 'on' ) { $analytics_button = ''; }

        echo wp_sprintf(
            '<div class="mdp-speaker-audio-info-general">
                <div class="mdp-speaker-audio-date-of-creation">
                    <span class="dashicons dashicons-clock" title="%s"><i></i></span>
                    <span class="mdp-speaker-audio-timestamp">%s</span>
                </div>
                %s
            </div>',
            esc_html__( 'Date of creation audio file', 'speaker' ),
            esc_attr( $this->calculate_timestamp( $f_time ) ),
            $analytics_button
        );

    }

    /**
     * Get audio creation date and time.
     * @param $f_time
     * @return false|int|string
     */
    private function calculate_timestamp( $f_time ) {

        if ( ! $f_time ) {

            $timestamp = esc_html__( 'Some time ago', 'speaker' );

        } elseif ( $f_time === true ) {

            // Get post update timestamp fro remote audio file.
            $timestamp = get_the_modified_time( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) );

        } else {

            $seconds_ago = time() - $f_time;
            if ( $seconds_ago > 60 ) {

                $format = get_option('date_format') . ' ' . get_option('time_format');
                $timestamp =  date( $format, $f_time );

            } elseif ( $seconds_ago === 0 ) {

                $timestamp = esc_html__( 'Just now', 'speaker' );

            } else {

                $timestamp =  $seconds_ago . ' ' . esc_html__( 'seconds ago', 'speaker' );

            }

        }

        return $timestamp;

    }

    /**
     * Show warning for unpublished posts.
     *
     * @since 3.0.0
     * @access public
     *
     * @return void
     **/
	private function meta_box_html_status() {
        ?>
        <div class="mdp-warning">
            <?php esc_html_e( 'Publish a post before you can generate an audio version.', 'speaker' ); ?>
        </div>
        <?php
    }

    /**
     * Show warning for password protected posts.
     *
     * @since 3.0.0
     * @access public
     *
     * @return void
     **/
    private function meta_box_html_password() {
        ?>
        <div class="mdp-warning">
            <?php esc_html_e( 'Speaker reads only publicly available posts.', 'speaker' ); ?><br>
            <?php esc_html_e( 'Remove the password from the post, create an audio version, then close the post again with a password.', 'speaker' ); ?><br>
            <?php esc_html_e( 'This is a necessary safety measure.', 'speaker' ); ?>
        </div>
        <?php
    }

	/**
	 * Speech template controls.
	 *
	 * @param $default
	 *
	 * @return string
	 **/
	private function speech_template_controls( $default ): string {

		$material_button_class = 'mdp-speaker-icon-button material-icons';

		$buttons = [];

		$buttons[] = wp_sprintf(
			'<button id="mdp-speaker-add-speech-template-btn"
            data-post-url="%1$s"
            class="mdp-speaker-add %3$s"
            title="%2$s"
            aria-label="%2$s">add</button>',
			esc_url( SpeechTemplates::get_instance()->get_post_url() ),
			esc_attr__( 'Add Speech Template', 'speaker' ),
			$material_button_class
		);

		$buttons[] = wp_sprintf(
			'<button id="mdp-speaker-duplicate-speech-template-btn"
            data-post-url="%1$s"
            class="mdp-speaker-add %3$s"
            title="%2$s"
            aria-label="%2$s">content_copy</button>',
			esc_url( SpeechTemplates::get_instance()->get_post_url() ),
			esc_attr__( 'Duplicate Speech Template', 'speaker' ),
			$material_button_class
		);

		$buttons[] = wp_sprintf(
			'<button class="mdp-speaker-edit %2$s"
            title="%1$s"
            aria-label="%1$s">create</button>',
			esc_attr__( 'Edit Speech Template', 'speaker' ),
			$material_button_class
		);

		$buttons[] = wp_sprintf(
			'<button class="mdp-speaker-make-default %5$s"
            data-post-type="%1$s"
            data-default-for-post-type="%2$s"
            title="%3$s"
            aria-label="%3$s">%4$s</button>',
			esc_attr( get_post_type() ),
			esc_attr( $default ),
			esc_attr__( 'Save this Speech Template as Default for this Post Type', 'speaker' ),
			$default ? 'flag' : 'outlined_flag',
			$material_button_class
		);

		$buttons[] = wp_sprintf(
			'<button class="mdp-speaker-delete %2$s"
            title="%1$s"
            aria-label="%1$s">delete</button>',
			esc_attr__( 'Delete Speech Template', 'speaker' ),
			$material_button_class
		);

		return implode( "\n", $buttons );

	}

	/**
     * Render voice controls.
     *
	 * @return string
	 */
    private function post_voice_controls(): string {

	    $material_button_class = 'mdp-speaker-icon-button material-icons';

	    $buttons = [];

	    $buttons[] = wp_sprintf(
		    '<button id="mdp-speaker-preview-voice-btn" class="%2$s" title="%1$s" aria-label="%1$s">play_arrow</button>',
		    esc_attr__( 'Preview voice', 'speaker' ),
		    $material_button_class
	    );

	    $buttons[] = wp_sprintf(
		    '<button id="mdp-speaker-reset-voice-btn" class="%2$s" title="%1$s" aria-label="%1$s">history</button>',
		    esc_attr__( 'Reset post voice to default', 'speaker' ),
		    $material_button_class
	    );

	    return implode( "\n", $buttons );

    }

	/**
     * Render audio controls.
	 * @return string
	 */
    private function post_audio_controls(): string {

        $options = Settings::get_instance()->options;
        $material_button_class = 'mdp-speaker-icon-button material-icons';
        $post_id = get_the_ID();

        $buttons = [];

        if ( $options[ 'player_download_link_backend' ] === 'on' ) {
	        $buttons[] = wp_sprintf(
		        '<a id="mdp-speaker-download-audio-btn" href="%3$s" class="%2$s" title="%1$s" aria-label="%1$s" download="">browser_updated</a>',
		        esc_attr__( 'Download audio', 'speaker' ),
		        $material_button_class,
		        AudioFile::url( $post_id, SpeakerHelper::is_multipage( $post_id ) ? 1 : 0 )
	        );
        }

	    $is_onDrive = count( get_post_meta( get_the_ID(), 'mdp_speaker_gd' ) );
		if ( $is_onDrive ) {
            $buttons[] = wp_sprintf(
                '<span class="%2$s mdp-speaker-icon-status" title="%1$s" aria-label="%1$s">add_to_drive</span>',
                esc_attr__( 'Uploaded on Google Drive', 'speaker' ),
                $material_button_class
            );
        }

        if ( get_post_meta( $post_id, 'mdp_speaker_post_audio', true ) ) {
	        $buttons[] = wp_sprintf(
		        '<button id="mdp-speaker-clean-audio-btn" class="%2$s" title="%1$s" aria-label="%1$s">cleaning_services</button>',
		        esc_attr__( 'Reset custom audio file', 'speaker' ),
		        $material_button_class
	        );
        }

        return implode( "\n", $buttons );

    }

	/**
     * Render template editor.
	 * @return string
	 */
    private function metabox_template_select(): string {

        // Add editor to the page footer.
	    add_action( 'admin_footer', [$this, 'render_template_editor'] );

	    // Prepare options for ST select.
	    $st_options = $this->get_st_options();

        // Define selected ST.
	    $default = $this->get_default_st();
	    $custom_st = get_post_meta( get_the_ID(), 'mdp_speaker_custom_speech_template', true );
	    $selected = $custom_st ?: $default;

        // Options markup.
        $options_markup = [];
        foreach ( $st_options as $value => $label ) {

	        if ( $value === 'content' ) {
		        $value = '';
	        }

            $options_markup[] = wp_sprintf(
                '<option value="%s"%s>%s</option>',
                esc_attr( $value ),
	            $value === $selected ? ' selected' : '',
                esc_html( $label )
            );

        }

        return wp_sprintf(
        '<div class="mdp-speaker-meta-box-controls">
                <div class="mdp-speaker-st-controls">%4$s</div>
                <div id="mdp-speaker-post-template" class="mdp-speaker-st-select choices-wrapper" data-id="%5$s">
                    <label>%1$s</label>
                    <select id="mdp-speaker-speech-templates-template" name="mdp_speaker_speech_templates_template">
                        %2$s
                    </select>
                </div>
                <p>%3$s</p>
            </div>',
	        esc_html__( 'Speech Template', 'speaker' ),
            implode( '', $options_markup ),
            wp_sprintf(
                /* translators: %s: link to docs */
                esc_html__( 'Select one of the %s or create a new one.', 'speaker' ),
                '<a href="https://docs.merkulov.design/speech-templates/" target="_blank">' . esc_html__( 'Speech Templates', 'speaker' ) . '</a>'
            ),
            $this->speech_template_controls( $default ),
            get_the_ID()
        );

    }

    /**
     * Add speech template editor form to the page.
     *
     * @since  3.0.0
     * @access public
     *
     * @throws ApiException
     * @return void
     **/
    public function render_template_editor() {
        ?>
        <div id="mdp-speaker-template-speech-template-editor" class="mdc-hidden">
            <div id="mdp-speaker-speech-template-editor-box">
                <iframe id="mdp-speaker-speech-template-editor-iframe" src=""></iframe>
                <div class="mdp-iframe-preloader"><div></div></div>
                <div class="mdp-side-panel">
                    <header>
                        <h4 class="mdp-title"><?php esc_html_e( 'Speech Template Editor', 'speaker' ); ?><span>beta</span></h4>
                        <button class="mdp-close-btn mdc-icon-button material-icons mdc-ripple-upgraded mdc-ripple-upgraded--unbounded" title="<?php esc_html_e( 'Close Speech Template Editor', 'speaker' ); ?>">close</button>
                    </header>

                    <main>

                        <?php $this->render_template_name(); // Render template name filed. ?>

                        <div class="mdp-list-box">
                            <p class="mdp-stb-subtitle mdc-hidden"><?php esc_html_e( 'Template Elements', 'speaker' ); ?></p>
                            <ul id="mdp-speaker-st-list" class="mdc-list mdc-list--two-line"></ul>
                            <p class="mdp-stb-info"><?php esc_html_e( 'Click on one of the elements in the page preview to add this element for speech synthesis.', 'speaker' ); ?></p>
                            <p class="mdp-stb-info mdp-help-1 mdc-hidden"><?php esc_html_e( 'Drag and drag an item to change the voicing order.', 'speaker' ); ?></p>
                            <p class="mdp-stb-info mdp-help-2 mdc-hidden"><?php esc_html_e( 'Click on the pencil icon to edit an item.', 'speaker' ); ?></p>
                            <p class="mdp-stb-info mdp-help-3 mdc-hidden"><?php esc_html_e( 'Ctrl+Click or ⌘+Click to simulate a click to the page to open a tab or hidden content.', 'speaker' ); ?></p>

                            <p class="mdp-stb-help">
                                <span class="material-icons">info</span>
                                <a href="https://docs.merkulov.design/speech-templates/" target="_blank" rel="noopener"><?php esc_html_e( 'Need help?', 'speaker' ); ?></a>
                            </p>
                        </div>

                        <div id="mdp-speaker-element-form" class="mdc-hidden">

                            <div class="mdp-element-form-wrap">

                                <div class="mdp-element-form">

                                    <div>
                                        <?php

                                        /** Render form title and close button. */
                                        $this->render_header();

                                        /** Render element name filed. */
                                        $this->render_name();

                                        /** Render xPath input. */
                                        $this->render_xpath();

                                        /** Render content textarea. */
                                        $this->render_content();

                                        /** Render Language. */
                                        $this->render_language();

                                        /** Render Say As. */
                                        $this->render_say_as();

                                        /** Render Emphasis. */
                                        $this->render_emphasis();

                                        /** Render Pause Time. */
                                        $this->render_pause_time();

                                        /** Render Pause Strength. */
                                        $this->render_pause_strength();

                                        ?>

                                    </div>

                                </div>
                                <button class="mdp-close-btn mdc-icon-button material-icons mdc-ripple-upgraded mdc-ripple-upgraded--unbounded" title="Close Element Editor">close</button>

                                <?php
                                /** Render Actions Buttons. */
                                $this->render_dialog_actions();
                                ?>

                            </div>

                        </div>

                    </main>

                    <footer>
                        <div class="mdc-menu-surface--anchor">
                            <button class="mdp-add-btn mdc-button mdc-button--outlined mdc-ripple-upgraded">
                                <span class="mdc-button__ripple"></span>
                                <i class="material-icons mdc-button__icon">add</i>
                                <span class="mdc-button__label"><?php esc_html_e( 'Add Element', 'speaker' ); ?></span>
                            </button>

                            <div class="mdc-menu mdc-menu-surface" data-open="false">
                                <ul class="mdc-list" role="menu" aria-hidden="true" aria-orientation="vertical" tabindex="-1">
                                    <li class="mdp-add-element mdc-list-item" role="menuitem">
                                        <span class="mdc-list-item__text"><?php esc_html_e( 'Element', 'speaker' ); ?></span>
                                    </li>
                                    <li class="mdp-add-text mdc-list-item" role="menuitem">
                                        <span class="mdc-list-item__text"><?php esc_html_e( 'Text', 'speaker' ); ?></span>
                                    </li>
                                    <li class="mdp-add-pause mdc-list-item" role="menuitem">
                                        <span class="mdc-list-item__text"><?php esc_html_e( 'Pause', 'speaker' ); ?></span>
                                    </li>
                                </ul>
                            </div>
                        </div>

                        <button class="mdp-st-save-btn mdc-button mdc-button--dense mdc-button--raised mdc-ripple-upgraded" disabled="disabled">
                            <span class="mdc-button__ripple"></span>
                            <span class="mdc-button__label"><?php esc_html_e( 'Save Speech Template', 'speaker' ); ?></span>
                        </button>
                    </footer>

                </div>
            </div>
        </div>
        <?php

    }

    /**
     * Render dialog buttons.
     *
     * @since  3.0.0
     * @access public
     *
     * @return void
     **/
    private function render_dialog_actions() {
        ?>
        <footer class="mdc-dialog__actions">
            <button class="mdp-cancel-btn mdc-button mdc-button--outlined mdc-ripple-upgraded">
                <span class="mdc-button__ripple"></span>
                <span class="mdc-button__label"><?php esc_html_e( 'Cancel', 'speaker' ); ?></span>
            </button>
            <button class="mdp-save-btn mdc-button mdc-button--dense mdc-button--raised mdc-ripple-upgraded">
                <span class="mdc-button__ripple"></span>
                <span class="mdc-button__label"><?php esc_html_e( 'Save', 'speaker' ); ?></span>
            </button>
        </footer>
        <?php
    }

    /**
     * Render content textarea.
     *
     * @since  3.0.0
     * @access public
     *
     * @return void
     **/
    private function render_content() {

        ?><div class="mdp-speaker-content-box"><?php

        /** Render Content textarea. */
        UI::get_instance()->render_textarea(
            '',
            esc_html__( 'Content', 'speaker'),
            esc_html__( 'This content will be voiced.', 'speaker' ),
            [
                'name'  => 'mdp-speaker-element-form-content',
                'id'    => 'mdp-speaker-element-form-content'
            ]
        );

        ?></div><?php

    }

    /**
     * Render xPath input.
     *
     * @since  3.0.0
     * @access public
     *
     * @return void
     **/
    private function render_xpath() {

        ?><div class="mdp-speaker-xpath-box"><?php

        /** Render xPath input. */
        UI::get_instance()->render_input(
            '',
            esc_html__( 'xPath', 'speaker'),
            esc_html__( 'Enter xPath for selecting part to speech. Edit this field only if you understand what you are doing.', 'speaker' ),
            [
                'name'          => 'mdp-speaker-element-form-xpath',
                'id'            => 'mdp-speaker-element-form-xpath',
                'spellcheck'    => 'false'
            ]
        );

        ?></div><?php

    }

    /**
     * Render template name field.
     *
     * @since  3.0.0
     * @access public
     *
     * @return void
     **/
    private function render_template_name() {

        ?><div class="mdp-speaker-template-name-box"><?php

        /** Render Template Name input. */
        UI::get_instance()->render_input(
            'New Template',
            esc_html__( 'Template Name', 'speaker'),
            '',
            [
                'name'      => 'mdp-speaker-template-name',
                'id'        => 'mdp-speaker-template-name',
                'required'  => 'required'
            ]
        );

        ?></div><?php

    }

    /**
     * Render element name filed.
     *
     * @since  3.0.0
     * @access public
     *
     * @return void
     **/
    private function render_name() {

        ?><div class="mdp-speaker-name-box"><?php

        /** Render Name input. */
        UI::get_instance()->render_input(
            '',
            esc_html__( 'Name', 'speaker'),
            esc_html__( 'Enter name for current item. This field will not be voiced.', 'speaker' ),
            [
                'name'      => 'mdp-speaker-element-form-name',
                'id'        => 'mdp-speaker-element-form-name',
                'required'  => 'required'
            ]
        );

        ?></div><?php

    }

    /**
     * Render form title and close button.
     *
     * @since  3.0.0
     * @access public
     *
     * @return void
     **/
    private function render_header() {
        ?>
        <h3 class="mdp-title"></h3>
        <?php
    }

    /**
     * Render Languages fields.
     *
     * @since  3.0.0
     * @access public
     *
     * @return void
     **/
    private function render_language() {

        ?>
        <div class="mdp-speaker-voice-box">
            <div><?php esc_html_e( 'Voice now used:', 'speaker' ); ?></div>

            <?php
                /** Render current language. */
                TabVoice::current_language( false );
            ?>

            <?php
                /** Render voices. */
                TabVoice::st_element_voice();
            ?>

        </div>
        <?php

    }

    /**
     * Render Say as field in element edit form.
     *
     * @since 3.0.0
     * @access public
     *
     * @return void
     **/
    private function render_say_as() {

        /** Prepare options for select. */
        $options = [
            'none'       => esc_html__( 'None', 'speaker' ),
            'cardinal'   => esc_html__( 'Cardinal', 'speaker' ),
            'ordinal'    => esc_html__( 'Ordinal', 'speaker' ),
            'characters' => esc_html__( 'Characters', 'speaker' ),
            'fraction'   => esc_html__( 'Fraction', 'speaker' ),
            'expletive'  => esc_html__( 'Expletive', 'speaker' ),
            'unit'       => esc_html__( 'Unit', 'speaker' ),
            'verbatim'   => esc_html__( 'Verbatim', 'speaker' ),
            // TODO: Add date and time
        ];

        ?><div class="mdp-speaker-say-as-box"><?php

        /** Render select. */
        UI::get_instance()->render_select(
            $options,
            '',
            esc_html__( 'Say As', 'speaker' ),
            esc_html__( 'This option lets you indicate information about the type of text construct that is contained within the element.', 'speaker' ),
            [
                'name' => 'mdp-speaker-element-form-say-as',
                'id' => 'mdp-speaker-element-form-say-as'
            ]
        );

        ?></div><?php

    }

    /**
     * Render Pause Strength field in element edit form.
     *
     * @since 3.0.0
     * @access public
     *
     * @return void
     **/
    private function render_pause_strength() {

        /** Prepare options for select. */
        $options = [
            'none'     => esc_html__( 'None', 'speaker' ),
            'x-weak'   => esc_html__( 'X-Weak', 'speaker' ),
            'weak'     => esc_html__( 'Weak', 'speaker' ),
            'medium'   => esc_html__( 'Medium', 'speaker' ),
            'strong'   => esc_html__( 'Strong', 'speaker' ),
            'x-strong' => esc_html__( 'X-Strong', 'speaker' ),
        ];

        ?><div class="mdp-speaker-strength-box"><?php

        /** Render select. */
        UI::get_instance()->render_select(
            $options,
            '',
            esc_html__( 'Strength', 'speaker' ),
            esc_html__( 'Sets the strength of the output\'s prosodic break by relative terms.', 'speaker' ),
            [
                'name' => 'mdp-speaker-element-form-strength',
                'id' => 'mdp-speaker-element-form-strength'
            ]
        );

        ?></div><?php

    }

    /**
     * Render Pause Time field in element edit form.
     *
     * @since 3.0.0
     * @access public
     *
     * @return void
     **/
    private function render_pause_time() {

        ?><div class="mdp-speaker-time-box"><?php

        /** Render slider. */
        UI::get_instance()->render_slider(
            '300',
            50,
            10000,
            50,
            esc_html__( 'Time', 'speaker'),
            esc_html__( 'Current length of the pause:', 'speaker') . ' <strong>300</strong> ' . esc_html__( 'ms', 'speaker' ),
            [
                'name' => 'mdp-speaker-element-form-time',
                'class' => 'mdc-slider-width',
                'id' => 'mdp-speaker-element-form-time'
            ],
            false
        );

        ?></div><?php
    }

    /**
     * Render Emphasis field in element edit form.
     *
     * @since 3.0.0
     * @access public
     *
     * @return void
     **/
    private function render_emphasis() {

        /** Prepare options for select. */
        $options = [
            'none'     => esc_html__( 'None', 'speaker' ),
            'strong'   => esc_html__( 'Strong', 'speaker' ),
            'moderate' => esc_html__( 'Moderate', 'speaker' ),
            'reduced'  => esc_html__( 'Reduced', 'speaker' ),
        ];

        ?><div class="mdp-speaker-emphasis-box"><?php

        /** Render select. */
        UI::get_instance()->render_select(
            $options,
            '',
            esc_html__( 'Emphasis', 'speaker' ),
            esc_html__( 'This option lets you add or remove emphasis from text contained by the element.', 'speaker' ),
            [
                'name' => 'mdp-speaker-element-form-emphasis',
                'id' => 'mdp-speaker-element-form-emphasis'
            ]
        );

        ?></div><?php

    }

    /**
     * Return array with all Speech Templates.
     *
     * @since  3.0.0
     * @access public
     *
     * @return array
     **/
    public function get_st_options() {

        $options = [ 'content' => esc_html__( 'Content', 'speaker' ) ];

        /** @noinspection ClassConstantCanBeUsedInspection */
        if ( ! class_exists( '\Merkulove\SpeakerUtilities' ) ) {
//            return $options;
        }

        /** Read all ST from settings. */
        /** In this option we store all Speech Templates. */
        $st_opt_name = 'mdp_speaker_speech_templates';

        /** Get all Speech Templates. */
        $st = get_option( $st_opt_name, false );

        /** Return if no ST. */
        if ( ! $st ) { return $options; }

        /** If We have any ST. */
        if ( count( $st )  ) {

            /** Add add ST to list. */
            foreach ( $st as $template ) {

                $options[$template['id']] = $template['name'];

            }

        }

        return $options;

    }

    /**
     * Return default ST for current post type.
     *
     * @param null $post_type
     *
     * @since  3.0.0
     * @access public
     *
     * @return string
     **/
    public function get_default_st( $post_type = null ) {

        /** Use current post type if nothing received. */
        if ( ! isset( $post_type ) ) {

            /** Retrieves the post type of the current post. */
            $post_type = get_post_type();

        }

        /** In this option we store all Speech Templates. */
        $st_opt_name = 'mdp_speaker_speech_templates';

        /** Get all Speech Templates. */
        $st = get_option( $st_opt_name, false );

        /** We haven't any ST. */
        if ( ! $st ) { return 'content'; }

        /** For each ST. */
        foreach ( $st as $template ) {

            /** Skip if empty. */
            if ( ! isset( $template['default'] ) ) { continue; }

            /** Skip if not array. */
            if ( ! is_array( $template['default'] ) ) { continue; }

            if ( in_array( $post_type, $template['default'], false ) ) {

                return $template['id'];

            }

        }

        return 'content';

    }

	/**
	 * Main MetaBox Instance.
	 *
	 * @return MetaBox
	 **/
	public static function get_instance(): MetaBox {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
