<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Plugin as Plugin;
use Merkulove\Speaker\Unity\Settings;
use Merkulove\Speaker\Unity\UI;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove/Speaker
 */
final class Notice {

	/**
	 * @var Notice
	 **/
	private static $instance;

    private static $is_api_key = false;

    private static $is_audio_created = false;

	/**
	 * @return void
	 */
	private function __construct() {

        // Check if API Key is set
		self::$is_api_key = SpeakerHelper::is_key_exists();

        // Check if at least one audio is created
		$upload_dir     = wp_get_upload_dir();
		$upload_basedir = $upload_dir[ 'basedir' ];

        // Is speaker directory exists in uploads
        if ( ! is_dir( $upload_basedir . '/speaker/' ) ) {
            self::$is_audio_created = false;
        } else {
	        $dir_files = scandir( $upload_basedir . '/speaker/' );
	        if ( is_array( $dir_files ) ) {
		        self::$is_audio_created = count( $dir_files ) > 2;
	        } else {
		        self::$is_audio_created = false;
	        }
        }

		add_action( 'admin_notices', [ $this, 'admin_notice' ] );
        add_action( 'admin_footer', [ $this, 'plugin_notice' ] );

	}

	/**
	 * Notices for WordPress admin area
	 **/
	public function admin_notice() {

		// Get current screen.
		$screen = get_current_screen();
		if ( $screen === null ) { return; }

        // Start notice for first installing
        if ( ! self::$is_api_key || ! self::$is_audio_created ) {
            self::start_notice( $screen );
        }

        // Check if Google Drive Token saved
        if ( /*isset( $_GET[ 'code' ] ) && */isset( $_GET[ 'scope' ] ) ) {
	        self::gd_token_notice( $screen );
        }

	}

	/**
     * Notice for plugins settings page
	 * @return void
	 */
    public function plugin_notice() {

	    // Get current screen.
	    $screen = get_current_screen();
	    if ( $screen === null ) { return; }

	    // Check if API Key is set
	    if ( ! self::$is_api_key ) {
		    self::tts_key_notice( $screen );
	    }

    }

	/**
     * Start notice for first installing
	 * @param $screen
	 *
	 * @return void
	 */
    private static function start_notice( $screen ) {

        // Don't show notice on settings page
	    if ( in_array( $screen->base, Plugin::get_menu_bases() ) ) { return; }

	    echo self::start_notice_markup(
            'notice',
            [
                '<a class="speaker-notice--action-button" href="https://docs.merkulov.design/about-key-file-for-the-voicer-wordpress-plugin/" target="_blank" rel="noreferrer">' . esc_html__( 'Get a key', 'speaker' ) . '</a>',
                '<a class="speaker-notice--action-button" href="' . admin_url( 'admin.php?page=mdp_speaker_settings' ) . '" rel="noreferrer">' . esc_html__( 'Settings', 'speaker' ) . '</a>',
                '<a class="speaker-notice--action-button" href="' . admin_url( 'post.php?post=' . self::get_last_post_id() . '&action=edit' ) . '" target="_blank" rel="noreferrer">' . esc_html__( 'Edit post', 'speaker' ) . '</a>'
            ]
        );

    }

	/**
	 * Start notice markup
	 *
	 * @param string $css_class
	 * @param array $buttons
	 *
	 * @return string
	 */
    public static function start_notice_markup( string $css_class = '', array $buttons = [] ): string {

	    return wp_sprintf(
		    '<div class="%s notice-speaker">
                <h2 class="speaker-notice--title">%s</h2>
                <div class="speaker-notice--grid">
                    %s
                    %s
                    %s
                </div>
            </div>',
		    esc_attr__( $css_class ),
		    wp_sprintf(
		    /* translators: %s: Plugin name. */
			    esc_html__( '3 steps to get started with %s Plugin', 'speaker' ),
			    'Speaker'
		    ),
		    self::step_markup(
                '01',
                esc_html__( 'Get a key file', 'speaker' ),
                esc_html__( 'Set up your Google Cloud Platform project and authorization to creating audio from text.', 'speaker' ),
                $buttons[ 0 ] ?? ''
            ),
            self::step_markup(
                '02',
                esc_html__( 'Setup plugin', 'speaker' ),
                esc_html__( 'Use your Google Cloud Text-to-Speech API key to set up the plugin and choose a language & voice.', 'speaker' ),
                $buttons[ 1 ] ?? ''
            ),
            self::step_markup(
                '03',
                esc_html__( 'Voice your first post', 'speaker' ),
                esc_html__( 'Open post editor and click the "Create audio". The plugin will create an audio version of your post.', 'speaker' ),
                $buttons[ 2 ] ?? ''
            )
	    );

    }

    /**
     * Reset notice markup
     * @param string $message
     * @return void
     */
    public static function reset_notice_markup( string $message = '' ) {

        UI::get_instance()->render_snackbar(
            esc_html__( 'The Google API key was reset due to an error', 'speaker' ) . ': ' . esc_html( $message ),
            'error',
            -1
        );

    }

	/**
     * Step markup for start notice
	 * @param $number
	 * @param $title
	 * @param $description
	 * @param $button
	 *
	 * @return string
	 */
    private static function step_markup(
        $number,
        $title,
        $description,
        $button
    ): string {

	    return wp_sprintf(
		    '<div class="speaker-notice--grid-item%5$s">
                    <p class="speaker-notice--grid-item-title"><span>%1$s</span>%2$s</p>
                    <p>%3$s</p>
                    %4$s
                </div>',
            esc_attr( $number ),
		    esc_html( $title ),
		    esc_html( $description ),
		    $button,
		    self::$is_api_key ? ' speaker-notice--grid-item--done' : ''
	    );

    }

	/**
     * Get id of last post with supported post type
	 * @return int
	 */
    private static function get_last_post_id(): int {

        $options = Settings::get_instance()->options;

        $supported_post_types = $options[ 'cpt_support' ] ?? array( 'post' );
	    $latest_cpt = get_posts( 'post_type=' . esc_attr( $supported_post_types[ 0 ] ) . '&numberposts=1' );
	    return $latest_cpt[0]->ID ?? 0;

    }

	/**
     * Text to speech API key notice
	 * @param $screen
	 *
	 * @return void
	 */
	private static function tts_key_notice( $screen ) {

		// Speaker Settings Page.
		if ( ! in_array( $screen->base, Plugin::get_menu_bases() ) ) { return; }

        UI::get_instance()->render_snackbar(
            esc_html__( 'This plugin uses the Google Cloud Text-to-Speech API Key File. Set up your Google Cloud Platform project before the start.', 'speaker' ),
            'warning',
            -1,
            true,
            [
                [
                    'caption' => esc_html__( 'Get Key File', 'speaker' ),
                    'link' => 'https://docs.merkulov.design/about-key-file-for-the-voicer-wordpress-plugin/'
                ]
            ]
        );

	}

	/**
	 * Google Drive token success message
	 */
	private static function gd_token_notice( $screen ) {

		/** Speaker Settings Page. */
		if ( in_array( $screen->base, Plugin::get_menu_bases() ) ) {

			UI::get_instance()->render_snackbar(
				esc_html__( 'Great job! Google Drive token is saved.', 'speaker' ),
				'info',
				3000
			);

		} else {

			?>
			<div class="settings-error notice notice-success">
				<p><strong><?php esc_html_e( 'Great job!', 'speaker' ); ?></strong></p>
				<p><?php esc_html_e( 'Speaker saved your Google Drive token. Your site is now linked to your Google Drive account.', 'speaker' ); ?></p>
			</div>
			<?php

		}

	}

	/**
	 * Main Instance.
	 *
	 * @return Notice
	 **/
	public static function get_instance(): Notice {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
