<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Plugin;
use Merkulove\Speaker\Unity\Settings;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove/Speaker
 */
final class PostEdit {

	/**
	 * @access private
	 * @var PostEdit
	 **/
	private static $instance;

	/**
	 * Post types with Speaker support.
	 * @access private
	 * @var array
	 **/
	private static $cpt_posts = [];

	/**
	 * Run actions and filters related to post edit.
	 * @return void
	 */
	private function __construct() {

		/** Options */
		$options = Settings::get_instance()->options;
		self::$cpt_posts = $options['cpt_support'] ?? array();
		if ( empty( self::$cpt_posts ) ) { return; }

		/** Add scripts and styles */
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_styles' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts' ] );

	}

	/**
	 * Enqueue scripts
	 * @return void
	 */
	public function enqueue_scripts() {

		/** Get current screen. */
		$screen = get_current_screen();
		if ( $screen === null ) { return; }

		if ( $screen->base !== 'edit' && in_array( $screen->post_type, self::$cpt_posts ) ) {

			wp_enqueue_script( 'merkulov-ui', Plugin::get_url() . 'src/Merkulove/Unity/assets/js/merkulov-ui' . Plugin::get_suffix() . '.js', [], Plugin::get_version(), true );
			wp_enqueue_script( 'mdp-sortable', Plugin::get_url() . 'js/Sortable' . Plugin::get_suffix() . '.js', [], Plugin::get_version(), true );
			wp_enqueue_script( 'data-tables', Plugin::get_url() . 'js/datatables' . Plugin::get_suffix() . '.js', [ 'jquery' ], Plugin::get_version(), true );

			wp_enqueue_script( 'speaker-admin-post', Plugin::get_url() . 'js/speaker-admin-post' . Plugin::get_suffix() . '.js', array(), Plugin::get_version(), true );
			wp_localize_script(
				'speaker-admin-post',
				'mdpSpeakerAdminPost',
				[
					'translations' => [

						'play_rate' => __( 'The ratio of clicks on the play button to the total number of visitors', 'speaker' ),
						'time_average' => __( 'The average number of seconds played', 'speaker' ),
						'init' => __( 'The number of times the player was initialized', 'speaker' ),
						'play' => __( 'The number of times the play button was clicked', 'speaker' ),
						'pause' => __( 'The number of times the pause button was clicked', 'speaker' ),
						'time' => __( 'The total number of seconds played', 'speaker' ),
						'bounce_rate' => __( 'The ratio of visitors who did not listen to the end of the audio', 'speaker' ),

						'noVoices' => __( 'No one found', 'speaker' ),
						'placeholderValue' => __( 'Choose an option', 'speaker' ),
						'searchPlaceholderValue' => __( 'Language or locale code', 'speaker' ),
						'loadingText' => __( 'Loading...', 'speaker' ),
						'noResultsText' => __( 'No results found', 'speaker' ),
						'noChoicesText' => __( 'No choices to choose from', 'speaker' ),
						'addItemText' => __( 'Press Enter', 'speaker' ),
						'voiceSelectLabel' => __( 'Post voice', 'speaker' ),

						'error' => __( 'Error', 'speaker' ),
						'done' => __( 'Done!', 'speaker' ),
						'requestError' => __( 'Request error', 'speaker' ),
						'requestAborted' => __( 'Request aborted', 'speaker' ),

						'youCannotDuplicate' => __( 'You cannot duplicate this template', 'speaker' ),

					],
					'metaboxOptions' => [

						'ajaxUrl' => admin_url( 'admin-ajax.php' ),
						'nonce' => wp_create_nonce( 'speaker-nonce-metabox' ),
						'pluginUrl' => Plugin::get_url(),

					],

				]
			);

			wp_enqueue_script( 'speaker-speech-templates', Plugin::get_url() . 'js/speaker-speech-templates' . Plugin::get_suffix() . '.js', ['jquery', 'mdp-sortable', 'data-tables'], Plugin::get_version(), true );
			wp_localize_script(
				'speaker-speech-templates',
				'mdpSpeaker',
				[
					'post_id'               => get_the_ID(), // Current post ID.
					'nonce'                 => wp_create_nonce( 'speaker-nonce' ), // Nonce for security.
					'audio_url'             => SpeakerHelper::get_instance()->get_audio_upload_url(), // Upload folder URL.
					'voice'                 => Settings::get_instance()->options['language'], // Default voice.
					'speechTemplateCount'   => count( MetaBox::get_instance()->get_st_options() )
				]
			);

		}

	}

	/**
	 * Enqueue styles
	 * @return void
	 */
	public function enqueue_styles() {

		/** Get current screen. */
		$screen = get_current_screen();
		if ( $screen === null ) { return; }

		if ( $screen->base !== 'edit' && in_array( $screen->post_type, self::$cpt_posts ) ) {

			/** Add class .mdc-disable to body. So we can use UI without overrides WP CSS, only for this page.  */
			add_action( 'admin_body_class', function( $classes ) {
				return $classes . ' mdc-disable ';
			} );

			wp_enqueue_style( 'merkulov-ui', Plugin::get_url() . 'src/Merkulove/Unity/assets/css/merkulov-ui' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );
			wp_enqueue_style( 'speaker-admin-post', Plugin::get_url() . 'css/admin-post' . Plugin::get_suffix() . '.css', [], Plugin::get_version() );

		}

	}

	/**
	 * Main Instance.
	 * @return PostEdit
	 **/
	public static function get_instance(): PostEdit {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
