<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Plugin;
use Merkulove\Speaker\Unity\Settings;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove/Speaker
 */
final class PostList {

	/**
	 * @access private
	 * @var PostList
	 **/
	private static $instance;

	/**
	 * Post types with Speaker support.
	 * @access private
	 * @var array
	 **/
	private static $cpt_posts = [];

	/**
	 * Run actions and filters related to post list.
	 * @return void
	 */
	private function __construct() {

		/** Options */
		$options = Settings::get_instance()->options;
		self::$cpt_posts = $options['cpt_support'] ?? array();
		if ( empty( self::$cpt_posts ) ) { return; }

		foreach ( self::$cpt_posts as $post_type ) {

			/** Add column to selected post types. */
			add_filter( "manage_" . $post_type . "_posts_columns", [ $this, 'add_head_column' ] );
			add_filter( "manage_edit-" . $post_type. "_columns", [ $this, 'add_head_column' ] );
			add_action( "manage_" . $post_type. "_posts_custom_column", [ $this, 'add_content_column' ], 20, 2 );

			/** Add Speaker bulk action to dropdown. */
			add_filter( "bulk_actions-edit-" . $post_type, [ $this, 'bulk_actions' ] );

		}

		/** Add scripts and styles */
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_styles' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts' ] );

	}

	/**
	 * Enqueue scripts
	 * @return void
	 */
	public function enqueue_scripts() {

		/** Get current screen. */
		$screen = get_current_screen();
		if ( $screen === null ) { return; }

		foreach ( self::$cpt_posts as $post_type ) {

			if ( $screen->id === 'edit-' . $post_type ) {

				wp_enqueue_script(
					'mdp-speaker-admin-edit',
					Plugin::get_url() . 'js/speaker-admin-edit' . Plugin::get_suffix() . '.js',
					[ 'jquery' ],
					Plugin::get_version(),
					true
				);

				wp_localize_script(
					'mdp-speaker-admin-edit',
					'mdpSpeaker',
					[
						'nonce' => wp_create_nonce( 'speaker-nonce' ),
						'audioUrl'  => SpeakerHelper::get_instance()->get_audio_upload_url(),
						'pluginUrl' => Plugin::get_url(),
						'translations' => [

							'done' => esc_html__( 'Done!', 'speaker' ),
							'error' => esc_html__( 'Error!', 'speaker' ),
							'complete' => esc_html__( 'Generation of Audio Files is complete.', 'speaker' ),

							'play_rate' => __( 'The ratio of clicks on the play button to the total number of visitors', 'speaker' ),
							'time_average' => __( 'The average number of seconds played', 'speaker' ),
							'init' => __( 'The number of times the player was initialized', 'speaker' ),
							'play' => __( 'The number of times the play button was clicked', 'speaker' ),
							'pause' => __( 'The number of times the pause button was clicked', 'speaker' ),
							'time' => __( 'The total number of seconds played', 'speaker' ),
							'bounce_rate' => __( 'The ratio of visitors who did not listen to the end of the audio', 'speaker' ),

						],
					]
				);

				break;

			}

		}

	}

	/**
	 * Enqueue styles
	 * @return void
	 */
	public function enqueue_styles() {

		/** Get current screen. */
		$screen = get_current_screen();
		if ( $screen === null ) { return; }

		foreach ( self::$cpt_posts as $post_type ) {

			if ( $screen->id === 'edit-' . $post_type ) {

				wp_enqueue_style(
					'mdp-speaker-admin-edit',
					Plugin::get_url() . 'css/admin-edit' . Plugin::get_suffix() . '.css',
					[],
					Plugin::get_version()
				);
				break;

			}

		}

	}

	/**
	 * Add HEAD for Speaker column with results.
	 * @param array $columns
	 * @return array
	 **/
	public function add_head_column( array $columns ): array {

		/** Work only with PUBLISHED posts or for ALL. */
		$post_status = filter_input( INPUT_GET, 'post_status' );
		if ( ! ( $post_status === 'publish' || $post_status === null ) ) { return $columns; }

		/** If we have title, comments column or author columns add after it. */
		$add_after = 'cb';
		if ( isset( $columns['title'] ) ) {
			$add_after = 'title';
		} elseif ( isset( $columns['comments'] ) ) {
			$add_after = 'comments';
		} elseif ( isset( $columns['author'] ) ) {
			$add_after = 'author';
		} elseif ( isset( $columns['date'] ) ) {
			$add_after = 'date';
		}

		$options = Settings::get_instance()->options;

		/** Add new column to the existing columns. */
		$new = [];
		foreach ( $columns as $key => $col ) {

			$new[$key] = $col;

			/** Add after comments column. */
			if ( $key === $add_after ) {

				/** @noinspection HtmlUnknownTarget */
				$new[ 'speaker_audio' ] = wp_sprintf(
					'<div class="mdp-speaker-head-column">
						<img src="%1$s" alt="%2$s"/>
						<span>%2$s</span>
					</div>',
					Plugin::get_url() . 'images/logo-color.svg',
					esc_attr( 'Speaker' )
				);

				if ( $options[ 'analytics' ] === 'on' && $options[ 'analytics_column' ] === 'on' ) {
					$new[ 'speaker_analytics' ] = wp_sprintf(
						'<div class="mdp-speaker-analytics-head-column">
						<span>%2$s</span>
					</div>',
						Plugin::get_url() . 'images/logo-color.svg',
						esc_attr( 'Playback analytics' )
					);
				}

			}

		}

		/** Return a new column array to WordPress. */
		return $new;

	}

	/**
	 * Add CONTENT for Speaker column with results.
	 * @param string $column_name
	 * @param $post_ID
	 * return void
	 **/
	public function add_content_column( string $column_name, $post_ID ) {

		if ( ! in_array( $column_name, [ 'speaker_audio', 'speaker_analytics' ] ) ) { return; }

		switch ( $column_name ) {

			case 'speaker_analytics':

				Analytics::get_instance()->analytics_column( $post_ID );
				break;

			case 'speaker_audio':

				$is_multipage = SpeakerHelper::is_multipage( $post_ID );

				if ( $is_multipage ) {

                    if ( AudioFile::audio_exists( $post_ID, 1 ) ) {
                        $this->download_button_group( $post_ID, $is_multipage );
                    } else if (SpeakerHelper::is_key_exists()){
                        $this->render_generate_btn( $post_ID );
                    }

				} else {

                    if ( AudioFile::audio_exists( $post_ID ) ) {
                        $this->download_button_group( $post_ID );
                    } elseif (SpeakerHelper::is_key_exists()) {
                        $this->render_generate_btn( $post_ID );
                    }

				}

				break;

			default:

				break;

		}

	}

	/**
	 * Render download audio button for bulk processing.
	 *
	 * @param int $post_id
	 * @param bool $is_multipage
	 *
	 * return void
	 */
	private function download_button_group( int $post_id, bool $is_multipage = false ) {

		if ( $is_multipage ) {

			for( $page_index = 1; $page_index <= SpeakerHelper::pages_counter( $post_id ); $page_index++ ) {

				$this->render_download_button( $post_id, $page_index );

			}

		} else {

			$this->render_download_button( $post_id );

		}

		$this->render_post_custom_template_name( $post_id );

	}

	/**
	 * Markup of generate audio button for bulk processing.
	 *
	 * @param int $post_id
	 * @param int $page_index
	 *
	 * @return void
	 */
	private function render_download_button( int $post_id, int $page_index = 0 ) {

		/** URL to download audio file. */
		$audio_url = SpeakerCaster::get_instance()->get_audio_url( $post_id, $page_index );

		/** @noinspection HtmlUnknownTarget */
		echo wp_sprintf(
			'<a href="%1$s" class="mdp-speaker-download" download="" title="%2$s">
				<img src="%3$s" alt="%2$s">
				<span>%4$s</span>',
			esc_url( $audio_url ),
			esc_html__( 'Download audio', 'speaker' ),
			esc_url( Plugin::get_url() . 'images/status/have-audio.svg' ),
			SpeakerCaster::get_instance()->get_audio_meta( $post_id, 'length_formatted', $page_index ) ?? ''
		);

		echo wp_sprintf(
			'<a href="#" class="mdp-speaker-gen" data-post-id="%s" data-stid="%s" style="display: none;"></a>',
			esc_attr( $post_id ),
			esc_attr( SpeechTemplates::get_instance()->get_post_template( $post_id ) )
		);

	}

	/**
	 * Render generate audio button for bulk processing.
	 * @param $post_id
	 **/
	private function render_generate_btn( $post_id ) {

		/** @noinspection HtmlUnknownTarget */
		echo wp_sprintf(
			'<a href="#" class="mdp-speaker-gen" data-post-id="%1$s" data-stid="%2$s" title="%3$s">
                <img src="%5$s" alt="%3$s">
                <span class="mdp-speaker-gen--create">%3$s</span>
                <span class="mdp-speaker-gen--processing">%4$s</span>
            </a>',
			esc_attr( $post_id ),
			esc_attr( SpeechTemplates::get_instance()->get_post_template( $post_id ) ),
			esc_html__( 'Create audio', 'speaker' ),
			esc_html__( 'Processing...', 'speaker' ),
			esc_attr( Plugin::get_url() . 'images/status/no-audio.svg' )
		);

		$this->render_post_custom_template_name( $post_id );

	}

	/**
	 * Print custom speech template name.
	 * @param $post_id
	 * @return void
	 **/
	private function render_post_custom_template_name( $post_id ) {

		/** If current post have custom Speech Template, show it. */
		$template_id = get_post_meta( $post_id, 'mdp_speaker_custom_speech_template', true );
		if ( ! $template_id ) { return; }

		/** Return Speech Template Name by ID. */
		$template_name = SpeechTemplates::get_instance()->get_speech_template_name( $template_id );

		/** Show custom Speech Template name. */
		if ( $template_name ) {

			/** @noinspection HtmlUnknownTarget */
			echo wp_sprintf(
				'<a href="%3$s" class="mdp-custom-template" title="%2$s">
							<img src="%1$s" alt="%2$s">
							<span>%2$s</span>
						</a>',
				esc_url( Plugin::get_url() . 'images/status/custom-template.svg' ),
				esc_attr( $template_name ),
				esc_url( get_edit_post_link( $post_id ) . '&speaker-action=edit-template' )
			);

		}

	}

	/**
	 * Add Speaker bulk action to dropdown.
	 * @param $actions
	 * @return mixed
	 **/
	public function bulk_actions( $actions ) {

		/** Work only with PUBLISHED posts or for ALL. */
		$post_status = filter_input( INPUT_GET, 'post_status' );

		if ( $post_status === 'publish' || $post_status === null ) {

			$actions['speaker'] = esc_html__( 'Create Audio', 'speaker' );

		}

		return $actions;

	}

	/**
	 * Main Instance.
	 * @return PostList
	 **/
	public static function get_instance(): PostList {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
