<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Alexander Khmelnitskiy (info@alexander.khmelnitskiy.ua), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\TabAssignments;
use Merkulove\Speaker\Unity\Settings;
use Merkulove\Speaker\Unity\Plugin as Plugin;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * SINGLETON: Class contain all Speaker logic.
 *
 * @since 3.0.0
 *
 **/
final class SpeakerCaster {

	/**
	 * The one true SpeakerCaster.
	 *
	 * @var SpeakerCaster
	 * @since 3.0.0
	 **/
	private static $instance;

	/**
	 * Add Ajax handlers and before_delete_post action.
	 *
	 * @since 3.0.0
	 * @access public
	 *
	 * @return void
	 **/
	public function add_actions() {

		/** Ajax Create Audio on Backend. */
		add_action( 'wp_ajax_gspeak', [ SpeechGeneration::get_instance(), 'google_speak' ] );

		/** Ajax Remove Audio on Backend. */
		add_action( 'wp_ajax_remove_audio', [ $this, 'remove_audio' ] );

		/** Remove audio file on remove post record. */
		add_action( 'before_delete_post', [ $this, 'before_delete_post' ] );

        /** Process Speech Template Create/Update/Delete on Backend. */
        add_action( 'wp_ajax_process_st', [ $this, 'process_st' ] );

        /** Get Speech Template Data by ID. */
        add_action( 'wp_ajax_get_st', [ $this, 'get_st_ajax'] );

        /** Set Speech Template as Default. */
        add_action( 'wp_ajax_set_default_st', [ $this, 'set_default_st' ] );

    }

    /**
     * Add accessibility elements for audio shortcode
     * @param $html
     * @return array|string|string[]
     */
    public function audio_shortcode_filter( $html ) {

        $id = get_the_ID();
        if ( ! AudioFile::audio_exists( $id ) ) { return $html; }

        $post = get_post( $id );
        $title = esc_html( $post->post_title );
        return str_replace( '<audio ', '<audio aria-label="' . esc_html__( 'Audio of', 'speaker' ) . ' ' .  $title . '"', $html );

    }

    /**
     * Speaker use custom page template to parse content without garbage.
     *
     * @param $template - The path of the template to include.
     *
     * @return mixed
     *
     * @access public
     */
    public static function speaker_page_template( $template ) {

        /** Template must be a string */
        if ( gettype( $template ) !== 'string' ) {
            return $template;
        }

        /** Change template for correct parsing content. */
        if ( isset( $_GET['speaker-template'] ) && 'speaker' === $_GET['speaker-template'] ) {

            /** Disable admin bar. */
            show_admin_bar( false );

            $template = Plugin::get_path() . 'src/Merkulove/Speaker/speaker-template.php';

        }

        return $template;

    }

    /**
     * Hide admin bar for Speech Template Editor.
     *
     * @since 3.0.0
     * @access public
     *
     * @return void
     **/
    public static function hide_admin_bar() {

        if ( isset( $_GET['speaker_speech_template'] ) && '1' === $_GET['speaker_speech_template'] ) {

            /** Hide admin bar for Speech Template Editor. */
            show_admin_bar( false );

        }

    }

	/**
	 * Get audio meta
	 *
	 * @param $id
	 * @param $key
	 * @param int $page_index
	 *
	 * @return false|void
	 */
    public function get_audio_meta( $id, $key, int $page_index = 0 ) {

        if ( ! is_admin() ) require_once ABSPATH . 'wp-admin/includes/media.php';

        $audio_path = $this->get_audio_path( $id, $page_index );
        $audio_meta = wp_read_audio_metadata( $audio_path );

        if ( ! is_array( $audio_meta ) ) { return false; }

        return $audio_meta[ $key ] ?? false;

    }

	/**
	 * Return URL to audio version of the post.
	 *
	 * @param int $id - Post/Page id.
	 * @param int $page_index - Page index.
	 *
	 * @return bool|string
	 */
	public function get_audio_url( int $id = 0, int $page_index = 0 ) {

        /** Get plugin settings. */
        $options = Settings::get_instance()->options;
        $audio_format = $options[ 'audio-format' ] ?? 'mp3';

		/** If audio file not exist. */
		$f_time = AudioFile::audio_exists( $id, $page_index );
		if ( ! $f_time ) { return false; }

		/** Current post ID. */
		if ( ! $id ) {

            /** @noinspection CallableParameterUseCaseInTypeContextInspection */
            $id = get_the_ID();

			if ( ! $id ) { return false; }

		}

		/** URL to post audio file. */
		$audio_url = AudioFile::url( $id, $page_index );

		/** Cache Busting. '.mp3' is needed. */
		$audio_url .= '?cb=' . $f_time . '.' . $audio_format;

		return $audio_url;
	}

	/**
	 * Return path to audio version of the post.
	 *
	 * @param int $id
	 * @param int $page_index
	 *
	 * @return false|string
	 */
    public function get_audio_path( int $id = 0, int $page_index = 0 ) {

        /** If audio file not exist. */
        $f_time = AudioFile::audio_exists( $id, $page_index );
        if ( ! $f_time ) { return false; }

        /** Current post ID. */
        if ( ! $id ) {

            /** @noinspection CallableParameterUseCaseInTypeContextInspection */
            $id = get_the_ID();

            if ( ! $id ) { return false; }

        }

        /** URL to post audio file. */
        return AudioFile::path( $id, $page_index );

    }

	/**
	 * Remove audio on remove post record.
	 *
	 * @param $post_id - The post id that is being deleted.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function before_delete_post( $post_id ) {

		// Remove Post audio.
		if ( AudioFile::audio_exists( $post_id ) ) {

			AudioFile::remove_by_id( $post_id );

		}

		// Remove audio in multipage posts.
		if ( SpeakerHelper::is_multipage( $post_id ) ) {

			for( $page_index = 1; $page_index <= SpeakerHelper::pages_counter( $post_id ); $page_index++ ) {

				if ( AudioFile::audio_exists( $post_id, $page_index ) ) {
					AudioFile::remove_by_id( $post_id, $page_index );
				}

			}

		}

	}

	/**
	 * Ajax Remove Audio action hook here.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function remove_audio() {

		/** Security Check. */
		check_ajax_referer( 'speaker-nonce', 'nonce' );

		/** Current post ID. */
		$post_id = (int)$_POST['post_id'];

		/** Remove multipage audio or single audio */
		if ( SpeakerHelper::is_multipage( $post_id ) ) {

			$pages_counter = SpeakerHelper::pages_counter( $post_id );

			for( $page_index = 1; $page_index <= $pages_counter; $page_index++ ) {

				/** Remove audio file. */
				wp_delete_file( AudioFile::path( $post_id, $page_index ) );

			}

		} else {

			/** Remove audio file. */
			wp_delete_file( AudioFile::path( $post_id ) );

		}

		echo 'ok';

		wp_die();

	}

    /**
     * Ajax set Speech template as default.
     *
     * @since  3.0.0
     * @access public
     *
     * @return void
     **/
	public function set_default_st() {

        /** Security Check. */
        check_ajax_referer( 'speaker-nonce', 'nonce' );

        /** Get Speech Template ID. */
        $stid = filter_input(INPUT_POST, 'stid' );

        /** Error, no Speech Template ID */
        if ( ! trim( $stid ) ) {

            $return = [
                'success' => false,
                'message' => esc_html__( 'Error: There are no Speech Template ID received.', 'speaker' ),
            ];

            wp_send_json( $return );

        }

        /** Get Post Type. */
        $post_type = filter_input(INPUT_POST, 'postType' );

        /** In this option we store all Speech Templates. */
        $st_opt_name = 'mdp_speaker_speech_templates';

        /** Get all Speech Templates. */
        $st = get_option( $st_opt_name, false );

        /** We haven't any ST. */
        if ( ! $st ) {

            $return = [
                'success' => false,
                'message' => 'Speech Templates not found.',
            ];

            wp_send_json( $return );

        }

        // Process each speech template.
        $st_updated = [];
        foreach( $st as $template ) {

            // Skip empty or broken templates
            if ( $template['id'] === '' ) { continue; }
            if ( ! isset( $template[ 'default'] ) ) { continue; }

            // Erase the default template for this Post type.
            if ( in_array( $post_type, $template['default'] ?? [] ) ) {

                $default = [];

                // Remove post_type from all Speech Templates as default.
                foreach ( $template['default'] as $pt ) {

                    // Skip the current post type.
                    if ( $pt === $post_type ) { continue; }
                    $default[] = $pt;

                }

            } else {

                $default = $template['default'];

            }

            // Set Content ST as default for this post types.
            if ( $stid !== 'content' && $template['id'] === $stid ) {

                $default[] = $post_type;

            }

            if ( is_array( $default ) ) {
                $template[ 'default' ] = array_unique( $default );
            }

            // Add updated template to a new array.
            $st_updated[] = $template;

        }

        /** Update Speech Templates in option. */
        $updated = update_option( $st_opt_name, $st_updated );

        if ( ! $updated ) {

            $return = [
                'success' => false,
                'message' => esc_html__( 'Failed to install Speech Template as Default.', 'speaker' ),
            ];

            wp_send_json( $return );

        }

        $return = [
            'success' => true,
            'message' => esc_html__( 'Speech Template is installed as Default successfully.', 'speaker' ),
        ];

        wp_send_json( $return );

    }

    /**
     * Return Speech Template data by STID.
     *
     * @param $stid
     *
     * @since  3.0.0
     * @access public
     *
     * @return array|false
     **/
    public function get_st( $stid ) {

        /** In this option we store all Speech Templates. */
        $st_opt_name = 'mdp_speaker_speech_templates';

        /** Get all Speech Templates. */
        $st = get_option( $st_opt_name, false );

        /** We haven't any ST. */
        if ( ! $st ) { return false; }

        /** Search for existing st. */
        foreach ( $st as $key => $template ) {

            /** Update Speech template if we found same id. */
            if ( $template['id'] === $stid ) {

                return $st[$key];

            }

        }

        return false;

    }

    /**
     * Ajax get Speech template data.
     *
     * @since  3.0.0
     * @access public
     *
     * @return void
     **/
	public function get_st_ajax() {

        /** Security Check. */
        check_ajax_referer( 'speaker-nonce', 'nonce' );

        /** Get Speech Template ID. */
        $stid = filter_input(INPUT_POST, 'stid' );

        /** In this option we store all Speech Templates. */
        $st_opt_name = 'mdp_speaker_speech_templates';

        /** Get all Speech Templates. */
        $st = get_option( $st_opt_name, false );

        /** We haven't any ST. */
        if ( ! $st ) {

            $return = [
                'success' => false,
                'message' => 'Speech Templates not found.',
            ];

            wp_send_json( $return );

        }

        /** Search for existing st. */
        foreach ( $st as $key => $template ) {

            /** Update Speech template if we found same id. */
            if ( $template['id'] === $stid ) {

                $return = [
                    'success' => true,
                    'message' => $st[$key],
                ];

                wp_send_json( $return );

                break;

            }

        }

        /** Add new one if not found st with same id. */
        $return = [
            'success' => false,
            'message' => esc_html__( 'Speech Template not found.', 'speaker' )
        ];

        wp_send_json( $return );

    }

    /**
     * Ajax Create/Update/Delete Speech template.
     *
     * @since  3.0.0
     * @access public
     *
     * @return void
     **/
	public function process_st() {

        /** Security Check. */
        check_ajax_referer( 'speaker-nonce', 'nonce' );

        /** Get Speech Template data. */
        $st = filter_input(INPUT_POST, 'st' );

        /** Speech Template JSON to Object. */
        $st = json_decode( $st, true );

        /** Add default if needed */
        if ( ! isset( $st['default'] ) ) { $st['default'] = []; }

        /** Do we delete this Speech Template? */
        $delete = filter_input(INPUT_POST, 'delete', FILTER_VALIDATE_BOOLEAN );

        /** Remove Speech Template, */
        if ( $delete ) {

            if ( $this->delete_st( $st['id'] ) ) {

                $return = [
                    'success' => true,
                    'message' => esc_html__( 'Speech Template removed successfully.', 'speaker' )
                ];

            /** On fail. */
            } else {

                $return = [
                    'success' => false,
                    'message' => esc_html__( 'Failed to remove Speech Template', 'speaker' )
                ];

            }

            wp_send_json( $return );

        }

        /** Update or create Speech Template. */
        if ( ! $this->update_st( $st ) ) {

            $return = [
                'success' => false,
                'message' => esc_html__( 'Failed to update Speech Template.', 'speaker' )
            ];

            wp_send_json( $return );

        }

        $return = [
            'success' => true,
            'message' => esc_html__( 'Speech Template updated successfully.', 'speaker' )
        ];

        wp_send_json( $return );

    }

    /**
     * Update existing or create new Speech Template.
     *
     * @param $new_st
     *
     * @since  3.0.0
     * @access public
     *
     * @return boolean
     **/
    private function update_st( $new_st ) {

        /** In this option, we store all Speech Templates. */
        $st_opt_name = 'mdp_speaker_speech_templates';

        /** Get all Speech Templates. */
        $st = get_option( $st_opt_name, false );

        /** We haven't any ST. */
        if ( ! $st ) {

            /** Add first one. */
            $st = [];
            $st[] = $new_st;

        /** Search for existing st. */
        } else {

            $found = false;
            foreach ( $st as $key => $template ) {

                /** Update Speech template if we found same id. */
                if ( $template['id'] === $new_st['id'] ) {

                    $found = true;
                    $st[$key] = $new_st;

                    break;

                }

            }

            /** Add new one if not found st with same id. */
            if ( ! $found ) {
                $st[] = $new_st;
            }

        }

        /** Save Speech Templates in option. */
        $updated = update_option( $st_opt_name, $st, false );
        if ( ! $updated ) { return false; }

        return true;

    }

    /**
     * Remove Speech Template by ID.
     *
     * @param string $id - Unique id of Speech Template.
     *
     * @since  3.0.0
     * @access public
     *
     * @return boolean
     **/
    private function delete_st( $id ) {

        /** In this option we store all Speech Templates. */
        $st_opt_name = 'mdp_speaker_speech_templates';

        /** Get all Speech Templates. */
        $st = get_option( $st_opt_name, false );

        /** We haven't any ST, nothing to remove. */
        if ( ! $st ) { return true; }

        /** Search for existing st. */
        foreach ( $st as $key => $template ) {

            /** Remove Speech template if we found same id. */
            if ( $template['id'] === $id ) {

                unset( $st[$key] ); // Remove ST.

                break;

            }

        }

        /** Save Speech Templates in option. */
        $updated = update_option( $st_opt_name, $st, false );
        if ( ! $updated ) { return false; }

        return true;

    }

    /**
     * Display structured data
     * @return void
     */
    public function structured_data() {

        $post_id = get_the_ID();

        /** Get Plugin Settings. */
        $options = Settings::get_instance()->options;

        /** Checks if plugin should work on this page. */
        if( ! TabAssignments::get_instance()->display() ) { return; }

        /** Exit on the Category and Tag pages */
        if ( is_archive() || is_404()||  is_category() || is_tag() || is_attachment() ) { return; }

        /** Exit is post type not listed in settings */
        if ( ! is_singular( $options[ 'cpt_support' ] ) ) { return; }

        /** Show ld+json Markup if it is enabled in settings. */
        if ( 'on' !== $options['schema'] || $options['json_ld'] === '' ) { return; }

        /** Return if post have no one audio file and schema for all posts is disabled */
        if ( $options[ 'schema_for_all' ] !== 'on' && ! AudioFile::audio_exists( $post_id ) ) { return; }

        /** Get JSON+LD Markup from settings. */
        $json_ld = $options['json_ld'];

        /** Get post info */
        $post = get_post( $post_id );
        $type = get_post_type( $post_id ) === 'page' ? 'WebPage' : 'Article';
        $title = esc_html( $post->post_title );
        $permalink = esc_url( get_permalink( $post_id ) );

        /** Apply variables replacements. */
        $json_ld = str_replace( [ '[title]', '[type]', '[permalink]' ], [ $title, $type, $permalink ], $json_ld );

        /** Output ld+json Markup. */
        printf( '<script id="mdp-speaker-speakable" type="application/ld+json">%s</script>', $json_ld );

    }

	/**
	 * Main SpeakerCaster Instance.
	 *
	 * Insures that only one instance of SpeakerCaster exists in memory at any one time.
	 *
	 * @static
	 * @return SpeakerCaster
	 * @since 3.0.0
	 **/
	public static function get_instance() {

        /** @noinspection SelfClassReferencingInspection */
        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof SpeakerCaster ) ) {

            /** @noinspection SelfClassReferencingInspection */
            self::$instance = new SpeakerCaster;

		}

		return self::$instance;

	}

} // End Class SpeakerCaster.
