<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

use Merkulove\Speaker\Unity\Plugin as Plugin;
use Merkulove\Speaker\Unity\Settings;
use WP_Filesystem_Direct;

/**
 * SINGLETON: Class used to implement work with WordPress filesystem.
 *
 * @since 2.0.0
 *
 **/
final class SpeakerHelper {

	/**
	 * The one true Helper.
	 *
	 * @var SpeakerHelper
	 * @since 1.0.0
	 **/
	private static $instance;

    /**
     * Parse a string between two strings.
     *
     * @param string $string
     * @param string $start
     * @param string $end
     *
     * @since  3.0.0
     * @access public
     *
     * @return string
     **/
	public function get_string_between( $string, $start, $end ) {

        $string = ' ' . $string;
        $ini = strpos( $string, $start );
        if ( $ini === 0 ) { return ''; }

        $ini += strlen( $start );
        $len = strpos( $string, $end, $ini ) - $ini;

        return substr( $string, $ini, $len );

    }

	/**
	 * Delete Plugin Options.
	 *
	 * @since 3.0.0
	 * @access public
	 **/
	public static function remove_settings() {

		$settings = [
			'mdp_speaker_envato_id',
			'mdp_speaker_settings',
			'mdp_speaker_design_settings',
            'mdp_speaker_post_types_settings',
            'mdp_speaker_storage_settings',
			'mdp_speaker_assignments_settings',
			'mdp_speaker_uninstall_settings',
			'mdp_speaker_updates_settings',
			'mdp_speaker_developer_settings',
            'mdp_speaker_send_action_install',
            'mdp_speaker_speech_templates',
            'mdp_speaker_gd_token',
            'mdp_speaker_gd_folder',
			'envato_purchase_code_24336046'
		];

		/** For Multisite. */
		if ( is_multisite() ) {

			foreach ( $settings as $key ) {

				if ( ! get_site_option( $key ) ) { continue; }

				delete_site_option( $key );

			}

			/** For Singular site. */
		} else {

			foreach ( $settings as $key ) {

				if ( ! get_option( $key ) ) { continue; }

				delete_option( $key );

			}

		}

	}

	/**
	 * Remove all speaker audio files.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return void
	 **/
	public function remove_audio_files() {

		/** Remove /wp-content/uploads/speaker/ folder. */
		$dir = trailingslashit( wp_upload_dir()['basedir'] ) . 'speaker';
		$this->remove_directory( $dir );

	}

	/**
	 * Remove directory with all contents.
	 *
	 * @param $dir - Directory path to remove.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return void
	 **/
	public function remove_directory( $dir ) {

		require_once ( ABSPATH . 'wp-admin/includes/class-wp-filesystem-base.php' );
		require_once ( ABSPATH . 'wp-admin/includes/class-wp-filesystem-direct.php' );
		$fileSystemDirect = new WP_Filesystem_Direct( false );
		$fileSystemDirect->rmdir( $dir, true );

	}

	/**
	 * Get remote contents.
	 *
	 * @access public
	 * @since 2.0.0
	 * @param  string $url  The URL we're getting our data from.
	 *
	 * @return false|string The contents of the remote URL, or false if we can't get it.
	 **/
	public function get_remote( $url ) {

		$args = [
			'timeout'    => 30,
			'user-agent' => 'speaker-user-agent',
            'sslverify'  => false
		];

		$response = wp_remote_get( $url, $args );
		if ( is_array( $response ) ) {
			return $response['body'];
		}

		// TODO: Add a message so that the user knows what happened.
		/** Error while downloading remote file. */
		return false;
	}

	/**
	 * Send Action to our remote host.
	 *
	 * @param $action - Action to execute on remote host.
	 * @param $plugin - Plugin slug.
	 * @param $version - Plugin version.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 **/
	public function send_action( $action, $plugin, $version ) {

        if ( class_exists('Merkulove\Speaker\Settings') ) {
            if ( Settings::get_instance()->options[ 'check_server' ] === 'off' ) { return false; }
        }

		$domain = parse_url( site_url(), PHP_URL_HOST );
		$admin = base64_encode( get_option( 'admin_email' ) );
		$pid = get_option( 'envato_purchase_code_24336046' );

		$curl = curl_init();

		$url = 'https://merkulove.host/wp-content/plugins/mdp-purchase-validator/src/Merkulove/PurchaseValidator/Validate.php?';
		$url .= 'action=' . $action . '&'; // Action.
		$url .= 'plugin=' . $plugin . '&'; // Plugin Name.
		$url .= 'domain=' . $domain . '&'; // Domain Name.
		$url .= 'version=' . $version . '&'; // Plugin version.
		$url .= 'pid=' . $pid . '&'; // Purchase Code.
		$url .= 'admin_e=' . $admin;

		curl_setopt( $curl, CURLOPT_URL, $url );
		curl_setopt( $curl, CURLOPT_RETURNTRANSFER, 1 );

		curl_exec( $curl );

	}

	/**
	 * Create folder for audio files.
	 *
	 * @since 3.0.0
	 * @access private
	 * @return void
	 **/
	public function create_speaker_folder() {

		/** Create /wp-content/uploads/speaker/ folder. */
		wp_mkdir_p( trailingslashit( wp_upload_dir()['basedir'] ) . 'speaker' );

	}

	/**
	 * Return URL to  audio upload folder.
	 *
	 * @return string
	 * @since 3.0.0
	 * @access public
	 **/
	public function get_audio_upload_url() {

		/** Get URL to upload folder. */
		$upload_dir     = wp_get_upload_dir();
		$upload_baseurl = $upload_dir['baseurl'];

		/** URL to audio folder. */
		return $upload_baseurl . '/speaker/';

	}

    /**
     * Return true if remote requests not suspended.
     *
     * @since 3.0.4
     * @access public
     *
     * @return bool
     **/
    public function can_do_remote_requests() {

        /** If we have this option all remote requests stopped. */
        $opt_pause = 'mdp_speaker_pause_remote_requests';

        /** Trying to get from cache first. */
        $pause = get_transient( $opt_pause );

        return false === $pause;

    }

    /**
     * Suspend remote requests for 1 hour.
     *
     * @since 3.0.4
     * @access public
     *
     * @return void
     **/
    public function suspend_remote_requests() {

        /** If we have this option all remote requests stopped. */
        $opt_pause = 'mdp_speaker_pause_remote_requests';

        set_transient( $opt_pause, '1', 3600 ); // 1 Hour.

    }

    /**
     * Return allowed tags for wp_kses filtering with svg tags support.
     *
     * @since 3.0.5
     * @access public
     *
     * @return array
     **/
    public static function get_kses_allowed_tags_svg() {

        /** Allowed HTML tags in post. */
        $kses_defaults = wp_kses_allowed_html( 'post' );

        /** Allowed HTML tags and attributes in svg. */
        $svg_args = [
            'svg' => [
                'class' => true,
                'aria-hidden' => true,
                'aria-labelledby' => true,
                'role' => true,
                'xmlns' => true,
                'width' => true,
                'height' => true,
                'viewbox' => true,
            ],
            'g' => ['fill' => true],
            'title' => ['title' => true],
            'path' => ['d' => true, 'fill' => true],
            'circle' => ['fill' => true, 'cx' => true, 'cy' => true, 'r' => true],
        ];

        return array_merge( $kses_defaults, $svg_args );

    }

	/**
	 * Get post locale for multilingual sites.
	 *
	 * @return mixed|string|null
	 */
	public static function get_post_locale( $post_id = null ) {

        // Is WP Globus active?
        if ( class_exists( 'WPGlobus' ) ) {

            $locale = get_post_meta( $post_id, 'wpglobus_language', true );

        // WPML and Polylang
        } else if ( $post_id ) {

			$post_language_details = apply_filters( 'wpml_post_language_details', NULL, $post_id );
            $locale = $post_language_details[ 'language_code' ] ?? '';

            if ( function_exists( 'pll_get_post_language' ) ) {
                $locale = pll_get_post_language( $post_id ) ?? $locale;
            }

		} else {

			$locale = apply_filters( 'wpml_current_language', NULL );

		}

        if ( ! $locale ) { $locale = get_locale(); }

		return $locale;

	}

	/**
	 * Return language code from voice code.
	 *
	 * @param $voice
	 *
	 * @return string
	 */
	public static function lang_from_voice( $voice ): string {

		$pieces = explode('-', $voice );
		return $pieces[0] . '-' . $pieces[1];

	}

	/**
	 * Return saved locales as array with language codes.
	 *
	 * @param array $options
	 *
	 * @return array
	 */
	public static function get_saved_locales( array $options = array() ): array {

		$locales = array();
		foreach ( $options as $key => $lang_code ) {

			// If locale exists and not empty
			if ( strpos( $key, 'multilang_locale_' ) !== false && ! empty( $lang_code ) ) {

				$locales[] = [
					'lang_code' => $lang_code,
					'voice_name' => $options[ 'multilang_voice_' . str_replace( 'multilang_locale_', '', $key ) ] ?? '',
				];

			}

		}

		return $locales;

	}

	/**
	 * Get multilingual properties.
	 *
	 * @param array $options
	 * @param string $locale
	 *
	 * @return array
	 */
	public static function multilang_properties( array $options = array(), string $locale = '' ): array {

		$lang_code = '';
		$voice_name = '';
		foreach ( SpeakerHelper::get_saved_locales( $options ) as $saved ) {

			// If locale is equal to saved locale
			if ( $saved[ 'lang_code' ] === $locale || strpos( $saved[ 'lang_code' ], $locale ) === 0 ) {
				$lang_code = $saved[ 'lang_code' ];
				$voice_name = $saved[ 'voice_name' ];
				break;
			}

		}

		return [ $lang_code, $voice_name ];

	}

	/**
	 * Return true if post is multipage.
	 *
	 * @param int $post_id
	 *
	 * @return bool
	 */
	public static function is_multipage( int $post_id ): bool {

		$the_post = get_post( $post_id );

		return isset( $the_post->post_content ) && strpos( $the_post->post_content, '<!--nextpage-->' ) !== false;

	}

	/**
	 * Get pages counter for post
	 *
	 * @param int $post_id
	 *
	 * @return int
	 */
	public static function pages_counter( int $post_id ): int {

		$the_post = get_post( $post_id );

		return substr_count( $the_post->post_content, '<!--nextpage-->' ) + 1;

	}

	/**
	 * Get a vendor path related to a PHP version.
	 * @return string
	 */
	public static function get_vendor_path(): string {

        $vendor_version = self::get_vendor_version();

        switch ( $vendor_version ) {
            case 7:
                $speaker_php7_path = str_replace( 'plugins/speaker', 'plugins/speaker-php7', Plugin::get_path() );
                return $speaker_php7_path . 'vendor/autoload.php';
            case 8:
            default:
                return Plugin::get_path() . 'vendor/autoload.php';
        }

	}

    /**
     * Get a vendor version.
     * @return int
     */
    public static function get_vendor_version(): int {

        // Is SpeakerPHP7 not active?
        if ( ! is_plugin_active( 'speaker-php7/speaker-php7.php' ) ) {
            return 8;
        }

        // Get SpeakerPHP7 options
        $options = Settings::get_instance()->options;
        if ( ! isset( $options[ 'vendor_version' ] ) ) {
            return 8;
        }

        switch ( $options[ 'vendor_version' ] ) {
            case '7.4':
                return 7;
            case '8.1':
            default:
                return 8;
        }

    }

    /**
     * Return true if key exists
     * @return bool
     */
    public static function is_key_exists(): bool {

        $options = Settings::get_instance()->options;

        return isset( $options[ 'dnd-api-key' ] ) && $options[ 'dnd-api-key' ] !== '';

    }

	/**
	 * @return SpeakerHelper
	 */
	public static function get_instance(): SpeakerHelper {

		if ( ! isset( SpeakerHelper::$instance ) && ! ( SpeakerHelper::$instance instanceof SpeakerHelper ) ) {

			SpeakerHelper::$instance = new SpeakerHelper;

		}

		return SpeakerHelper::$instance;

	}

}
