<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Alexander Khmelnitskiy (info@alexander.khmelnitskiy.ua), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use DOMException;
use DOMXPath;
use DOMDocument;
use Exception;
use Google\Cloud\TextToSpeech\V1\SynthesizeSpeechRequest;
use Google\Cloud\TextToSpeech\V1\TextToSpeechClient as TextToSpeechClient7;
use Google\Cloud\TextToSpeech\V1\Client\TextToSpeechClient as TextToSpeechClient8;
use Google\ApiCore\ApiException;
use Google\Cloud\TextToSpeech\V1\AudioConfig;
use Google\Cloud\TextToSpeech\V1\SynthesisInput;
use Google\Cloud\TextToSpeech\V1\VoiceSelectionParams;
use Merkulove\Speaker\Unity\Settings;
use Merkulove\Speaker\Unity\Plugin as Plugin;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove\Speaker
 * @since 4.0.0
 */
final class SpeechGeneration {

	/**
	 * @var SpeechGeneration
	 */
	private static $instance;

    private static $html_lang_code;

	/**
	 * Ajax action to Create Audio with Google Cloud Text-to-Speech.
	 * @return void
	 * @throws DOMException
	 */
	public function google_speak() {

		/** Security Check. */
		check_ajax_referer( 'speaker-nonce', 'nonce' );

		/** Current post ID. */
		$post_id = (int)$_POST['post_id'] ?? 0;

		/** Get Speech Template ID. */
		$stid = filter_input(INPUT_POST, 'stid' );

		wp_send_json( $this->post_audio_generation( $post_id, $stid ) );

	}

	/**
	 * @throws DOMException
	 */
	private function post_audio_generation( $post_id, $stid ): array {

		$voice_acting_status = 0;

		// Is multipage post?
		if ( SpeakerHelper::is_multipage( $post_id ) ) {

			$message = '';

			// Loop through pages
			for ( $page_index = 1; $page_index <= SpeakerHelper::pages_counter( $post_id ); $page_index++ ) {

				$voice_acting_status = $this->voice_acting( $post_id, $stid, array(), $page_index );

				if ( $voice_acting_status ) {
					$message .= wp_sprintf(
						/* translators: %d: page number */
						esc_html__( 'Page %d: Audio Generated Successfully', 'speaker' ),
						$page_index
					);
				} else {
					$message .= wp_sprintf(
						/* translators: %d: page number */
						esc_html__( 'Page %d: An error occurred while generating the audio.', 'speaker' ),
						$page_index
					);
				}

			}

		} else {

			$voice_acting_status = $this->voice_acting( $post_id, $stid );

			if ( $voice_acting_status ) {
				$message = esc_html__( 'Audio Generated Successfully', 'speaker' );
			} else {
				$message = esc_html__( 'The page does not contain content for voicing in accordance with the Speech Template. ', 'speaker' );
			}

		}

		// Re-creat custom RSS file
		do_action( 'speaker_update_custom_rss_file' );

		return array(
			'success' => (bool) $voice_acting_status,
			'message' => $message
		);

	}

	/**
	 * Let me speak. Create audio version of post.
	 *
	 * @param int $post_id Post ID
	 * @param string $stid Speech Template ID
	 * @param array $html_parts
	 * @param int $page_index
	 *
	 * @return boolean
	 * @throws DOMException
	 */
	public function voice_acting( int $post_id = 0, string $stid = 'content', array $html_parts = array(), int $page_index = 0 ): bool {

		$parts = Parser::content_parts( $stid, $post_id, $page_index, $html_parts );
		if ( empty( $parts ) ) {
			return false;
		}

		/** Create audio file for each part. */
		$audio_parts = [];
		foreach ( $parts as $part ) {

			try {

				/** Convert HTML to temporary audio file. */
                $part = $this->part_speak( $part, $post_id );
                if ( $part ) {
                    $audio_parts[] = $part;
                }

			} catch ( ApiException $e ) {

				/** Show error message. */
				echo esc_html__( 'Caught exception: ', 'speaker' ) . $e->getMessage() . "\n";

				// Remove temporary audio files
				foreach ( $audio_parts as $temp_audio_part ) {
					wp_delete_file( $temp_audio_part );
				}

			}

		}

		/** Combine multiple files to one. */
		$this->glue_audio( $audio_parts, $post_id, $page_index );

		return true;

	}

	/**
	 * Prepare parts for generate audio for whole post content.
	 *
	 * @param int $post_id
	 * @param int $page_index
	 *
	 * @return array
	 * @throws DOMException
	 */
	public function content_based_generation( int $post_id, int $page_index ): array {

		$options = Settings::get_instance()->options;

		/**
		 * Get Current Post-Content.
		 * Many shortcodes do not work in the admin area, so we need this trick.
		 * We open the frontend page in custom template and parse content.
		 **/
		$post_content = Parser::parse_post_content( $post_id, $page_index, 'speaker' );

        /** Get html lang code */
        if ( $options[ 'multilang' ] === 'on' ) {
            $html_tag = SpeakerHelper::get_instance()->get_string_between($post_content, '<html', '>');
            self::$html_lang_code = SpeakerHelper::get_instance()->get_string_between($html_tag, 'lang="', '"');
        }

		/** Get only content part from full page. */
		$post_content = SpeakerHelper::get_instance()->get_string_between( $post_content, '<div class="mdp-speaker-content-start"></div>', '<div class="mdp-speaker-content-end"></div>' );

		/**
		 * Filters the post content before any manipulation.
		 *
		 * @since 3.0.0
		 *
		 * @param string    $post_content   Post content.
		 * @param int       $post_id        Post ID.
		 **/
		$post_content = apply_filters( 'speaker_before_content_manipulations', $post_content, $post_id );

		$post_content = Parser::regex_content_replace( $post_content );

        $post_content = $this->markup_pause( $options, $post_content );

		$post_content = Parser::clean_content( $post_content );

		/**
		 * Filters the post content before split to parts by 4500 chars.
		 *
		 * @since 3.0.0
		 *
		 * @param string    $post_content   Post content.
		 * @param int       $post_id        Post ID.
		 **/
		$post_content = apply_filters( 'speaker_before_content_dividing', $post_content, $post_id );

		/** If all content is bigger than the quota. */
		$parts[] = $post_content;
		if ( strlen( $post_content ) >= (int)$options[ 'part_length' ] ) {

			/**
			 * Split to parts. Google have limits Total characters per request.
			 * See: https://cloud.google.com/text-to-speech/quotas
			 **/
			$parts = Parser::great_divider( $post_id, $post_content, (int)$options[ 'part_length' ] );

		}

		/**
		 * Filters content parts before voice_divider.
		 *
		 * @since 3.0.0
		 *
		 * @param string    $parts          Post content parts.
		 * @param int       $post_id        Post ID.
		 **/
		$parts = apply_filters( 'speaker_before_voice_divider', $parts, $post_id );

		/** Divide parts by voice. One part voiced by one voice */
		$parts = $this->voice_divider( $parts );

		/**
		 * Filters content parts before adding watermarks.
		 *
		 * @since 3.0.0
		 *
		 * @param string    $parts          Post content parts.
		 * @param int       $post_id        Post ID.
		 **/
		$parts = apply_filters( 'speaker_before_adding_watermarks', $parts, $post_id );

		/** Add custom text before/after audio. */
		return $this->add_watermark( $parts );

	}

	/**
	 * Combine multiple audio files to one .mp3.
	 *
	 * @param $files - Audio files for gluing into one big.
	 * @param $post_id - ID of the Post/Page.
	 *
	 * @since 3.0.0
	 * @access public
	 *
	 * @return void
	 **/
	public function glue_audio( $files, $post_id, $page_index ) {

		/** Get plugin settings. */
		$options = Settings::get_instance()->options;

		/** Path to post audio file. */
		$audio_file = AudioFile::path( $post_id, $page_index );

		/** Just in case, if it exists. */
		wp_delete_file( $audio_file );

		/** Clue files content */
//        if ( $options['audio_encoding'] == 'LINEAR16') {
//
//            file_put_contents( $audio_file, $this->joinwavs( $files ), FILE_APPEND );
//
//        }

		foreach ( $files as $audio ) {

			/** Add new audio part to file. */
			file_put_contents( $audio_file, file_get_contents( $audio ), FILE_APPEND );

			/** Remove temporary audio files. */
			wp_delete_file( $audio );

		}

		/** Store file meta to the post meta */
		if ( $options[ 'post_meta' ] === 'on' ) {

			$this->create_post_meta( $post_id );

		}

		/** Create Media Library record */
		if ( $options[ 'media_library' ] === 'on' ) {

			Attachment::get_instance()->create_attachment( $post_id, $page_index );

		}

		/** Google drive uploading */
		if ( in_array( $options[ 'storage_select' ], [ 'drive', 'library+drive' ] ) ) {

			StorageGoogle::get_instance()->gd_upload_file( $audio_file, $post_id );

		}

	}

	/** Join WAV with joining file header
	 * @param $wavs
	 * @return string
	 */
	private function joinwavs($wavs){

		$fields = join('/',array( 'H8ChunkID', 'VChunkSize', 'H8Format',
			'H8Subchunk1ID', 'VSubchunk1Size',
			'vAudioFormat', 'vNumChannels', 'VSampleRate',
			'VByteRate', 'vBlockAlign', 'vBitsPerSample' ));

		$data = '';

		foreach($wavs as $wav){
			$fp     = fopen($wav,'rb');
			$header = fread($fp,36);
			$info   = unpack($fields,$header);
			// read optional extra stuff
			if($info['Subchunk1Size'] > 16){
				$header .= fread($fp,($info['Subchunk1Size']-16));
			}
			// read SubChunk2ID
			$header .= fread($fp,4);
			// read Subchunk2Size
			$size  = unpack('vsize',fread($fp, 4));
			$size  = $size['size'];
			// read data
			$data .= fread($fp,$size);
		}

		return $header.pack('V',strlen($data)).$data;

	}

	/**
	 * Get audio meta and store it in the post meta
	 *
	 * @param $post_id
	 * @return void
	 */
	private function create_post_meta( $post_id ) {

		$audio_file = AudioFile::path( $post_id );
		$audio_url = AudioFile::url( $post_id );

		/** Store file meta to the post meta */
		if( ! is_admin() ) require_once ABSPATH . 'wp-admin/includes/media.php';
		$audio_meta = wp_read_audio_metadata( $audio_file );
		if ( ! is_array( $audio_meta ) ) { return; }

		/** Store file props to post the post meta */
		update_post_meta( $post_id, 'speaker-url', $audio_url );
		update_post_meta( $post_id, 'speaker-timestamp', current_time( 'r' ) );
		update_post_meta( $post_id, 'speaker-duration', $audio_meta[ 'length_formatted' ] );
		update_post_meta( $post_id, 'speaker-filesize', $audio_meta[ 'filesize' ] );

	}

	/**
	 * Convert HTML to temporary audio file.
	 *
	 * @param $html - Content to be voiced.
	 * @param $post_id - ID of the Post/Page.
	 *
	 * @return string
     * @since 3.0.0
	 * @access public
	 **/
	public function part_speak( $html, $post_id ) {

		/**
		 * Filters html part before speak it.
		 *
		 * @since 3.0.0
		 *
		 * @param string    $html       Post content part.
		 * @param int       $post_id    Post ID.
		 **/
		$html = apply_filters( 'speaker_before_part_speak', $html, $post_id );

		/** Strip all html tags, except SSML tags.  */
		$html = strip_tags( $html, '<p><break><say-as><sub><emphasis><prosody><voice>');

		/** Remove the white spaces from the left and right sides.  */
		$html = trim( $html );

		/** Convert HTML entities to their corresponding characters: &quot; => " */
		$html = html_entity_decode( $html );

		/**
		 * Replace special characters with HTML Ampersand Character Codes.
		 * These codes prevent the API from confusing text with SSML tags.
		 * '&' --> '&amp;'
		 **/
		$html = str_replace( '&', '&amp;', $html );

		/**
		 * Remove soft hyphens.
		 * '&shy;' --> ''
		 */
		$html = str_replace('&shy;','', $html );

        /** Get plugin settings. */
        $options = Settings::get_instance()->options;

		/** Get voice properties */
		list( $lang_code, $lang_name ) = $this->voice_properties( $post_id, $html, $options );

		/** We don’t need <voice> tag anymore. */
		$html = strip_tags( $html, '<p><break><say-as><sub><emphasis><prosody>');

		/** Add punctuation pause. */
		$html = $this->punctuation_pause( $options, $html );

        // Remove empty p
        if ( str_contains( $html, '<p>' ) ) {
            $html = preg_replace( '/<p>\s*<\/p>/', '', $html );
        }

        /** Force to SSML. */
        $ssml = "<speak>";
        $ssml .= $html;
        $ssml .= "</speak>";

        /** Remove all markup for Chirp(previously Journey) */
        if ( !$this->is_ssml_supported($lang_name) ) {
            $ssml = strip_tags( $ssml );
        }

		/**
		 * Filters $ssml content before Google Synthesis it.
		 *
		 * @since 3.0.0
		 *
		 * @param string    $ssml       Post content part.
		 * @param int       $post_id    Post ID.
		 **/
		$ssml = apply_filters( 'speaker_before_synthesis', $ssml, $post_id, $lang_code, $lang_name );

        // Check if SSML is empty
        if (trim($ssml) === '') {
            return null;
        }

		/** Perform text-to-speech request on the text input with selected voice. */
		try {

            $vendor_version = SpeakerHelper::get_vendor_version();
            switch ($vendor_version) {

                case 7:

                    /** Create a new TextToSpeechClient. */
                    $client = new TextToSpeechClient7();

                    /** Sets text to be synthesized. */
                    if ( $this->is_ssml_supported($lang_name) ) {
                        $input = (new SynthesisInput())
                            ->setSsml( $ssml );
                    } else {
                        $input = (new SynthesisInput())
                            ->setText( $ssml );
                    }

                    /** Build the voice request, select the language. */
                    $voice = ( new VoiceSelectionParams() )
                        ->setLanguageCode( $lang_code )
                        ->setName( $lang_name );

                    $response = $client->synthesizeSpeech(
                        $input,
                        $voice,
                        $this->get_audio_config( $options )
                    );

                    break;

                case 8:
                default:

                    /** Create a new TextToSpeechClient. */
                    $client = new TextToSpeechClient8();

                    /** Sets text to be synthesized. */
                    if ( $this->is_ssml_supported($lang_name) ) {
                        $input = (new SynthesisInput())
                            ->setSsml( $ssml );
                    } else {
                        $input = (new SynthesisInput())
                            ->setText( $ssml );
                    }

                    $voice = (new VoiceSelectionParams())
                        ->setLanguageCode($lang_code)
                        ->setName( $lang_name );

                    $request = (new SynthesizeSpeechRequest())
                        ->setInput($input)
                        ->setVoice($voice)
                        ->setAudioConfig($this->get_audio_config( $options ));

                    $response = $client->synthesizeSpeech($request);

                    break;

            }
			DashboardWidget::get_instance()->countChars( $ssml, $lang_name );

		} catch ( Exception $e) {

			if ( $e->getCode() === 3 ) {

				$googleMsg = $e->getMessage();
				$googleJson = json_decode( $googleMsg, true );

				echo __( 'Error message', 'speaker' ) . ': ' . $googleJson[ 'message' ] ?? __( 'Unknown error', 'speaker' );
				echo "\n";
				echo __( 'Error code', 'speaker' ) . ': (' . ( $googleJson[ 'code' ] ?? __( 'Unknown code', 'speaker' ) ) . ') ';
				echo $googleJson[ 'status' ] ?? __( 'Unknown status', 'speaker' );

			} else {

				echo __( 'Caught exception', 'speaker' ) . ': ',  $e->getMessage(), "\n";

			}

			wp_die();

		}

		/** The response's audioContent is binary. */
		$audioContent = $response->getAudioContent();

		/** Get path to upload folder. */
		$audio_file = AudioFile::path( $post_id, 0, 'tmp-' . uniqid() . '-' );
		file_put_contents( $audio_file, $audioContent );

		return $audio_file;

	}

    /**
     * Check if SSML is supported for the given language
     * @param $lang_name
     * @return bool
     */
    private function is_ssml_supported($lang_name)
    {
        $lang_name = strtolower($lang_name);
        return !( str_contains($lang_name, 'chirp') || str_contains($lang_name, 'journey'));
    }

    /**
     * Close unclosed HTML tags
     * @param $html
     * @param string[] $tags
     * @return string
     */
    private function fix_html_markup($html, array $tags = [] ): string{

        foreach ( $tags as $tag ) {

            // Count open <p> tags
            $open_tag = substr_count( $html, '<' . $tag );
            // Count close </p> tags
            $close_tag = substr_count( $html, '/'. $tag .'>' );

            // If the number of open tags is not equal to the number of close tags, then we add the missing tags
            if ( $open_tag > $close_tag ) {
                $html .= str_repeat( '</' . $tag . '>', $open_tag - $close_tag );
            } elseif ( $open_tag < $close_tag ) {
                $html = str_repeat( '<' . $tag . '>', $close_tag - $open_tag ) . $html;
            }

        }

        return $html;

    }

	/**
	 * Add punctuation pause.
	 *
	 * @param array $options
	 * @param string $html
	 *
	 * @return string
	 */
	private function punctuation_pause(array $options = array(), string $html = '' ): string {

		if ( $options[ 'punctuation_pause' ] === 'on' ) {

			// Pause after dot.
			if ( $options[ 'punctuation_pause_dot' ] > 0 ) {
				$html = preg_replace( '/\./', '.<break time="' . $options[ 'punctuation_pause_dot' ] . 'ms"/>', $html );
			}

			// Pause after comma.
			if ( $options[ 'punctuation_pause_comma' ] > 0 ) {
				$html = preg_replace( '/, /', ',<break time="' . $options[ 'punctuation_pause_comma' ] . 'ms"/>', $html );
			}

		}

		return $html;

	}

    /**
     * Add pauses related to HTML markup tags
     * @param array $options
     * @param string $html
     * @return string
     */
    private function markup_pause( array $options = array(), string $html = '' ): string {

        if ( $options[ 'punctuation_pause' ] === 'on' ) {

            // Pause after table row
            if ( $options[ 'markup_pause_tr' ] > 0 ) {
                $html = preg_replace( '/<\/tr>/', '</tr><break time="' . $options[ 'markup_pause_tr' ] . 'ms"/>', $html );
            }

            // Pause after table cell
            if ( $options[ 'markup_pause_td' ] > 0 ) {
                $html = preg_replace( '/<\/td>/', '</td><break time="' . $options[ 'markup_pause_td' ] . 'ms"/>', $html );
                $html = preg_replace( '/<\/th>/', '</th><break time="' . $options[ 'markup_pause_td' ] . 'ms"/>', $html );
            }

        }

        return $html;

    }

	/**
	 * Voice properties.
	 *
	 * @param $post_id
	 * @param string $html
	 * @param array $options
	 *
	 * @return array
	 */
	public function voice_properties( $post_id, string $html = '', $options = array() ): array {

		// Voice shortcode
		if ( strpos( $html, '<voice name=' ) !== false ) {

			return XMLHelper::get_instance()->get_lang_params_from_tag( $html );

		}

		// Post voice
		$post_voice = get_post_meta( $post_id, 'mdp_speaker_post_voice', true );
		if ( $post_voice && $post_voice !== '' ) {

			$lang_name = $post_voice;
			$lang_code = SpeakerHelper::lang_from_voice( $post_voice );

			return [ $lang_code, $lang_name ];

		}

		// Author voice
		$user_voice = get_user_meta( get_post( $post_id )->post_author, 'mdp_speaker_author_voice', true );

		if ( $user_voice && $user_voice !== '' ) {

			$lang_name = $user_voice;
			$lang_code = SpeakerHelper::lang_from_voice( $user_voice );

			return [ $lang_code, $lang_name ];

		}

		// Multilingual locale voice
		if ( $options[ 'multilang' ] === 'on' ) {

            $post_locale = SpeakerHelper::get_post_locale( $post_id );
            if (! $post_locale) {
                $post_locale = self::$html_lang_code ?? '';
            }

            /**
             * Represents a locale used for language and regional.
             *
             * @var string $locale The locale identifier in BCP 47 format.
             */
            $locale = apply_filters( 'speaker_multilang_locale', $post_locale, $post_id );

			return SpeakerHelper::multilang_properties( $options, $locale );

		}

		// Shortcodes voices or global voice
		return XMLHelper::get_instance()->get_lang_params_from_tag( $html );

	}

	/**
	 * Get audio config
	 * @param $options
	 * @return AudioConfig
	 */
	private function get_audio_config( $options ) {

		if ( $options[ 'advanced_voice' ] !== 'on' ) {

			$options = Settings::get_instance()->get_default_settings();

		}

		return ( new AudioConfig() )
			->setAudioEncoding( $this->get_audio_encoding( $options ) )
			->setEffectsProfileId( [ $options['audio-profile'] ] )
			->setSpeakingRate( $options['speaking-rate'] )
			->setPitch( $options['pitch'] )
			->setSampleRateHertz( intval( $options[ 'sample_rate' ] ) )
			->setVolumeGainDb( $options['volume'] );

	}

	/**
	 * Get audio encoding
	 * @param $options
	 * @return int
	 */
	private function get_audio_encoding( $options ) {

		switch ( $options[ 'audio_encoding' ] ) {

			// TODO: LINEAR16, ALAW and  OGG_OPUS required to merge the file headers

//            case 'LINEAR16':
//                $audio_encoding = 1;
//                break;
			case 'MP3':
				$audio_encoding = 2;
				break;
//            case 'OGG_OPUS':
//                $audio_encoding = 3;
//                break;
			case 'MULAW':
				$audio_encoding = 4;
				break;
//            case 'ALAW':
//                $audio_encoding = 5;
//                break;
			default:
				$audio_encoding = 2;

		}

		return $audio_encoding;

	}

	/**
	 * Add custom text before/after audio.
	 *
	 * @param array $parts - Content split to parts about 4000. Google have limits Total characters per request.
	 *
	 * @return array
	 */
	public function add_watermark( array $parts ): array {

		/** Before Audio. */
		if ( Settings::get_instance()->options['before_audio'] ) {
			array_unshift( $parts, do_shortcode( Settings::get_instance()->options['before_audio'] ) );
		}

		/** After Audio. */
		if ( Settings::get_instance()->options['after_audio'] ) {
			$parts[] = do_shortcode( Settings::get_instance()->options['after_audio'] );
		}

		return $parts;

	}

	/**
	 * Divide parts by voice. One part voiced by one voice.
	 *
	 * @param array $parts HTML parts to be voiced.
	 *
	 * @return array
	 */
	public function voice_divider( array $parts ): array {

		/** Array with parts split by voice. */
		$result = [];
		foreach ( $parts as $part ) {

			/** Mark location of the cut. */
			$part = str_replace( ["<voice", "</voice>"], ["{|mdp|}<voice", "</voice>{|mdp|}"], $part );

			/** Cut by marks. */
			$arr = explode( "{|mdp|}", $part );

			/** Clean the array. */
			$arr = array_filter( $arr );

			/** Combine results. */
			$result = array_merge( $result, $arr );

		}

		/** Fix broken html of each part. */
		foreach ( $result as &$el ) {
			$el = XMLHelper::get_instance()->repair_html( $el );
		}
		unset( $el );

		/** Remove empty elements. */
		return array_filter( $result );

	}

	/**
	 * Return array content of each ST element.
	 *
	 * @param $post_id - ID of the Post/Page content from which we will parse.
	 * @param $stid - ID of Speech Template.
	 * @param int $page_index - Index of the page.
	 *
	 * @return array|mixed|object
	 * @throws DOMException
	 * @since 3.0.0
	 * @access public
	 **/
	private function parse_st_content( $post_id, $stid, int $page_index = 0 ) {

		$options = Settings::get_instance()->options;

		/** Get Speech Template data. */
		$st = SpeakerCaster::get_instance()->get_st( $stid );

		/** On error. */
		if ( ! $st ) { return false; }

		/** Get ST elements. */
		$elements = $st['elements'];

		/** On error. */
		if ( ! is_array( $elements ) ) { return false; }

		/** Use internal libxml errors -- turn on in production, off for debugging. */
		libxml_use_internal_errors( true );

		/** Create a new DomDocument object. */
		$dom = new DomDocument;

		/** Get Current post Content. */
		$post_content = Parser::parse_post_content( $post_id, $page_index );

		/** Load the HTML. */
		$dom->loadHTML( $post_content );

		$parts = [];

		/** Collect content foreach element. */
		foreach ( $elements as $element ) {

			/** Parse content for DOM Elements in ST. */
			if ( 'element' === $element['type'] ) {

				/** Create a new XPath object. */
				$xpath = new DomXPath( $dom );

				/** Query all elements with xPath */
				$nodes = $xpath->evaluate( $element['xpath'] );

				/** Skip element, if it's not found. */
				if ( ! $nodes->length ) { continue; }

				/** Get element content. */
				$content = XMLHelper::get_instance()->get_inner_html( $nodes->item( 0 ) );

				$content = Parser::regex_content_replace( $content );

				$content = Parser::clean_content( $content );

				/** Apply SSML tags to content. */
				$content = $this->apply_ssml_settings( $element, $content );

				if ( strlen( $content ) > 4500 ) {

					/**
					 * Split to parts about 4500. Google have limits Total characters per request.
					 * See: https://cloud.google.com/text-to-speech/quotas
					 **/
					/** @noinspection SlowArrayOperationsInLoopInspection */
					$parts = array_merge( $parts, Parser::great_divider( $post_id, $content, (int)$options[ 'part_length' ] ) );

				} else {

					/** Add first DomNode inner HTML. */
					$parts[] = $content;

				}

			} elseif ( 'text' === $element['type'] ) {

				/** Get custom content. */
				$content = $element['content'];

				$content = Parser::regex_content_replace( $content );

				$content = Parser::clean_content( $content );

				/** Apply SSML tags to content. */
				$content = $this->apply_ssml_settings( $element, $content );

				/** Add custom content. */
				$parts[] = $content;

				if ( strlen( $content ) > 4500 ) {

					/**
					 * Split to parts about 4500. Google have limits Total characters per request.
					 * See: https://cloud.google.com/text-to-speech/quotas
					 **/
					/** @noinspection SlowArrayOperationsInLoopInspection */
					$parts = array_merge( $parts, Parser::great_divider( $post_id, $content, (int)$options[ 'part_length' ] ) );

				}

			} elseif ( 'pause' === $element['type'] ) {

				/** Add pause element. */
				$parts[] = "<break time=\"{$element['time']}ms\" strength=\"{$element['strength']}\" />";

			}

		}

		return $parts;

	}

	/**
	 * Apply SSML tags to content.
	 *
	 * @param array $element
	 * @param string $content
	 *
	 * @since 3.0.0
	 * @access public
	 *
	 * @return array|mixed|object
	 **/
	private function apply_ssml_settings( $element, $content ) {

		/** Add 'Say As' if needed. */
		if ( ! in_array( $element['sayAs'], ['none', 'undefined'] ) ) {
			$content = "<say-as interpret-as=\"{$element['sayAs']}\">{$content}</say-as>";
		}

		/** Add 'Emphasis' if needed. */
		if ( ! in_array( $element['emphasis'], ['none', 'undefined'] ) ) {
			$content = "<emphasis level=\"{$element['emphasis']}\">{$content}</emphasis>";
		}

		/** If voice is different from default, change voice. */
		if (
			! in_array($element['voice'], ['none', 'undefined']) &&
			$element['voice'] !== Settings::get_instance()->options['language']
		) {

			$content = "<voice name=\"{$element['voice']}\">{$content}</voice>";

		}

		return $content;

	}

	/**
	 * Speaker use custom page template to parse content without garbage.
	 *
	 * @param string $template - The path of the template to include.
	 *
	 * @since 3.0.0
	 * @access public
	 *
	 * @return string
	 *
	 * @noinspection PhpUnused
	 **/
	public static function speaker_page_template( $template ) {

		/** Change template for correct parsing content. */
		if ( isset( $_GET['speaker-template'] ) && 'speaker' === $_GET['speaker-template'] ) {

			/** Disable admin bar. */
			show_admin_bar( false );

			$template = Plugin::get_path() . 'src/Merkulove/Speaker/speaker-template.php';

		}

		return $template;

	}

	/**
	 * Prepare parts for generate audio for post based on Speech Template.
	 *
	 * @param int $post_id
	 * @param string $stid
	 * @param int $page_index
	 *
	 * @return array|false
	 * @throws DOMException
	 */
	public function template_based_generation( int $post_id, string $stid, int $page_index = 0 ) {

		/** Get content of each element. */
		$parts = $this->parse_st_content( $post_id, $stid, $page_index );

		/** On error. */
		if ( empty( $parts ) ) { return false; }

		return $parts;

	}

	/**
	 * Main SpeechGeneration Instance.
	 *
	 * Insures that only one instance of SpeechGeneration exists in memory at any one time.
	 *
	 * @static
	 * @return SpeechGeneration
	 * @since 3.0.0
	 **/
	public static function get_instance() {

		/** @noinspection SelfClassReferencingInspection */
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof SpeechGeneration ) ) {

			/** @noinspection SelfClassReferencingInspection */
			self::$instance = new SpeechGeneration;

		}

		return self::$instance;

	}

} // End Class SpeechGeneration.
