<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Alexander Khmelnitskiy (info@alexander.khmelnitskiy.ua), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use DOMDocument;
use DOMException;
use DOMXPath;
use Merkulove\Speaker\Unity\Plugin;
use Merkulove\Speaker\Unity\Settings;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

final class SpeechTemplates {

	/**
	 * The one true MetaBox.
	 *
	 * @var MetaBox
	 **/
	private static $instance;

	private function __construct() {

		/** Add Speaker bulk action for each selected post type. */
		$cpt_posts = Settings::get_instance()->options['cpt_support'];

		foreach ( $cpt_posts as $post_type ) {

			/** Save Custom Post Template on post save. */
			add_action( "save_post_" . $post_type, [ $this, 'save_post' ] );

		}

		/** Automatic synthesis. */
		add_action( 'post_updated', [ $this, 'auto_generation_on_update' ], 10, 3 );

	}


	/**
	 * Render speech template controls.
	 *
	 * @param $default
	 *
	 * @return void
	 **/
	public function render_speech_template_controls( $default ) {
		?>
		<button id="mdp-speaker-add-speech-template-btn"
		        data-post-url="<?php echo esc_url( $this->get_post_url() ); ?>"
		        class="mdp-speaker-add mdc-icon-button material-icons mdc-ripple-upgraded--unbounded mdc-ripple-upgraded"
		        title="<?php esc_attr_e( 'Add Speech Template', 'speaker' ); ?>"
		        aria-label="<?php esc_attr_e( 'Add Speech Template', 'speaker' ); ?>" >add</button>

		<button class="mdp-speaker-edit mdc-icon-button material-icons mdc-ripple-upgraded--unbounded mdc-ripple-upgraded"
		        title="<?php esc_attr_e( 'Edit Speech Template', 'speaker' ); ?>"
		        aria-label="<?php esc_attr_e( 'Edit Speech Template', 'speaker' ); ?>" >create</button>

		<button class="mdp-speaker-make-default mdc-icon-button material-icons mdc-ripple-upgraded--unbounded mdc-ripple-upgraded"
		        data-post-type="<?php esc_attr_e( get_post_type() ); ?>"
		        data-default-for-post-type="<?php esc_attr_e( $default ); ?>"
		        title="<?php esc_attr_e( 'Save this Speech Template as Default for this Post Type', 'speaker' ); ?>"
		        aria-label="<?php esc_attr_e( 'Save this Speech Template as Default for this Post Type', 'speaker' ); ?>" >
			<?php if ( $default ) : ?>
				flag
			<?php else: ?>
				outlined_flag
			<?php endif; ?>
		</button>

		<button class="mdp-speaker-delete mdc-icon-button material-icons mdc-ripple-upgraded--unbounded mdc-ripple-upgraded"
		        title="<?php esc_attr_e( 'Delete Speech Template', 'speaker' ); ?>"
		        aria-label="<?php esc_attr_e( 'Delete Speech Template', 'speaker' ); ?>" >delete</button>

		<?php

	}

	/**
	 * Return frontend post url for Speech Template Editor.
	 *
	 * @return string
	 **/
	public function get_post_url(): string {

		/** Current post id. */
		$post_id = get_the_ID();

		/** Get full permalink for the current post. */
		$url = get_permalink( $post_id );

		/** Returns a string if the URL has parameters or NULL if not. */
		$query = parse_url( $url, PHP_URL_QUERY );

		/** Add speaker_speech_template param to URL. */
		if ( $query ) {
			$url .= '&speaker_speech_template=1';
		} else {
			$url .= '?speaker_speech_template=1';
		}

		return $url;

	}


	/**
	 * Return Speech Template Name by ID.
	 *
	 * @param $template_id
	 *
	 * @since 3.0.0
	 * @access public
	 *
	 * @return string
	 **/
	public function get_speech_template_name( $template_id ): string {

		/** Read all ST from settings. */
		/** In this option we store all Speech Templates. */
		$st_opt_name = 'mdp_speaker_speech_templates';

		/** Get all Speech Templates. */
		$st = get_option( $st_opt_name, false );

		/** Return if no ST. */
		if ( ! $st ) { return ''; }

		/** If We have any ST. */
		if ( count( $st )  ) {

			/** Add ST to list. */
			foreach ( $st as $template ) {

				if ( $template['id'] === $template_id ) {
					return $template['name'];
				}

			}

		}

		return '';

	}

	/**
	 * Return Speech Template by post id.
	 * @param $post_id
	 * @return string
	 **/
	public function get_post_template( $post_id ): string {

		/** If current post have custom Speech Template, show it. */
		$template_id = get_post_meta( $post_id, 'mdp_speaker_custom_speech_template', true );
		if ( $template_id ) { return $template_id; }

		/** Return default ST for current post type. */
		$default = MetaBox::get_instance()->get_default_st( get_post_type( $post_id ) );

		if ( $default ) { return $default; }

		/** Default Speech Template. */
		return 'content';

	}

	/**
	 * Save Custom Post Template on post save.
	 *
	 * @param $post_id
	 *
	 * @since  3.0.0
	 * @access public
	 *
	 * @return void
	 **/
	public function save_post( $post_id ) {

		/** Get new custom template. */
		$new_custom_st = filter_input( INPUT_POST, 'mdp_speaker_speech_templates_template' );

		$meta_key = 'mdp_speaker_custom_speech_template';

		/** Clear post custom template for 'content' */
		if ( 'content' === $new_custom_st ) {

			update_post_meta( $post_id, $meta_key, '' );

			return;

		}

		/** If template equal default template for current post type -> Clear */
		if ( MetaBox::get_instance()->get_default_st( get_post_type( $post_id ) ) === $new_custom_st ) {

			update_post_meta( $post_id, $meta_key, '' );

			return;

		}

		/** Remember custom template for current post type. */
		update_post_meta( $post_id, $meta_key, $new_custom_st );

	}

	/**
	 * Generate audio version on every post update.
	 *
	 * @param $post_ID
	 * @param $post_after
	 * @param $post_before
	 *
	 * @since  3.0.0
	 * @access public
	 *
	 * @return void
	 **/
	public function auto_generation_on_update( $post_ID, $post_after, $post_before ) {

		$options = Settings::get_instance()->options;

		/** Work only if Automatic synthesis is enabled. */
		if ( 'on' !== $options['auto_generation'] ) { return; }

		/** Get supported post types from plugin settings. */
		$cpt_support = $options['cpt_support'];

		/** Work only with supported post types. */
		if ( ! in_array( $post_before->post_type, $cpt_support ) ) { return; }

		/** Work only if content was changed. */
//		if ( $post_before->post_content === $post_after->post_content ) { return; }

		/** Generate Audi version for current post. */
		SpeechGeneration::get_instance()->voice_acting( $post_ID );

	}

	/**
	 * @throws DOMException
	 */
	public function apply_ssml_attributes($post_content ): string {

		/** Hide DOM parsing errors. */
		libxml_use_internal_errors( true );
		libxml_clear_errors();

		/** Load the possibly malformed HTML into a DOMDocument. */
		$dom          = new DOMDocument();
		$dom->recover = true;
		$dom->loadHTML( '<?xml encoding="UTF-8"><body id="repair">' . $post_content . '</body>' ); // input UTF-8.

		$selector = new DOMXPath( $dom );

		/** Say as */
		foreach( $selector->query( '//*[@speaker-say-as]') as $e ) {

			// Create SSML node
			$ssmlNode = $dom->createElement('say-as', $e->nodeValue );
			$ssmlNode->setAttribute('interpret-as', $e->getAttribute( 'speaker-say-as' ) );
			$ssmlNode->setAttribute('format', $e->getAttribute( 'format' ) );
			$ssmlNode->setAttribute('detail', $e->getAttribute( 'detail' ) );

			// Replace HTML node by SSML node
			$e->parentNode->replaceChild( $ssmlNode, $e );

		}

		/** Prosody */
		foreach( $selector->query( '//*[@speaker-prosody]') as $e ) {

			// Create SSML node
			$ssmlNode = $dom->createElement('prosody', $e->nodeValue );
			$ssmlNode->setAttribute('rate', $e->getAttribute( 'rate' ) );
			$ssmlNode->setAttribute('pitch', $e->getAttribute( 'pitch' ) );
			$ssmlNode->setAttribute('volume', $e->getAttribute( 'volume' ) );

			// Replace HTML node by SSML node
			$e->parentNode->replaceChild( $ssmlNode, $e );

		}

		/** Substitution or alias */
		$dom = $this->inject_ssml_markup( $dom, $selector, 'speaker-sub', 'sub', 'alias' );

		/** Emphasis */
		$dom = $this->inject_ssml_markup( $dom, $selector, 'speaker-emphasis', 'emphasis', 'level' );

		/** Voice name */
		$dom = $this->inject_ssml_markup( $dom, $selector, 'speaker-voice', 'voice', 'name' );

		/** HTML without muted tags. */
		$body = $dom->documentElement->lastChild;

		return trim( XMLHelper::get_instance()->get_inner_html( $body ) );

	}

	/**
	 * Inject SSML tag instead HTML with data attributes
	 * @param $dom
	 * @param $selector
	 * @param $speaker_attribute
	 * @param $ssml_tag
	 * @param $ssml_attribute
	 * @return mixed
	 */
	private function inject_ssml_markup( $dom, $selector, $speaker_attribute, $ssml_tag, $ssml_attribute ) {

		foreach( $selector->query( '//*[@' . $speaker_attribute . ']') as $e ) {

			// Create SSML node
			$ssmlNode = $dom->createElement( $ssml_tag, $e->nodeValue );
			$ssmlNode->setAttribute( $ssml_attribute, $e->getAttribute( $speaker_attribute ) );

			// Replace HTML node by SSML node
			$e->parentNode->replaceChild( $ssmlNode, $e );

		}

		return $dom;

	}

	/**
	 * Main SpeechTemplates Instance.
	 *
	 * @return SpeechTemplates
	 **/
	public static function get_instance() {

		/** @noinspection SelfClassReferencingInspection */
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof SpeechTemplates ) ) {

			self::$instance = new SpeechTemplates;

		}

		return self::$instance;

	}

}
