<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.1.10
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Alexander Khmelnitskiy (info@alexander.khmelnitskiy.ua), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

use Google\Exception;
use Google\Service\Drive;
use Google_Client;
use Merkulove\Speaker;
use Merkulove\Speaker\Unity\Settings;

final class StorageGoogle {

	/**
	 * The one true StorageGoogle.
	 *
	 * @var StorageGoogle
	 * @since 3.0.0
	 **/
	private static $instance;

	/**
	 * Error icon
	 * @var string
	 */
	private static $icon_error = '<span class="material-icons material-icons-outlined material-icons-warn">error</span>';

	/**
	 * OK icon
	 * @var string
	 */
	private static $icon_ok = '<span class="material-icons material-icons-outlined material-icons-ok">check_circle</span>';

	/**
	 * Sets up a new StorageGoogle instance.
	 *
	 * @since 3.0.0
	 * @access public
	 **/
	private function __construct()
    {

	    /** Reset API Key on fatal error. */
	    if ( isset( $_GET['reset-drive-key'] ) && '1' === $_GET['reset-drive-key'] ) {
		    $this->reset_drive_key();
	    }

        /** Catch Google Drive authorization code */
        if ( isset( $_GET[ 'code' ] ) && isset( $_GET[ 'scope' ] ) ) {
            add_action( 'wp_loaded', [ $this, 'gd_catcher' ] );
        }

	}

	/**
	 * Reset Drive API Key on fatal error.
	 *
	 * @since  3.0.0
	 * @access public
	 * @return void
	 **/
	private function reset_drive_key() {

		$options = get_option( 'mdp_speaker_storage_settings' );
		$options[ 'storage-api-key' ] = '';

		/** Save new value. */
		update_option( 'mdp_speaker_storage_settings', $options );

		/** Go to first tab. */
		wp_redirect( admin_url( '/admin.php?page=mdp_speaker_settings&tab=storage' ) );
		exit;

	}

    /**
     * Generate authorization link for Google Drive
     * @return string
     * @throws Exception
     */
	public function gd_authorization_url()
    {

//        $options = Settings::get_instance()->options;
        $authUrl = '';

        $client = $this->get_Google_Client();

        // If there is no previous token, or it's expired.
        if ( $client->isAccessTokenExpired() ) {

            // Refresh the token if possible, else fetch a new one.
            if ($client->getRefreshToken()){
                $client->fetchAccessTokenWithRefreshToken($client->getRefreshToken());
            } else {
                // Request authorization from the user.
                $authUrl = $client->createAuthUrl();
            }

        }

        return $authUrl;

    }

    /**
     * Return token link markup
     */
    public function gd_token_link(): string {

		// Remove token if key is not exist
	    $isKeyExists = StorageGoogle::is_key_exists();
        if ( ! $isKeyExists[ 'status' ] ) {

			// Remove token
	        delete_option( 'mdp_speaker_gd_token' );
			return $isKeyExists[ 'msg' ];

		}

		// Get token
        $token = get_option( 'mdp_speaker_gd_token' );

        // Icon
        if ( $token ) {
			$icon = $this->get_drive_client()->isAccessTokenExpired() ? self::$icon_error : self::$icon_ok;
        } else {
            $icon = self::$icon_error;
        }

        /** @noinspection HtmlUnknownTarget */
        return wp_sprintf(
            '<p class="mdp-speaker-gb-warn">
                %s
                <a href="%s" target="_blank">%s %s</a>
            </p>',
            $icon,
            esc_url( $this->gd_authorization_url() ),
            $token ? esc_html__( 'Re-validate token', 'speaker' ) : esc_html__( 'Get a token', 'speaker'),
            esc_html__( 'to link this site to the Google Drive', 'speaker' )
        );

    }

    /**
     * Returns non-authorized API client
     * @return Google_Client/bool
     * @throws Exception
     */
    private function get_Google_Client()
    {

        if ( ! SpeakerHelper::is_key_exists() ) { return false; }

		// Un-code API key from base64 and decode json
        $options = Settings::get_instance()->options;
	    $json = json_decode( base64_decode( $options[ 'storage-api-key' ] ), true );

		try {

	        $client = new Google_Client();
		    $client->setApplicationName( Speaker\Unity\Plugin::get_name() );
		    $client->setScopes( [ Drive::DRIVE_METADATA_READONLY, Drive::DRIVE ] );
		    $client->setAuthConfig( $json );
		    $client->setAccessType('offline');
		    $client->setPrompt('select_account consent');

			return $client;

	    } catch (Exception $e) {

			// Remove token
		    delete_option( 'mdp_speaker_gd_token' );

			// Show error
		    echo esc_html__( 'Cannot access Google Drive', 'speaker' );

	    }

    }

    /**
     * Returns an authorized API client.
     * @return Google_Client the authorized client object
     * @throws Exception
     */
    public function get_drive_client()
    {

        $access_token = get_option( 'mdp_speaker_gd_token' );

        $client = $this->get_Google_Client();
        $client->setAccessToken( $access_token );

        if ( $client->isAccessTokenExpired() ) {

            if ( $client->getRefreshToken() ) {
                $client->fetchAccessTokenWithRefreshToken( $client->getRefreshToken() );
            }

        }

        return $client;

    }

    /**
     * Store access token
     * @throws Exception
     */
	public function gd_catcher(){

        $auth_code = $_GET[ 'code' ];
        if ( ! is_admin() && empty( $auth_code ) ) { return; }

        $client = $this->get_Google_Client();
        $token = $client->fetchAccessTokenWithAuthCode( $auth_code );

        /** Store token to the DB */
        if ( is_array( $token ) && isset( $token[ 'access_token' ] ) ) {

            update_option( 'mdp_speaker_gd_token', $token ); // TODO: extend token lifetime

        }

    }

	/**
	 * Upload to Google Drive
	 *
	 * @param $file_path
	 * @param $post_id
	 *
	 * @throws Exception
	 */
    public function gd_upload_file( $file_path, $post_id ) {

        $options = Settings::get_instance()->options;

        /** Remove token and exit if api key is not exist */
        if ( strlen( $options[ 'storage-api-key' ] ) < 400 ) {

            delete_option( 'mdp_speaker_gd_token' );
            return;

        }

        /** Exit if token is not exist */
        $access_token = get_option( 'mdp_speaker_gd_token' );
        if ( ! $access_token ) { return; }

        /** Get the API client and construct the service object */
        $client = StorageGoogle::get_instance()->get_drive_client();
        $service = new Drive( $client );
        $file = new Drive\DriveFile();

        /** Get folder ID */
        $folder_id = $this->gd_folder_id( $service, $file );

        /** Find folder on the Google Drive */
        $has_folder = $this->gb_folder_exist( $service, $folder_id );

        if ( $has_folder ) {

            $this->gd_create_file( $service, $file, $folder_id, $file_path, $post_id );

        } else {

            echo esc_html__( 'Cannot access Google Drive', 'speaker' );

        }

    }

    /**
     * Create file on the Google Drive
     *
     * @param $service
     * @param $file
     * @param $folder_id
     * @param $file_path
     * @param $post_id
     */
    private function gd_create_file( $service, $file, $folder_id, $file_path, $post_id )
    {

        $options = Settings::get_instance()->options;

        $file_content = file_get_contents( $file_path );
        $file_mime = mime_content_type( $file_path );
        $post_name = get_the_title( $post_id );
        $time = current_time( 'Y-m-d H:i:s' );
        $site = get_bloginfo( 'name' );

        $file->setName( $post_name . ' - ' . $time . '.' . $options[ 'audio-format' ] );
        $file->setDescription( $post_name . ' ' . esc_html__( 'from' ) . ' ' . $site );
        $file->setMimeType( $file_mime );
        $file->setParents( [ $folder_id ] );

        try {

            $results = $service->files->create( $file, array(
                'data' => $file_content,
                'mimeType' => $file_mime,
                'uploadType' => 'multipart'
            ) );

            /** Store audio ID to the post meta */
            if ( isset( $results->id ) ) {

                update_post_meta( $post_id, 'mdp_speaker_gd', $results->id );

            }

        } catch( Exception $e ){

            echo 'An error occurred : ' . $e->getMessage();

        }

    }

    /**
     * Create folder on the Google Drive
     *
     * @param $service
     * @param $file
     * @return string
     */
    private function gd_create_folder( $service, $file )
    {

        $file->setName( 'Speaker - ' . get_bloginfo( 'name' ) );
        $file->setDescription( esc_html__( 'Records created by Speaker WordPress Plugin', 'speaker' ) );
        $file->setMimeType( 'application/vnd.google-apps.folder' );

        $gd_response = $service->files->create( $file );
        $folder_id = $gd_response->id ?? '';

        if ( isset( $gd_response->id ) ) {
            update_option( 'mdp_speaker_gd_folder', $folder_id );
        }

        return $folder_id;

    }

    /**
     * Check if Drive has folder by ID
     *
     * @param $service
     * @param $folder_id
     * @return bool
     */
    private function gb_folder_exist( $service, $folder_id )
    {

        $gd_response = $service->files->get( $folder_id );
        return isset( $gd_response->id );

    }

    /**
     * Get Google Drive folder id for the current site
     *
     * @param $service
     * @param $file
     * @return mixed|string|void
     */
    private function gd_folder_id( $service, $file )
    {

        $folder_id = get_option( 'mdp_speaker_gd_folder' );
        return ! $folder_id ? $this->gd_create_folder( $service, $file ) : $folder_id;

    }

	/**
	 * Is key exists
	 */
	public static function is_key_exists(): array {

		$options = Settings::get_instance()->options;

        // Check Google Text-to-Speech API key
        if ( ! SpeakerHelper::is_key_exists() ) {

            return [
                'status'    => false,
                'msg'       => wp_sprintf( '<br><b style="color: orangered">%s <a style="color: orangered" href="%s">%s<a/>. </b>',
                    esc_html__( 'First, you need to add the Google API key file to', 'speaker' ),
                    admin_url( '/admin.php#mdp-dnd-api-key-drop-zone?page=mdp_speaker_settings&tab=general' ),
                    esc_html__( 'Speech Synthesis', 'speaker' )
                )
            ];

        }

        // Check Google Drive API key
		if ( isset( $options[ 'storage-api-key' ] ) && strlen( $options[ 'storage-api-key' ] ) > 400 ) {

			// Show message if key is not valid
			$json = json_decode( base64_decode( $options[ 'storage-api-key' ] ), true );

			if ( ! isset( $json[ 'web' ] ) ) {
				return [
					'status'    => false,
					'msg'       => '<br><b style="color: orangered">' . esc_html__( 'Wrong json API key format', 'speaker' ) . '</b>'
				];
			}

			if ( ! isset( $json[ 'web' ][ 'redirect_uris' ] ) ) {
				return [
					'status'    => false,
					'msg'       => '<br><b style="color: orangered">' . esc_html__( 'Wrong json API key format. Missing redirect URL info', 'speaker' ) . '</b>'
				];
			}

			return [
				'status'    => true,
				'msg'       => ''
			];

		} else {

			return [
                'status'    => false,
				'msg'       => ''
			];

		}

	}

	/**
	 * Main StorageGoogle Instance.
	 *
	 * Insures that only one instance of StorageGoogle exists in memory at any one time.
	 *
	 * @static
	 * @return StorageGoogle
	 * @since 3.0.0
	 **/
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof StorageGoogle ) ) {

			self::$instance = new StorageGoogle;

		}

		return self::$instance;

	}

} // End Class StorageGoogle.
