<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.0.0
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Plugin;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove\Speaker
 **/
final class TabDesign {

	/**
	 * Controls for Design tab.
	 * @return void
	 */
	public static function controls( $options ) {

		$tabs = Plugin::get_tabs();
		$fields = array();

		$fields[ 'before_player_switcher' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Content before the player', 'speaker' ),
			'description'       => esc_html__( 'Add a text or HTML markup before the player', 'speaker' ),
			'default'           => 'off',
		];

		$fields[ 'before_player' ] = [
			'type'              => 'editor',
			'description'       => esc_html__( 'Add a text or HTML markup before the player', 'speaker' ),
			'default'           => '',
			'attr'              => [
				'media_buttons' => false,
				'textarea_rows' => 4
			]
		];

		$fields[ 'after_player_switcher' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Content after the player', 'speaker' ),
			'description'       => esc_html__( 'Add a text or HTML markup after the player', 'speaker' ),
			'default'           => 'off',
		];

		$fields[ 'after_player' ] = [
			'type'              => 'editor',
			'description'       => esc_html__( 'Add a text or HTML markup after the player', 'speaker' ),
			'default'           => '',
			'attr'              => [
				'media_buttons' => false,
				'textarea_rows' => 4
			]
		];

		// TODO: SpeakerUtilities
		$fields[ 'in_loop' ] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Display in loop', 'speaker' ),
			'description'       => esc_html__( 'Display player only for loop queries', 'speaker' ),
			'default'           => 'on',
		];

		$default_part_length = 4500;
		$current_part_length = $options[ 'part_length' ] ?? $default_part_length;
		$fields[ 'part_length' ] = [
			'type'              => 'slider',
			'label'             => esc_html__( 'Part length', 'speaker' ),
			'description'       => wp_sprintf(
				/* translators: %s: part length */
				esc_html__( 'The length of the fragment into which long texts are divided. Current fragment length %s of 5000 bytes.', 'speaker' ),
				'<strong>' . esc_attr( $current_part_length ) . '</strong>'
			),
			'default' => $default_part_length,
			'min' => 100,
			'max' => 5000,
			'step' => 100,
		];

		$fields[ 'capability_divider' ] = [
			'type'              => 'divider',
		];

		$fields[ 'capability_audio' ] = [
			'type'              => 'select',
			'label'             => esc_html__( 'Can manage audio files', 'speaker' ),
			'placeholder'       => esc_html__( 'Select user role', 'speaker' ),
			'description'       => wp_sprintf(
				/* translators: %1$s: bold text, %2$s: link */
				esc_html__( 'Select the minimal user roles that will have access to %1$s. Find out more about %2$s in WordPress', 'speaker' ),
				'<b style="color: blueviolet">' . __( 'create, re-create, remove audio and speech templates', 'speaker' ) . '</b>',
				'<a href="https://wordpress.org/documentation/article/roles-and-capabilities/" target="_blank">' . esc_html__( 'user roles and capabilities', 'speaker' ) . '</a>'
			),
			'default'           => 'manage_options',
			'options'           => [
				'manage_options'            => esc_html__( 'Administrator', 'speaker' ),
				'edit_pages'                => esc_html__( 'Editor', 'speaker' ),
				'publish_posts'             => esc_html__( 'Author', 'speaker' ),
				'edit_posts'                => esc_html__( 'Contributor', 'speaker' ),
				'read'                      => esc_html__( 'Subscriber', 'speaker' ),
			]
		];

        // Speaker PHP 7 conditional fields
        if ( is_plugin_active( 'speaker-php7/speaker-php7.php' ) ) {

            $fields[ 'php_divider' ] = [
                'type'              => 'divider',
            ];

            $php_version = PHP_VERSION;
            $fields[ 'vendor_version' ] = [
                'type'              => 'select',
                'label'             => esc_html__( 'Compatibility', 'speaker' ),
                'placeholder'       => esc_html__( 'Compatibility', 'speaker' ),
                'description'       => wp_sprintf(
                /* translators: %1$s: bold text, %2$s: br, %3$s: b */
                    esc_html__( 'Select a compatible Google Library version. Current server PHP version: %1$s. %2$s Attention: Incorrect selection of the version libraries of the version will throw a %3$s on the entire site.', 'speaker' ),
                    '<strong>' . esc_attr( $php_version ) . '</strong>',
                    '<br>',
                    '<b style="color: red">' . esc_html__( 'Fatal Error', 'speaker' ) . '</b>'
                ),
                'default'           => '8.1',
                'options'           => [
                    '7.4'           => 'For PHP 7.4+',
                    '8.1'           => 'For PHP 8.1+',
                ]
            ];
        }

		$tabs[ 'design' ][ 'fields' ] = $fields;

		Plugin::set_tabs( $tabs );

	}

}
