<?php
/**
 * Speaker
 * Create an audio version of your posts, with a selection of more than 340 voices across more than 52 languages and variants.
 * Exclusively on https://1.envato.market/speaker
 *
 * @encoding        UTF-8
 * @version         4.0.0
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Speaker;

use Merkulove\Speaker\Unity\Plugin;
use Merkulove\Speaker\Unity\Settings;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove\Speaker
 **/
final class TabMultilingual {

	/**
	 * Controls for Schema tab.
	 * @return void
	 */
	public static function controls() {

		$tabs = Plugin::get_tabs();
		$fields = array();

		$options = Settings::get_instance()->options;

		$fields[ 'multilang' ] = [
			'type'          => 'switcher',
			'label'         => esc_html__( 'Multilingual', 'speaker' ),
			'description'   => esc_html__( 'Enable multilingual audion generation', 'speaker' ),
			'default'       => 'off',
		];

		$locales_count = self::locales_count( $options ) ?: 1; // Always render at least one locale
		for ( $i = 1; $i <= $locales_count; $i++ ) {

			$land_code = $options[ 'multilang_locale_' . $i ] ?? '';
			$fields[ 'multilang_locale_' . $i ] = [
				'type'              => 'custom_type',
				'render'            => [ TabMultilingual::class, 'voices_choices' ],
				'label'             => '<b>' . $land_code . '</b> ' . esc_html__( 'Locale settings', 'speaker' ),
				'default'           => '',
			];

		}

		$fields[ 'multilang_buttons' ] = [
			'type'              => 'custom_type',
			'render'            => [ TabMultilingual::class, 'locale_buttons' ],
			'label'             => '',
		];

		$tabs[ 'multilingual' ][ 'fields' ] = $fields;
		Plugin::set_tabs( $tabs );

	}

	/**
	 * Voices choices
	 */
	public static function voices_choices( $field ) {

		$options = Settings::get_instance()->options;
		$i = str_replace( 'multilang_locale_', '', $field );

		echo wp_sprintf(
			'<div class="speaker-locale-container choices-wrapper" data-lang="%s" data-voice="%s"></div>',
			esc_attr( $options[ 'multilang_locale_' . $i ] ?? '' ),
			esc_attr( $options[ 'multilang_voice_' . $i ] ?? '' )
		);

	}

	/**
	 * Add/remove locale buttons
	 * @return void
	 */
	public static function locale_buttons() {

		echo wp_sprintf(
			'<div class="speaker-locale-buttons">
				<button
					id="mdp_speaker_multilingual_settings_multilang_add_locale"
					name="mdp_speaker_multilingual_settings[multilang_add_locale]"
					class="mdc-button mdc-button--outlined mdc-ripple-upgraded"
					value="%1$s">
					<span class="mdc-button__ripple"></span>
					<i class="material-icons mdc-button__icon" aria-hidden="true">add</i>%1$s
				</button>
				<button
					id="mdp_speaker_multilingual_settings_multilang_remove_locale"
					name="mdp_speaker_multilingual_settings[multilang_remove_locale]"
					class="mdc-button mdc-button--outlined mdc-ripple-upgraded"
					value="%2$s">
					<span class="mdc-button__ripple"></span>
					<i class="material-icons mdc-button__icon" aria-hidden="true">remove</i>%2$s
				</button>
			</div>',
			esc_html__( 'Add locale', 'speaker' ),
			esc_html__( 'Remove locale', 'speaker' )
		);

	}

	/**
	 * Calculate saved locales in settings
	 *
	 * @param array $options
	 *
	 * @return int
	 */
	private static function locales_count( array $options = array() ): int {

		$locales_count = 0;
		foreach ( $options as $key => $value ) {

			// If locale exists and not empty
			if ( strpos( $key, 'multilang_locale_' ) !== false && $value !== ''  ) {

				$locales_count++;

			}

		}

		return $locales_count;

	}

}
